/* SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Cfg - A library for Virgo process implementation
 *
 * Copyright © 2001-2021 Laboratoire de physique des particules d'Annecy - CNRS
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Fatih Bellachia <fatih.bellachia@lapp.in2p3.fr>
 *   Laurent Fournier <laurent.fournier@lapp.in2p3.fr>
 *   Alain Masserot <alain.masserot@lapp.in2p3.fr>
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <CbfHash.h>
#include <CbfClean.h>

#ifdef USE_MW
#include <memwatch.h>
#endif /* USE_MW */
#ifdef USE_MPATROL
#include <mpatrol.h>
#endif /* USE_MPATROL */

extern void psignal(int, const char *);

struct sigdesc {
  char *name;
  int  number;
};

struct sigdesc CbfSigDesc[] = {
  {"SIGHUP",	SIGHUP},
  {"SIGINT",	SIGINT},
  {"SIGQUIT",	SIGQUIT},
  {"SIGILL",	SIGILL},
  {"SIGTRAP",	SIGTRAP},
  {"SIGABRT",	SIGABRT},
  {"SIGIOT", 	SIGIOT},
  {"SIGBUS",  	SIGBUS},
  {"SIGFPE",  	SIGFPE},
  {"SIGKILL",  	SIGKILL},
  {"SIGUSR1", 	SIGUSR1},
  {"SIGSEGV", 	SIGSEGV},
  {"SIGUSR2", 	SIGUSR2},
  {"SIGPIPE", 	SIGPIPE},
  {"SIGALRM", 	SIGALRM},
  {"SIGTERM", 	SIGTERM},
  {"SIGSTKFLT", SIGSTKFLT},
  {"SIGCHLD",	SIGCHLD},
  {"SIGCONT",	SIGCONT},
  {"SIGSTOP",	SIGSTOP},
  {"SIGTSTP",	SIGTSTP},
  {"SIGTTIN",	SIGTTIN},
  {"SIGTTOU",	SIGTTOU},
  {"SIGURG",	SIGURG},
  {"SIGXCPU",	SIGXCPU},
  {"SIGXFSZ",	SIGXFSZ},
  {"SIGVTALRM", SIGVTALRM},
  {"SIGPROF",	SIGPROF},
  {"SIGWINCH",	SIGWINCH},
  {"SIGIO",	SIGIO},
  {"SIGPWR",	SIGPWR},
  {"SIGSYS",	SIGSYS},
  {"SIGUNUSED", 31},
  {NULL, 0}
};

#define CBFCLEAN_HTAG "CbfClean"

#ifndef __linux__
extern int putenv(const char *);
#endif /* __linux__ */
extern int unlocked_printf(const char *, ...);

static struct sigaction CbfCleanOldAct[NSIG+1]; /*-------- Keep old sigaction */
static int CbfFinished = 0;

static int CbfCleanChainRemove(CbfCleanChain_t **head, CbfCleanChain_t *goner);
static void CbfCleanDoPrint( char *format, ... );
static int CbfCleanGetDebug( char *name );
static char *CbfCleanGetSigList(void);
static CbfHEntry_t *CbfCleanFindEntry( char *fName, void *arg );
static int CbfCleanFindSignal( char *signame );
static CbfClean_t *_CbfCleanFree( CbfClean_t *clean, char *upTag );
static CbfClean_t *_CbfCleanNew( char *name );
static void CbfCleanRootFree( void );
static CbfClean_t *CbfCleanRootNew( void );
static void CbfCleanSigInit( void );
static void CbfCleanSigHandler( int signo );

static CbfCleanPrinter_t CbfCleanPrinter = vprintf;
static CbfClean_t *CbfCleanRoot = (CbfClean_t*)NULL; 
static CbfCleanExitFnt_t CbfExitFnt = (CbfCleanExitFnt_t)NULL;
static const char *CbfCleanSigList = \
  "SIGHUP:SIGINT:SIGQUIT:SIGABRT:SIGPIPE:SIGALRM:SIGTERM:SIGUSR1:SIGUSR2";

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------ CbfCleanChainRemove -*/
/*----------------------------------------------------------------------------*/
static int CbfCleanChainRemove( CbfCleanChain_t **head,
				CbfCleanChain_t *goner ) {
  CbfCleanChain_t *chain;

  if( goner == *head ) { /*------------------- The goner is the first element */
    *head = goner->next; 
  } else { /*------------------------------------- Look for the goner element */
    for( chain = *head;
	(chain != (CbfCleanChain_t *)NULL) && (chain->next != goner);
        chain = chain->next );
    if(chain == (CbfCleanChain_t *)NULL){ return (CBF_FAIL); }
    /*-------------------------------------------------------- Find the goner */
    chain->next = goner->next;
  }

  /*------------------------------------------ Free the CbfCleanChain elememt */
  bzero((char*)goner, sizeof(CbfCleanChain_t));
  free(goner);

  return(CBF_OK);
}

/*----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CbfCleanDoPrint -*/
/*----------------------------------------------------------------------------*/
static void CbfCleanDoPrint( char *format, ... ) {
  va_list args; 

  if( !CbfCleanPrinter || !format ) return;
  /*--------------------------------------------------------------------------*/
  va_start( args, format );
  CbfCleanPrinter( format, args );
  va_end( args );
  /*--------------------------------------------------------------------------*/
  return;
}

/*----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CbfCleanGetDebug -*/
/*----------------------------------------------------------------------------*/
static int CbfCleanGetDebug( char *name ) {
  char *debug;
  char who[CBFCLEAN_MAX];
  int level;

  level = 0; who[0] = '\0';
  debug = getenv("CBFCLEANDEBUG");
  if( debug == (char*)NULL ) return(0);
  sscanf( debug, "%d%s", &level, who );
  if(name == NULL) return(level);
  if(strlen(name) == 0) return(level);
  if(( who[0] != '\0' ) && ( strcmp( name, who ) != 0 )) level = 0;

  return( level );
}


/*----------------------------------------------------------------------------*/
/*------------------------------------------------------- CbfCleanGetSigList -*/
/*----------------------------------------------------------------------------*/
static char *CbfCleanGetSigList(void){
char *sigList = getenv("CBFCLEAN_SIGLIST");

  return(sigList? sigList: (char *)CbfCleanSigList);
}

/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CbfCleanFindEntry -*/
/*----------------------------------------------------------------------------*/
static CbfHEntry_t *CbfCleanFindEntry( char *fName, void *arg ) {
  char  buf[CBFCLEAN_MAX];

  sprintf(buf, "%s:%s_%p", CBFCLEAN_HTAG, fName, arg);  
  return( CbfHFind( (CbfH_t*)NULL, buf) );
}

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------- CbfCleanFindSignal -*/
/*----------------------------------------------------------------------------*/
static int CbfCleanFindSignal(char *signame){
register int found = 0;
char *sigList = strdup(CbfCleanGetSigList());
char *name;

  if((name = strtok(sigList, ":")) != (char *)NULL){
    if(strcmp(name, signame) == 0){ found = 1; }
    else{
      while((name = strtok((char *)NULL, ":")) != (char *)NULL){
        if(strcmp(name, signame) == 0){
          found = 1;
          break;
        }
      }
    }
  }

  free(sigList);

  return(found);
}

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------ _CbfCleanFree -*/
/*----------------------------------------------------------------------------*/
static CbfClean_t *_CbfCleanFree( CbfClean_t *clean, char *upTag ) {
  CbfCleanChain_t *chain;
  int i;
 
  if( CbfExitFnt ) { CbfExitFnt(); CbfExitFnt = (CbfCleanExitFnt_t)NULL; }
  /*---------------------------------------------------- Lock the clean chain */
  pthread_mutex_lock(&clean->mutex);
  /*------------------------------------- Free the clean chained list element */
  for( chain = clean->chain; chain != (CbfCleanChain_t*)NULL; 
       chain = clean->chain ) {
    /*--------------------------------------------------------- Check the tag */
    if( upTag && !strcmp(upTag, chain->name)) break;
    /*------------------------------------------------ Unlock the clean chain */
    pthread_mutex_unlock(&clean->mutex);
    /*------------------------------------------------------------------ Loop */
    for( i = chain->loop; i > 0; i-- ){
      chain->function( chain->arg ); /*--------- Execute the cleanup function */
    }
    /*-------------------------------------------------- Lock the clean chain */
    pthread_mutex_lock(&clean->mutex);      
    if( chain != clean->chain ) continue;
    /*--------------------------------------------------------- Cleanup error */
    CbfCleanDoPrint( "CbfClean> %-13s> _CbfCleanFree: %s Fail\n", clean->name,
		     chain->name );
    CbfCleanChainRemove( &clean->chain, chain );
  }
  /*-------------------------------------------------- Unlock the clean chain */
  pthread_mutex_unlock(&clean->mutex);

  return(clean);
}

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------- _CbfCleanNew -*/
/*----------------------------------------------------------------------------*/
static CbfClean_t *_CbfCleanNew( char *name ) {
  int length;
  CbfClean_t *clean;

  /*------------------------------------------ Allocate the cleanup structure */
  if((clean = (CbfClean_t*)malloc(sizeof(CbfClean_t))) == (CbfClean_t*)NULL ) {
    CbfCleanDoPrint("CbfClean> CbfCleanNew: malloc failed");
    return(clean);
  }
  bzero((char*)clean, sizeof(CbfClean_t));
  /*----------------------------------------- Fill the name and nbElem fields */
  length = strlen(name);
  if( length >= CBFCLEAN_MAX ) length = CBFCLEAN_MAX-1;
  strncpy( clean->name, name, length );
  clean->name[length] = '\0';

  /*-------------------------------------------------- Initialalize the mutex */
#ifdef RIO806X /*----------------------------------------------------- LynxOs */
  if( pthread_mutex_init( &clean->mutex, pthread_mutexattr_default) == -1 ) {
#else         /*-------------------------------------------------------- Osf1 */
  if( pthread_mutex_init( &clean->mutex, NULL ) == -1 ){
#endif
    CbfCleanDoPrint("CbfClean> CbfCleanNew: pthread_mutex_init failed\n");
    clean = CbfCleanFree(clean);
  }
  
  return(clean);
}

/*----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CbfCleanFreeRoot -*/
/*----------------------------------------------------------------------------*/
static void CbfCleanRootFree( void ) {
  CbfCleanRoot = CbfCleanFree( CbfCleanRoot );
  return;
}

/*----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CbfCleanRootNew -*/
/*----------------------------------------------------------------------------*/
static CbfClean_t *CbfCleanRootNew( void ) {
  if( CbfCleanRoot ) return(CbfCleanRoot);

  CbfHNew(CBFHROOT); /*------------- Needed The root hTable should be created */
  /*----------------------------------------------------- Create CbfCleanRoot */
  CbfCleanRoot = _CbfCleanNew("CbfCleanRoot");
  CbfCleanSigInit();
  CbfCleanPrint( CbfCleanRoot, "CbfCleanRoot created\n" );
  return( CbfCleanRoot );
}

/*----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CbfCleanSigInit -*/
/*----------------------------------------------------------------------------*/
static void CbfCleanSigInit( void ) {
register int i, debug;
struct sigaction act, oldAct;

  if(!CbfCleanRoot){ return; }
  
  /*--------------------------------------------------------------------------*/
  atexit(CbfCleanRootFree);

  /*------------------------------------------ Initialize the signal handling */
  sigemptyset(&act.sa_mask);
  act.sa_handler = CbfCleanSigHandler;
  act.sa_flags   = 0;

  debug = CbfCleanGetDebug((char *)NULL);

  for(i=0; CbfSigDesc[i].name != (char *)NULL; i++){
    int sig;

    if(!CbfCleanFindSignal(CbfSigDesc[i].name)){
      if( debug >= -1 ) continue;
      CbfCleanDoPrint( "CbfClean> CbfCleanSigInit: Signal %9s %2d\n",
		       CbfSigDesc[i].name, CbfSigDesc[i].number);
    }

    sig = CbfSigDesc[i].number;

    /*---------------------------------------------- Install CbfClean Handler */
    if(sigaction(sig, &act, &oldAct) == -1) {
      CbfCleanDoPrint( "CbfClean> CbfCleanSigInit: sigaction(CbfClean handler)"
		       "failed for signal %s\n", CbfSigDesc[i].name );
      continue;
    }
    
    /*---------------------- Check if the previous handler is the default one */
    if( oldAct.sa_handler == CbfCleanSigHandler ) continue;

    /*-------------------------------------------- Store the previous handler */
    memcpy(&CbfCleanOldAct[sig], &oldAct, sizeof(struct sigaction));
    if( CbfCleanOldAct[sig].sa_handler != SIG_DFL ) {
      int rc;

      if(debug < -1) {
	CbfCleanDoPrint("CbfClean> CbfCleanSigInit: Signal %9s %2d catch by"
			" user's handler\n",
			CbfSigDesc[i].name, CbfSigDesc[i].number);
      } else {
	CbfCleanDoPrint("CbfClean> CbfCleanSigInit: Uninstall CbfClean handler"
			" for signal %s\n", CbfSigDesc[i].name );
      }
      /*------------------------------------------ Restore the User's handler */
      rc = sigaction(sig, &CbfCleanOldAct[sig], (struct sigaction *)NULL);
      if( rc == -1 ) {
	CbfCleanDoPrint( "CbfClean> CbfCleanSigInit: Unable to restore"
			 " User's handler for signal %s\n",CbfSigDesc[i].name );
      }
    } else if(debug < -1 ) {
	CbfCleanDoPrint("CbfClean> CbfCleanSigInit: Signal %9s %2d catch by"
			" CbfClean handler\n",
			CbfSigDesc[i].name, CbfSigDesc[i].number);
    }
  }

  return;
}

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------- CbfCleanSigHandler -*/
/*----------------------------------------------------------------------------*/
static void CbfCleanSigHandler( int signo ) {

  CbfCleanDoPrint("CbfClean> CbfCleanSigHandler receives signal %d\n", signo );
  CbfTerminate();
}

/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CbfCleanAdd -*/
/*----------------------------------------------------------------------------*/
CbfCleanChain_t *CbfCleanAdd( CbfClean_t *clean, char *fName,
			      void *(*function)(), void *arg ) {
  char buf[CBFCLEAN_MAX];
  int length;
  CbfCleanChain_t *chain;

  /*-------------------------------------------------------- Find the element */
  chain = CbfCleanFind( fName, arg);
  if( !chain ) {  /*--------------------------------------------- New element */
    /*-------------------------------------------------- Find the Clean chain */
    if( !clean && !(clean = CbfCleanRootNew())) return((CbfCleanChain_t *)NULL);
    /*------------------------------------ Allocate a CbfCleanChain_t element */
    if( (chain = (CbfCleanChain_t*)malloc(sizeof(CbfCleanChain_t)))
	== (CbfCleanChain_t*)NULL ){
      CbfCleanDoPrint("CbfClean> CbfCleanAdd: malloc failed\n");
      return(chain);
    }
    /*---------------------------------------------- Add it to the hash table */
    sprintf( buf, "%s:%s_%p", CBFCLEAN_HTAG, fName, arg );
    if( CbfHEnter( (CbfH_t*)NULL, buf, chain ) == (CbfHEntry_t*)NULL) {
      free(chain); 
      return((CbfCleanChain_t *)NULL);
    }
    /*----------------------------- Fill the CbfCleanChain_t structure fields */
    sprintf( buf, "%s_%p", fName, arg );
    length = strlen(buf);
    if( length >= CBFCLEAN_MAX ) length = CBFCLEAN_MAX-1;
    strncpy( chain->name, buf, length );
    chain->name[length] = '\0';
    chain->function = function;
    chain->arg      = arg;
    chain->loop     = 1;
    chain->cleanRef = clean;
    chain->next     = (CbfCleanChain_t*)NULL;
  
    /*----------------------------------------------- Update the chained list */
    /*-------------------------------------------------- Lock the clean chain */
    pthread_mutex_lock(&clean->mutex);
    chain->next = clean->chain;
    clean->chain = chain;
    /*------------------------------------------------ Unlock the clean chain */
    pthread_mutex_unlock(&clean->mutex);

  } else chain->loop++;

  CbfCleanPrint( clean, "CbfCleanAdd: %s(%d) (%p)\n", chain->name, chain->loop,
		 chain );
  return( chain );

 }

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CbfCleanDone -*/
/*----------------------------------------------------------------------------*/
int CbfCleanDone( char *fName, void *arg ) {
  CbfCleanChain_t *chain;
  CbfClean_t *clean;
  CbfHEntry_t *entry;
  int loop, rtn;

  /*------------------------------------------ Check the argument persistency */
  if( !(entry = CbfCleanFindEntry(fName, arg)) ) {
    CbfCleanDoPrint( "CbfClean> CbfCleanDone: %s_%p no exist\n", fName, arg);
    return(CBF_EXIT);
  }
  chain = (CbfCleanChain_t *)entry->data;
  /*--------------------------------------------------- Find the Clean chain */
  clean = chain->cleanRef;
  /*--------------------------------------------------------Treat the element */
  loop = chain->loop;
  if( loop > 1 ) { /*----------------------------- Many same element to clean */
    chain->loop--;
    rtn = CBF_OK;
  } else {        /*----------------------------------- Last element to clean */
    /*-------------------------------------------------- Lock the clean chain */
    pthread_mutex_lock(&clean->mutex);
    rtn = CbfCleanChainRemove( &clean->chain, chain );
    /*------------------------------------------------ Unlock the clean chain */
    pthread_mutex_unlock(&clean->mutex);
    CbfHEntryRemove( entry );  /*------------------ Remove the hTable element */
  }

  CbfCleanPrint( clean, "CbfCleanDone: %s_%p(%d) (%p) %s\n", fName, arg, loop,
		 chain,  ((rtn == CBF_OK) ? "Ok" : "Fail"));

  return( rtn );
}

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CbfCleanFind -*/
/*----------------------------------------------------------------------------*/
CbfCleanChain_t *CbfCleanFind( char *fName, void *arg ) {
  CbfHEntry_t *entry;
  CbfCleanChain_t *chain = ( CbfCleanChain_t *)NULL;

  entry = CbfCleanFindEntry( fName, arg );
  if( entry ) chain = (CbfCleanChain_t *)entry->data;
  return(chain);
}


/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CbfCleanFree -*/
/*----------------------------------------------------------------------------*/
CbfClean_t *CbfCleanFree( CbfClean_t *clean ) {
  if( !clean ) return((CbfClean_t*)NULL);

  if( (clean != CbfCleanRoot) && !CbfCleanFind("CbfCleanFree", clean) ) {
    return((CbfClean_t*)NULL);
  }
  /*-------------------------------------- Remove from the CbfCleanRoot chain */
  CbfCleanPrint( clean, "CbfCleanFree: started\n");
  clean = _CbfCleanFree(clean, NULL);
  /*-------------------------------------------- Release CbfClean_t structure */
  if( clean != CbfCleanRoot ) CbfCleanDone( "CbfCleanFree", clean );
  else CbfCleanRoot = (CbfClean_t*)NULL;

  CbfCleanPrint( clean, "CbfCleanFree: Terminated \n");
  bzero((char*)clean, sizeof(CbfClean_t));
  free( clean ); clean = (CbfClean_t*)NULL; 
  return( clean );
}


/*----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CbfCleanFreeTag -*/
/*----------------------------------------------------------------------------*/
CbfClean_t *CbfCleanFreeTag( char *fName, void *arg ) {
  CbfCleanChain_t *chain;
  CbfClean_t *clean;
  
  if( !(chain = CbfCleanFind( fName, arg)) ) return(CbfClean_t*)NULL;
  /*-------------------------------------------- Take the cleanup descpriptor */
  clean = chain->cleanRef;
  CbfCleanPrint( clean,"CbfCleanFreeTag: started to %s\n", 
		 clean->chain->name );

  /*---------------------------------- Free the cleanup chain upto <fName_arg>*/
  clean = _CbfCleanFree( clean, chain->name );
  CbfCleanPrint( clean, "CbfCleanFreeTag: Terminated before %s\n",
		 chain->name );
  return(clean);
}

/*----------------------------------------------------------------------------*/
/*---------------------------------------------------- CbfCleanInstallPrinter */
/*----------------------------------------------------------------------------*/
CbfCleanPrinter_t CbfCleanInstallPrinter( CbfCleanPrinter_t printer ) {
  CbfCleanPrinter_t old;

  /*--------------------------------------------------------------------------*/
  old = CbfCleanPrinter;
  CbfCleanPrinter = printer;
 /*--------------------------------------------------------------------------*/
  return(old);
}

/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CbfCleanNew -*/
/*----------------------------------------------------------------------------*/
CbfClean_t *CbfCleanNew( char *name ) {  
  CbfClean_t *clean;
  
  if( !name || !name[0] ) return(CbfCleanRootNew());
  clean = _CbfCleanNew( name );
  /*------------------------------------------- Add to the CbfCleanRoot chain */
  CbfCleanAdd( NULL, "CbfCleanFree", (CbfCleanFnt_t)CbfCleanFree, clean );
  return(clean);
}

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CbfCleanPrint -*/
/*----------------------------------------------------------------------------*/
void CbfCleanPrint( CbfClean_t *clean, char *format, ... ) {
  va_list args;
  CbfCleanChain_t *chain;
  int debug, i;

  /*---------------------------------------------------- Find the Clean chain */
  if( !clean && !(clean = CbfCleanRootNew()) ) return;
  /*------------------------------------------------- Treat the format string */
  if( (debug = CbfCleanGetDebug( clean->name )) < 1 ) return;
  CbfCleanDoPrint("CbfClean> %-13s>", clean->name );
  va_start( args, format );
  if( CbfCleanPrinter ) CbfCleanPrinter( format, args );
  va_end( args );
  

  /*----------------------------------------------- Treat CbfClean descriptor */
  if( (debug = CbfCleanGetDebug( clean->name )) < 2 ) return;

  /*---------------------------------------------------- Lock the clean chain */
  pthread_mutex_lock(&clean->mutex);
  for( i=0, chain = clean->chain; chain != (CbfCleanChain_t*)NULL;
       chain = chain->next, i++ ) {
    CbfCleanDoPrint("CbfClean> %-13s> El: %d(%p) name: %s(%d)\n",
		   clean->name,  i, chain, chain->name, chain->loop );   
  }
  /*-------------------------------------------------- Unlock the clean chain */
  pthread_mutex_unlock(&clean->mutex);

  if( debug > 2 ) _CbfHPrint( (CbfH_t*)NULL, CBFCLEAN_HTAG );

}

/*----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CbfCleanSetDebug -*/
/*----------------------------------------------------------------------------*/
void CbfCleanSetDebug( int debug, char *name ) {
  char str[CBFCLEAN_MAX];

  sprintf( str, "CBFCLEANDEBUG=%d %s", debug, (name? name: "") );
  CbfCleanDoPrint("CbfClean> Set environment variable \"%s\"\n", str );
  putenv( str );
  /*--------------------------------------------------------------------------*/
  return;
}

/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CbfCleanSetExitFnt */
/*----------------------------------------------------------------------------*/
CbfCleanExitFnt_t CbfCleanSetExitFnt( CbfCleanExitFnt_t exitFnt ) {
  CbfCleanExitFnt_t pExitFnt;

  /*--------------------------------------------------------------------------*/
  pExitFnt = CbfExitFnt;
  CbfExitFnt = exitFnt;
  /*--------------------------------------------------------------------------*/
  return( pExitFnt );
}


/*----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CbfCleanFinished */
/*----------------------------------------------------------------------------*/
int CbfCleanFinished( void ) {

  /*--------------------------------------------------------------------------*/
  /*--------------------------------------------------------------------------*/
  return(CbfFinished);
}


/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CbfTerminate */
/*----------------------------------------------------------------------------*/void CbfTerminate( void ) {

  /*--------------------------------------------------------------------------*/
  CbfFinished = 1;
  /*--------------------------------------------------------------------------*/
  return;
}
