/* SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Cfg - A library for Virgo process implementation
 *
 * Copyright © 2001-2021 Laboratoire de physique des particules d'Annecy - CNRS
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Fatih Bellachia <fatih.bellachia@lapp.in2p3.fr>
 *   Laurent Fournier <laurent.fournier@lapp.in2p3.fr>
 *   Alain Masserot <alain.masserot@lapp.in2p3.fr>
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <CbfHash.h>
#ifdef USE_MW
#include <memwatch.h>
#endif /* USE_MW */


extern int unlocked_printf(const char *, ...);
static void CbfHDoPrint( char *format, ... );
static unsigned int CbfHashKey( char *text, int nElem );
static CbfH_t *_CbfHFree( CbfH_t *hTable );
static void CbfHRootExit( void );

/*
 * table of primes just below 2^n, n=2..31 for use in finding the right prime
 * number to be the table size.  this table may be generally useful...
*/
static unsigned int CbfHSizeTab[] = {
  127, 251, 509, 1021, 2039, 4093, 8191, 16381, 32749, 65521, 131071,
  262139, 524287, 1048573, 2097143, 4194301, 8388593, 16777213, 33554393,
  67108859, 134217689, 268435399, 536870909, 1073741789, 2147483647
};

static CbfH_t *CbfHRoot = (CbfH_t *)NULL;
static CbfHPrinter_t CbfHPrinter = vprintf;

/*----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CbfHDoPrint */
/*----------------------------------------------------------------------------*/
static void CbfHDoPrint( char *format, ... ) {
  va_list args;

  if( !CbfHPrinter || !format ) return;
  /*--------------------------------------------------------------------------*/
  va_start( args, format );
  CbfHPrinter( format, args );
  va_end( args );
  /*--------------------------------------------------------------------------*/
  return;
}

/*----------------------------------------------------------------------------*/
/*------------------------------------ CbfHashit --- do the hashing algorithm */
/*----------------------------------------------------------------------------*/
static unsigned int CbfHashKey( char *text, int nElem ){
  register unsigned int sum = 0;
  register int i, tmp;

  for(i = 0; text[i] != '\0'; i++){
    sum = (sum << 4) + text[i];

    if((tmp = (sum & 0xF0000000))){
      sum = sum ^ (tmp >> 24);
      sum = sum ^ tmp;
    }
  }

  return(sum % nElem);
}

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- _CbfHFree */
/*----------------------------------------------------------------------------*/
static CbfH_t *_CbfHFree( CbfH_t *hTable ){
  CbfHEntry_t *entry, *pEntry;
  int hI;

  if( !hTable ) return((CbfH_t *)NULL);
  /*--------------------------------------------------------- Lock the hTable */
  pthread_mutex_lock(&hTable->mutex);
  /*---------------------------------------------- Free all available entries */
  for( hI = 0; hI < hTable->nElem; hI++ ){
    for( entry = hTable->entry[hI]; entry != (CbfHEntry_t *)NULL; 
	 free( pEntry ) ){
      pEntry = entry;
      entry  = pEntry->next;
      CbfHEntryRemove( pEntry );
    }
  }
  /*---------------------------------------------- Free hash table descriptor */
  free( hTable->entry ); 
  hTable->entry = (CbfHEntry_t **)NULL;
  /*------------------------------------------------------- Unlock the hTable */
  pthread_mutex_unlock(&hTable->mutex);
  free( hTable );

  return((CbfH_t *)NULL);
}

/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CbfHRootExit */
/*----------------------------------------------------------------------------*/
static void CbfHRootExit(void) {
  CbfHPrint( CbfHRoot );
  CbfHRoot = _CbfHFree( CbfHRoot ); /*------------------------- Free CbfHRoot */
  return;
}

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CbfHEnter */
/*----------------------------------------------------------------------------*/
CbfHEntry_t *CbfHEnter( CbfH_t *hTable, char *key, void *data ){
  CbfHEntry_t *entry;
  int hI;

  if( !hTable && !(hTable = CbfHNew(CBFHROOT)) ){
    CbfHDoPrint( "CbfH> CbfHEnter: Invalid hTable pointer\n" );
    return((CbfHEntry_t *)NULL);
  }
  /*--------------------------------------------------------- Lock the hTable */
  pthread_mutex_lock(&hTable->mutex);
  /*--------------------------------------------------------------------------*/
  hI = CbfHashKey( key, hTable->nElem );
  entry = hTable->entry[hI];

  /*----------------------------------------------- First element at this key */
  if( entry == (CbfHEntry_t *)NULL ){ 
    hTable->entry[hI] = (CbfHEntry_t*) malloc( sizeof(CbfHEntry_t) );
    entry  = hTable->entry[hI];
    if( entry == (CbfHEntry_t*)NULL ){
      CbfHDoPrint( "CbfH> CbfHEnter: malloc failed\n" );
      /*--------------------------------------------------- Unlock the hTable */
      pthread_mutex_unlock(&hTable->mutex);
      return(entry);
    }
    bzero((char*)entry, sizeof(CbfHEntry_t) );
  }

  /*------------------------------ Scan the chain list to find an empty place */
  for( ; entry != (CbfHEntry_t *)NULL;  entry = entry->next ){
    if( entry->key == NULL ) break;  
    if( strcmp( key, entry->key ) == 0 ) {
      CbfHDoPrint("CbfH> CbfHEnter: key %s already defined\n", key );
      /*--------------------------------------------------- Unlock the hTable */
      pthread_mutex_unlock(&hTable->mutex);
      return((CbfHEntry_t *)NULL);
    }
  }
  
  if( !entry ){ /*------------------------ Allocate a new CbfHEntrie structre */
    entry = (CbfHEntry_t*) malloc( sizeof(CbfHEntry_t) );
    if( entry == (CbfHEntry_t*)NULL ){
      CbfHDoPrint( "CbfH> CbfHEnter: malloc failed\n" );
      /*--------------------------------------------------- Unlock the hTable */
      pthread_mutex_unlock(&hTable->mutex);
      return(entry);
    }
    bzero((char*)entry, sizeof(CbfHEntry_t) );
    /*---------------- Chain List manage as a stak for cleanup functionnality */
    entry->next = hTable->entry[hI]; 
    hTable->entry[hI] = entry;
  }
  
  /*--------------------------------------- Store the string key and the data */
  entry->key = strdup(key);
  entry->data = data;

  /*------------------------------------------------------- Unlock the hTable */
  pthread_mutex_unlock(&hTable->mutex);

  return( entry );
}

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CbfHEntryRemove */
/*----------------------------------------------------------------------------*/
void CbfHEntryRemove( CbfHEntry_t *entry ){
  if( !entry ) return;
  if( entry->key ) free( entry->key ); /*----------------------- Free the key */
  entry->key  = NULL;
  entry->data = NULL;  
  return;
}

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------------ CbfHFind */
/*----------------------------------------------------------------------------*/
CbfHEntry_t *CbfHFind( CbfH_t *hTable, char *key ){
  CbfHEntry_t *entry;
  int hI;

  if( !key ) return((CbfHEntry_t *)NULL);
  if( !hTable && !(hTable = CbfHNew(CBFHROOT)) ) return((CbfHEntry_t *)NULL);
  /*--------------------------------------------------------- Lock the hTable */
  pthread_mutex_lock(&hTable->mutex);
  /*--------------------------------------------------------------------------*/
  hI = CbfHashKey( key, hTable->nElem );
  for( entry = hTable->entry[hI]; entry != (CbfHEntry_t *)NULL;
       entry = entry->next ){
    if( entry->key == NULL ) continue;            /*------------------- Empty */
    if( strcmp(entry->key, key) == 0 ) break;     /*--------- Same key string */
  }
  /*------------------------------------------------------- Unlock the hTable */
  pthread_mutex_unlock(&hTable->mutex);

  return( entry );
}

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CbfHFree */
/*----------------------------------------------------------------------------*/
CbfH_t *CbfHFree( CbfH_t *hTable ){
  if( hTable == CbfHRoot ) return(hTable);
  return(_CbfHFree(hTable));
}

/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CbfHInstallPrinter */
/*----------------------------------------------------------------------------*/
CbfHPrinter_t CbfHInstallPrinter( CbfHPrinter_t printer ){
  CbfHPrinter_t old;

  /*--------------------------------------------------------------------------*/
  old = CbfHPrinter;
  CbfHPrinter = printer;
  /*--------------------------------------------------------------------------*/
  return(old);
}

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------------- CbfHNew */
/*----------------------------------------------------------------------------*/
CbfH_t *CbfHNew( int nElem ){
  CbfH_t *hTable;
  int i, pN;

  if( (nElem <= CBFHROOT) && CbfHRoot ) return( CbfHRoot );
  /*--------------------------------------- Allocate an hash table descriptor */
  if( (hTable = (CbfH_t*)malloc(sizeof(CbfH_t))) == (CbfH_t*)NULL ){
    CbfHDoPrint("Cbf> CbfHNew: malloc failed\n");
    return(hTable);
  }
  bzero( (char*)hTable, sizeof(CbfH_t));

  /*-------------------------------------------------- Initialalize the mutex */
#ifdef RIO806X /*----------------------------------------------------- LynxOs */
  if( pthread_mutex_init( &hTable->mutex, pthread_mutexattr_default) == -1 ){
#else         /*-------------------------------------------------------- Osf1 */
  if( pthread_mutex_init( &hTable->mutex, NULL ) == -1 ){
#endif
    CbfHDoPrint("CbfH> CbfHNew: pthread_mutex_init failed\n");
    free( hTable );
    return((CbfH_t*)NULL);
  }

  /*--------------------------------------------- Find the corresponding size */
  if( nElem <= CBFHROOT ) hTable->nElem = (abs(nElem) < 509 ? 509 : abs(nElem));

  pN = sizeof (CbfHSizeTab) / sizeof(CbfHSizeTab[0]);
  for( i = 0; (i < pN) && (CbfHSizeTab[i] < hTable->nElem); i++ );
  if( i == pN ) hTable->nElem = nElem;
  else hTable->nElem = CbfHSizeTab[i];
  
  /*------------------------------------ Allocate the Hash table entrie array */
  hTable->entry = (CbfHEntry_t**) calloc( hTable->nElem, sizeof(CbfHEntry_t*) );
  if( hTable->entry == (CbfHEntry_t**) NULL ){
    CbfHDoPrint("CbfH> CbfHNew: calloc failed\n");
    hTable = _CbfHFree( hTable );
  }
  /*----------------------- Fill CbfHRoot pointer if nElem less than CBFHROOT */
  if( (nElem <= CBFHROOT) && (CbfHRoot = hTable) ) atexit( CbfHRootExit );

  return( hTable );
}

/*----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- _CbfHPrint */
/*----------------------------------------------------------------------------*/
void _CbfHPrint( CbfH_t *hTable, char *tag ){
  CbfHEntry_t *entry;
  int hI, i, header = 1 ;

  if( !hTable && !(hTable = CbfHNew(CBFHROOT)) ) return;
  /*--------------------------------------------------------- Lock the hTable */
  pthread_mutex_lock(&hTable->mutex);
  for( hI = 0; hI < hTable->nElem; hI++ ){
    entry = hTable->entry[hI];
    for( i = 0; entry != (CbfHEntry_t *)NULL;  entry = entry->next, i++ ){
      if( !entry->data ) continue;
      if( header ) {
	CbfHDoPrint("CbfH>%sSize: %d - Tag %s .....................\n", 
		   (hTable == CbfHRoot ? " CbfHRoot - " : " " ),
		   hTable->nElem, (tag && tag[0] ? tag : "All"));
	header = 0;
      }
      if( tag && tag[0] && !strstr(entry->key,tag) ) continue;
      CbfHDoPrint("CbfH> Key: %s(%d) (%d)- Data: %p - Next: %p\n",
		  entry->key, hI, i, entry->data, entry->next );
    }
  }
  /*------------------------------------------------------- Unlock the hTable */
  pthread_mutex_unlock(&hTable->mutex);
}

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CbfHPrint */
/*----------------------------------------------------------------------------*/
void CbfHPrint( CbfH_t *hTable ) { 
  _CbfHPrint( hTable, NULL );
  return;
}
