/* SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Cfg - A library for Virgo process implementation
 *
 * Copyright © 2001-2021 Laboratoire de physique des particules d'Annecy - CNRS
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Fatih Bellachia <fatih.bellachia@lapp.in2p3.fr>
 *   Laurent Fournier <laurent.fournier@lapp.in2p3.fr>
 *   Alain Masserot <alain.masserot@lapp.in2p3.fr>
 */

#define _GNU_SOURCE

#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <signal.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <netdb.h>
#include <pthread.h>
#include <sched.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/param.h>
#include <ctype.h>
#include <sys/resource.h>
#include <sys/file.h>
#include <CfgSnprintf.h>

#include <CfgCommon.h>
#include <CfgMsg.h>


#ifdef USE_MW
#include <memwatch.h>
#endif /* USE_MW */

#define CFG_HOUR 3600
#define CFG_DAY  (24 * CFG_HOUR)


extern void psignal (int, const char *);

struct CfgData {
  int argc;                    /*------------ number of command line arguments */
  char **argv;                 /*---------------------- command line arguments */
  int debug;                   /*--------------------- application debug level */
  time_t startTime;            /*------------------------------- Starting time */
  int pid;                     /*---------------------------------- Process Id */
  pthread_t tid;               /*------------------------------ Main Thread Id */
  int prio;                    /*---------------------------- Process priority */
  char hostname[64];           /*----------------------------------- Host name */
  char activation[MAXPATHLEN]; /*------------------ Activation path and server */
  char *config;                /*------------------------ Configuration buffer */
  int configLen;               /*------------ Size of the Configuration buffer */
  char *configBuilt;
  char *server;                /*----------------------------------Server name */
  char *tag;                   /*--------------------------- Configuration tag */
  char *cmName;                /*------------------------------------- Cm name */
  int cmNameSize;              /*--------------------------------- CmName Size */
  char *sysName;               /*----------------------------- Sub system name */
  CfgBool_t noDbSave;          /*------------------------- True: no save on Db */
  CfgBool_t noFileSave;        /*----------------------- True: no save as file */
  int state;                   /*------------------- current application state */
  int reqState;                /*----------------------------- Requested state */
  CfgMsgR_t *msgR;             /*---------------------------- Message reporter */
  CfgMsg_t *uiInfo;            /*------------------------- User's info message */
  pthread_mutex_t uiMutex;     /*--------------- Mutex for User's info message */
  int msgDbgMin;
  CfgBool_t stdoutFlag;        /*---------------------- True: report to stdout */
  CfgBool_t logFlag;           /*------------------ True: write to the logFile */
  char *logFilename;           /*-------------- logFileName(=cmName+startTime) */
  FILE *logStream;             /*---------------------------------- log stream */
  int cmTrace;                 /*------------------------------ Cm Trace level */
  time_t currentTime;          /*-------------------------------- Current time */
  time_t configTime;           /*--------------------------------- Config time */
  time_t lastTime;             /*------ Last time of a CfgMsg report to Log/EL */
  double periodTime;           /*-------------------- Period starting with 10s */
  CfgStateStrFnt_t stateStrFnt;/*----------------------- State string function */
  CfgConfigFnt_t configFnt;    /*--------------- Configuration string function */
  size_t memSize;              /*------------------------- Current Memory size */
  int msgLength;               /*----------------------- Stdout Message length */
  char strTime[50];            /*--------------------------------- Time string */
  char *str;                   /*-- String use to write inside the main thread */
  int strSize;
  double maxPeriodTime;        /*-------------------------------- Needed by Cl */
  int flush;                   /*-------------------------------Flush the MsgR */
  char *schedAffinity;
#ifdef Linux
  char *procFd;
#endif
  CfgPrinter_t *cfgPrn;
  int prnMsgLvl;
  CfgStateReachFnt_t stateReachFnt[CfgServerEnd];
  void *stateReachArg[CfgServerEnd]; 
  /*--------------------------------------------------------------------------*/
  CfgUserActivationGetFnt_t getActivation;
  void *getActiParam;
  char *userActivation;
  double cmCleanupTmo;
 };



/*----------------------------------------------------------------- Cm Printer */
struct CfgPrinter {
  char *buf;
  char *pBuf;
  int pSize;
  int size;
};

CfgData_t *CfgData; 

static char *CfgServerStateStr[CfgServerEnd+1]=
  { "Idle", "Configured", "Active", "Golden", "Error", "End" };

/*-------------------------------------------------------- Internal Cm Handler */
static CfgPrinter_t *CfgPrinterFree( CfgPrinter_t *cfgPrn );
static  CfgPrinter_t *CfgPrinterNew( int size );
static int CfgDataSetPrnMsgLvl( CfgData_t *p, int msgLvl );


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CfgPrinterFree */
/*-----------------------------------------------------------------------------*/
static CfgPrinter_t *CfgPrinterFree( CfgPrinter_t *p ) {

  if( !CbfCleanFind("CfgPrinterFree", p) ) return(p);
  /*---------------------------------------------------------------------------*/
  if( p->buf ) { free( p->buf ); p->buf = NULL; }
  memset((char*)p, 0, sizeof(CfgPrinter_t)); free(p);
  /*---------------------------------------------------------------------------*/
  CbfCleanDone("CfgPrinterFree", p);
  return((CfgPrinter_t *)NULL);
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgPrinterNew */
/*-----------------------------------------------------------------------------*/
static CfgPrinter_t *CfgPrinterNew( int size ) {
  CfgPrinter_t *p = (CfgPrinter_t *)NULL;

  if( size == 0 ) size = CFG_MSG_BUF_SIZE;
  /*---------------------------------------------------------------------------*/
  if( (p =  (CfgPrinter_t*)malloc(sizeof(CfgPrinter_t))) ==
      (CfgPrinter_t *)NULL ) return(p);
  memset((char*)p, 0, sizeof(CfgPrinter_t));
  CbfCleanAddRoot( "CfgPrinterFree", (CbfCleanFnt_t)CfgPrinterFree, p );
  /*---------------------------------------------------------------------------*/
  if( (p->buf = malloc(size)) == NULL ) p = CfgPrinterFree(p);
  else memset(p->buf, 0, size);
  /*------------------------------------------- Initialize CfgPrinter_t fields */
  p->size   = size-1;
  p->pBuf   = p->buf;
  p->pSize  = p->size;
  /*---------------------------------------------------------------------------*/
  return(p);
}



/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CfgDataSetPrnMsgLvl */
/*-----------------------------------------------------------------------------*/
static int CfgDataSetPrnMsgLvl( CfgData_t *p, int msgLvl ) {
  int pLvl;

  if( !p ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  pLvl = p->prnMsgLvl;
  p->prnMsgLvl = msgLvl;
  /*---------------------------------------------------------------------------*/
  return(pLvl);
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgBasename */
/*-----------------------------------------------------------------------------*/
char *CfgBasename(const char *name){
char *pstr = strrchr (name, '/');

  return(pstr? pstr+1: (char *)name);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------------ CfgCmName */
/*-----------------------------------------------------------------------------*/
char *CfgCmName( int argc, char *argv[] ) {
  int lastArgc;
  char *cmName=NULL, *strp, *lastArgv;

  if( !argc || !(lastArgc = argc-1) ) return(NULL);
  /*------------------------------------------------------ Last argument exist */
  lastArgv = strdup(argv[lastArgc]);
  if((strp = strrchr(lastArgv, '.')) != (char *)NULL){
   strp[0] = '\0';
  }
  cmName = strdup(CfgBasename(lastArgv));
  free(lastArgv); /*----------------------------------------- Release lastArgv */

  return(cmName);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CfgDirname */
/*-----------------------------------------------------------------------------*/
char *CfgDirname(char *path){
static const char dot[] = ".";
char *last_slash;

  /*------------------------------------------------------------ Find last '/' */
  last_slash = (path != (char *)NULL)? strrchr (path, '/'): (char *)NULL;

  if(last_slash == path){
    /*---------------------- The last slash is the first character in the string
                            We have to return "/" */
    ++last_slash;
  }
  else{
    if((last_slash != NULL) && (last_slash[1] == '\0')){
      /*--------------- The '/' is the last character, we have to look further */
      last_slash = memchr(path, last_slash - path, '/');
    }
  }

  if(last_slash != (char *)NULL){
    /*----------------------------------------------------- Terminate the path */
    last_slash[0] = '\0';
  }
  else{
    /*----------- This assignment is ill-designed but the XPG specs require to
                 return a string containing "." in any case no directory part
                 is found and so a static and constant string is required */
    path = (char *)dot;
  }

  return(path);
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgFindPath */
/*-----------------------------------------------------------------------------*/
char *CfgFindPath(char *filename, size_t len, char *pname, int mode){
static char fname[MAXPATHLEN];
char *path, *pstr;

  if(pname == (char *)NULL){
    fprintf(stderr, __FUNCTION__": invalid pointer argument\n");
    return(filename);
  }

  /*------------------------------------------------------------- Flush string */
  memset(filename, 0, len);

  /*------------------------------------- Try to resolve an absolute path name */
  if(realpath(pname, fname) != (char *)NULL){
    if(access(fname, mode) == 0){ return(strncpy(filename, fname, len)); }
  }

  /*---------------------------------------- Get the PATH environment variable */
  if((path = getenv("PATH")) == (char *)NULL){
    perror("getenv");
    return(strncpy(filename, pname, len));
  }

  /*--------------------------- Searches through the directories named in PATH */
  if((pstr = strtok(path, ":")) != (char *)NULL){
    char buffer[MAXPATHLEN];

    sprintf(buffer, "%s/%s", pstr, pname);
    /*----------------------------------- Try to resolve an absolute path name */
    if(realpath(buffer, fname) != (char *)NULL){
      if(access(fname, mode) == 0){ return(strncpy(filename, fname, len)); }
    }

    while((pstr = strtok((char *)NULL, ":")) != (char *)NULL){
      sprintf(buffer, "%s/%s", pstr, pname);
      /*--------------------------------- Try to resolve an absolute path name */
      if(realpath(buffer, fname) != (char *)NULL){
        if(access(fname, mode) == 0){ return(strncpy(filename, fname, len)); }
      }
    }
  }

  return(strncpy(filename, pname, len));
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgFindPathExec */
/*-----------------------------------------------------------------------------*/
char *CfgFindPathExec(char *filename, size_t len, char *pname){
  /*---------------------------------------------------------------------------*/
  return(CfgFindPath(filename, len, pname,  F_OK|X_OK));
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CfgGetDataSize */
/*-----------------------------------------------------------------------------*/
#ifdef __Lynx__
#include <fcntl.h>
#include <info.h>
#include <proc.h>
#ifdef VMTR2X
#include <sys/rtpc8067.h>
#else
#include <ces/rtpc8067.h>
#endif

size_t CfgGetDataSize(const char *path, int pid, size_t *freemem){
__pentry *procTab;
long     maxProc, offset;
size_t   size;
int      fd;


#ifdef VMTR2X
 return(0);
#else
  offset     = (info(_I_FREEPAGES) & 0x0FFFFFFF) | SEG9_BASE;
  *freemem = *((size_t *)offset) * getpagesize();
  maxProc    = info(_I_NPROCTAB);
  offset     = info(_I_PROCTAB);

  procTab = (__pentry *)malloc(sizeof(__pentry) * maxProc);
  if(procTab == (__pentry *)NULL){ return(0); }

  if((fd = open("/dev/mem", 0)) == -1){
    perror(__FUNCTION__": open(/dev/mem)");
    free(procTab);
    return(0);
  }

  lseek(fd, offset, 0);

  if(read(fd, (char *)procTab, sizeof(__pentry) * maxProc) <= 0){
    perror(__FUNCTION__": read");
    free(procTab);
    close(fd);
    return(0);
  }

  size = procTab[pid].bss.count * getpagesize();

  free(procTab);
  close(fd);

  return(size);
#endif
}
#endif


#ifdef __osf__
#include <mach.h>
#include <mach/mach_types.h>
#include <mach/vm_statistics.h>

size_t CfgGetDataSize(const char *path, int pid, size_t *freemem){
vm_statistics_data_t   vmstats;
task_basic_info_data_t taskinfo;
task_t                 thistask;
unsigned int           taskinfo_l = TASK_BASIC_INFO_COUNT;
size_t                 size = 0;

  if(task_by_unix_pid(task_self(), pid, &thistask) == KERN_SUCCESS){
    int rtn;

    rtn = task_info(thistask, TASK_BASIC_INFO,
                    (task_info_t)&taskinfo, &taskinfo_l);

    if(rtn == KERN_SUCCESS){ size = taskinfo.virtual_size; }
  }

  (void)vm_statistics(task_self(), &vmstats);

  *freemem = vmstats.free_count * getpagesize();

  return(size);
}

#endif /* __osf__ */

#ifdef Linux

size_t CfgGetDataSize( const char *path, int pid, size_t *freemem){
  size_t size[7], total;
  FILE *fd;
  
  if( (fd = fopen( path, "r" )) != (FILE*)NULL ) {
    fscanf( fd, "%zu%zu%zu%zu%zu%zu%zu", size, size+1, size+2, size+3, size+4,
	    size+5, size+6 );
    fclose( fd );
    /*-------------------------------- size[1]: VmRSS; size[5] = VmData+VmStk */ 
    total = ( size[1] + size[5] )*getpagesize();
  } else total = 0;
  return(total);
}

#endif


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CfgRunTimeStrBuilt */
/*-----------------------------------------------------------------------------*/
 char *CfgRunTimeStrBuilt( time_t startTime, char *runTime, size_t len){
  char buf[64];
  time_t dt=difftime(time((time_t *)NULL), startTime);
  int bLen=0, day=dt/CFG_DAY;

  if (day>0) bLen=sprintf(buf, "%d+", day);

  strftime(&buf[bLen], sizeof buf-bLen, (dt>=CFG_HOUR)?"%T":"%M:%S",
	   gmtime(&dt));

  return(strncpy(runTime, buf, len));
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgStrFmtCat */
/*-----------------------------------------------------------------------------*/
char *CfgStrFmtCat( char *str, CfgBool_t uniq, char *format, ... ) {
  va_list args;

  /*---------------------------------------------------------------------------*/
  va_start( args, format );
  str = CfgStrFmtCatv( str, uniq, format, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(str);
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgStrFmtCatv */
/*-----------------------------------------------------------------------------*/
char *CfgStrFmtCatv( char *str, CfgBool_t uniq, char *format, va_list args) {
  int pLen, len = ( str ? strlen(str) :  0 ), fLen, cat;
  char *pStr, buf[CFG_MSG_BUF_SIZE], *p, *pRef ;

  pLen = len; cat = 0;
  /*--------------------------------------------------------- Build the string */
  if( !format || !format[0] ) { /*--------- No format, perform a simple strcat */
    pStr = va_arg(args, char*);
    fLen = strlen(pStr);
    /*-------------------------------------------- Format exist - Use vsprintf */
  } else if( strcmp(format, " %s") && strcmp(format, "%s ") ) {
    buf[0] = '\0'; pStr = buf;
    fLen = vsnprintf( pStr, CFG_MSG_BUF_SIZE-1, format, args );
    if( (fLen < 0) || (fLen >= sizeof buf) ) {
      fprintf( stderr, "CfgStrFmtCat> vsnprintf error\n" );
      return(str);
    } else pStr[fLen] = '\0';
  } else { /*---------------------------- Add a space before the cat operation */
    pStr = va_arg(args, char*);
    fLen = strlen(pStr);
    if( pLen ) cat = 1;
  }
  /*-------------------------------------------- Check if string already exist */
  if( uniq && str && str[0]  ) {
    if( len >= strlen(pStr) ) { p = strstr( str, pStr); pRef = str; }
    else { p = strstr( pStr, str); pRef = pStr;  }
    if( p && ((p[fLen] == '\0') || (p[fLen] == ' ')) &&
	((p == pRef) || (p[-1] == ' ')) ) {
      
      if( pRef == pStr ) { free( str ); return( strdup(pRef) ); }
      else return(pRef);
    }
  }
  /*---------------------------------------------- Allocate the required space */
  len += fLen + cat;
  if( !(str = realloc(str, len + 1)) ){
    fprintf( stderr, "CfgStrFmtCat> realloc failed\n" );
    len = 0;
  } else {
    if( pLen && cat ) { str[pLen] = ' '; pLen++; };
    str[pLen] = '\0';
    strcat( str, pStr );
    str[len] = '\0';
  }
  /*---------------------------------------------------------------------------*/
  return(str);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgStrRemove */
/*-----------------------------------------------------------------------------*/
char *CfgStrRemove( char *str, char *del, char sep ) {
  int dLen, len, size;
  char *pDel;

  if( !str || !str[0] || !del || !del[0] ) return(str);
  /*---------------------------------------------------------------------------*/
  len = strlen(str); dLen = strlen(del);
  for( pDel = str; pDel != NULL; pDel = strchr( pDel, sep ) ) {
    if( pDel != str ) pDel++;
    if( !(pDel = strstr( pDel, del )) ) break;
    
    if( pDel == str) {/*---------------------------------------- First element */
      if( (len == dLen) || (pDel[dLen] == sep) ) break;
    } else if( pDel[dLen] == '\0' ) {  /*------------------------ Last element */
      if( (pDel[dLen] == '\0') && (pDel[-1] == sep) ) break;
    } else { /*------------------------------------------------- Other element */
      if( (pDel[-1] == sep) && (pDel[dLen] == sep) ) break;
    }
  }
  if( !pDel ) return(str);
  
  if( pDel != str ) { pDel--; dLen++;}
  else if( pDel[dLen] != '\0' ) dLen++;
  if( (size = strlen(pDel) - dLen) > 0 ) memmove( pDel, pDel+dLen, size );
  str[len-dLen] = '\0';
  /*---------------------------------------------------------------------------*/
  return(str);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CfgSysName */
/*-----------------------------------------------------------------------------*/
char *CfgSysName( int argc, char *argv[] ) {
  int lastArgc;
  char *sysName=NULL, *lastArgv;

  if( (lastArgc = argc-1) < 1 ) return(NULL);
  /*------------------------------------------------------ Last argument exist */
  lastArgv = argv[lastArgc];
  if( strrchr(lastArgv, '.') || !strchr(lastArgv, '/') ) { 
    sysName = NULL; /*---------------- From file or CmName - Not DBPATH/CmName */
  } else sysName = strdup(lastArgv); 
  /*---------------------------------------------------------------------------*/
  return(sysName);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CfgToLower */
/*-----------------------------------------------------------------------------*/
char *CfgTolower(char *word){
  char *t;

  if(word == (char *)NULL) return(word);
  for( t = word ; *t; t++ ) *t = tolower(*t);
  return(word);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CfgToupper */
/*-----------------------------------------------------------------------------*/
char *CfgToupper(char *word){
  char *t;

  if(word == (char *)NULL) return(word);
  for( t = word ; *t; t++ ) *t = toupper(*t);
  return(word);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CfgWaitWithTimeout */
/*-----------------------------------------------------------------------------*/
int CfgWaitWithTimeout( double seconds ) {
  struct timeval timeout;

  if( seconds <= 0. ) return(CFG_OK);
  /*---------------------------------------------------------------------------*/
  timeout.tv_sec = (int) seconds;
  timeout.tv_usec = (int) ((seconds - (int) seconds) * 1000000);
  

  /*---------------------------------------------------------------------------*/
  return( select( 0, NULL, NULL, NULL, &timeout) );
}



/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------ CfgDataCloseLogStrean */
/*-----------------------------------------------------------------------------*/
int CfgDataCloseLogStream( CfgData_t *p, int closeFlag ) {

  if( !p ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( (!closeFlag && p->logFlag) || !p->logStream ) return(CFG_OK);
  /*-------------------------------------------------------- Close the logFile */
  fprintf( p->logStream, "%s>CfgCloseLogStream> Close\n", 
	   p->strTime );
  fflush(p->logStream);
  fclose(p->logStream);
  p->logStream = (FILE*)NULL;
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgDataConfAddv */
/*-----------------------------------------------------------------------------*/
int CfgDataConfAddv( CfgData_t *p, char *fmt, va_list args ) {
  int length; 

  if( !p ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  p->configBuilt = CfgStrFmtCatv( p->configBuilt, CfgFalse, fmt, args );
  if( !p->configBuilt ) length = 0; 
  else length = strlen(p->configBuilt);
  /*---------------------------------------------------------------------------*/
  return(length);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataGetActivation */
/*-----------------------------------------------------------------------------*/
char *CfgDataGetActivation( CfgData_t *p ) {
  
  if( !CbfCleanFind( "CfgDataFree", p ) ) return(NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->activation );
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------- CfgDataGetCmCleanupTmo */
/*-----------------------------------------------------------------------------*/
double CfgDataGetCmCleanupTmo( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(1.);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->cmCleanupTmo );
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------- CfgDataGetCmDomainName */
/*-----------------------------------------------------------------------------*/
char *CfgDataGetCmDomainName( CfgData_t *p ) {

  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( CmConnectGetDomain() );
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgDataGetCmName */
/*-----------------------------------------------------------------------------*/
char *CfgDataGetCmName( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->cmName );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------ CfgDataGetCommintFlag */
/*-----------------------------------------------------------------------------*/
int CfgDataGetCommitFlag( CfgData_t *p, int *file, int *db ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( file ) file[0] = p->noFileSave;
  if( db ) db[0]     = p->noDbSave; 
  /*---------------------------------------------------------------------------*/
  return( CFG_OK );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------ CfgDataGetCurrentTime */
/*-----------------------------------------------------------------------------*/
time_t CfgDataGetCurrentTime( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return((time_t)NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->currentTime );
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgDataGetConfig */
/*-----------------------------------------------------------------------------*/
char *CfgDataGetConfig( CfgData_t *p ) {

  if( !p || !p->config ) return(NULL);
  /*---------------------------------------------------------------------------*/
  if( p->configFnt ) {
    if( p->configBuilt ) { free(p->configBuilt); p->configBuilt=NULL; }
    /*---------------------------------------- Excute the ConfigBuilt function */
    p->configFnt();
    /*------------------------------------------------- Store the built config */
    if( p->config ) free(p->config); 
    p->config     = p->configBuilt;
    p->configLen  = strlen( p->config ); 
    p->configBuilt = NULL;
    
  }
  /*---------------------------------------------------------------------------*/
  return( p->config );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataGetConfigTime */
/*-----------------------------------------------------------------------------*/
time_t CfgDataGetConfigTime( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return((time_t)NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->configTime );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgDataGetDebug */
/*-----------------------------------------------------------------------------*/
int CfgDataGetDebug( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->debug );
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CfgDataGetHostname */
/*-----------------------------------------------------------------------------*/
char *CfgDataGetHostname( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->hostname );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------ CfgDataGetLogFilename */
/*-----------------------------------------------------------------------------*/
char *CfgDataGetLogFilename( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->logFilename );
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CfgDataGetLogStream */
/*-----------------------------------------------------------------------------*/
FILE *CfgDataGetLogStream( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return((FILE*)NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->logStream );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgDataGetMemSize */
/*-----------------------------------------------------------------------------*/
size_t CfgDataGetMemSize( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return((size_t)NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->memSize );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CfgDataGetMsgR */
/*-----------------------------------------------------------------------------*/
CfgMsgR_t *CfgDataGetMsgR( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return((CfgMsgR_t*)NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->msgR );
}

/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgDataGetPid */
/*-----------------------------------------------------------------------------*/
int CfgDataGetPid( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->pid );
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------- CfgDataGetRequestedState */
/*-----------------------------------------------------------------------------*/
CfgServerState_t CfgDataGetRequestedState( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CfgServerUnknow);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->reqState );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataGetReportFlag */
/*-----------------------------------------------------------------------------*/
int CfgDataGetRFlag( CfgData_t *p, int *log, int *stdOut, int *cmTrace ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( log    )  *log     = p->logFlag;
  if( stdOut )  *stdOut  = p->stdoutFlag;
  if( cmTrace ) *cmTrace = p->cmTrace;
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}



/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgDataGetState */
/*-----------------------------------------------------------------------------*/
CfgServerState_t CfgDataGetState( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CfgServerUnknow);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->state );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgDataGetState */
/*-----------------------------------------------------------------------------*/
char* CfgDataGetStateStr( CfgData_t *p, CfgServerState_t state ) {
  char *stateStr;

  /*---------------------------------------------------------------------------*/
  if( p && p->stateStrFnt ) stateStr = p->stateStrFnt(state);
  else stateStr = CfgServerStateStr[state]; 
  /*---------------------------------------------------------------------------*/
  return( stateStr );
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CfgDataGetStartTime */
/*-----------------------------------------------------------------------------*/
time_t CfgDataGetStartTime( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return((time_t)NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->startTime );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgDataGetStrTime */
/*-----------------------------------------------------------------------------*/
char *CfgDataGetStrTime( CfgData_t *p ) {

  if( !p ) return(NULL);
  /*---------------------------------------------------------------------------*/
  time(&p->currentTime);                     
  strftime( p->strTime, sizeof p->strTime, CFG_TIME_FMT,
	    gmtime(&p->currentTime) );
  /*---------------------------------------------------------------------------*/
  return( p->strTime );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgDataGetSysName */
/*-----------------------------------------------------------------------------*/
char *CfgDataGetSysName( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->sysName );
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgDataGetTag */
/*-----------------------------------------------------------------------------*/
char *CfgDataGetTag( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(NULL);
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------------------------------------------*/
  return( p->tag );
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------- CfgDataGetUserActivation */
/*-----------------------------------------------------------------------------*/
char *CfgDataGetUserActivation( CfgData_t *p ) {
  char *userAct, *tmp;

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(NULL);
  /*---------------------------------------------------------------------------*/
  if( p->userActivation ) {
    free( p->userActivation );
    p->userActivation = NULL;
  }
  if( p->getActivation && (userAct = p->getActivation( p->getActiParam )) ) {
    if( (tmp = strdup(userAct)) ) p->userActivation = tmp;
  }
  /*---------------------------------------------------------------------------*/
  return( p->userActivation );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgDataFree */
/*-----------------------------------------------------------------------------*/
CfgData_t *CfgDataFree( CfgData_t *p ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(p);
  /*---------------------------------------------------------------------------*/
  p->msgR = CfgMsgRFree(p->msgR);
  CbfCleanInstallPrinter((CbfCleanPrinter_t)vprintf);
  p->cfgPrn = CfgPrinterFree( p->cfgPrn );
  CfgDataCloseLogStream( p, CfgTrue);
  if( p->logFilename) { 
    free(p->logFilename); 
    p->logFilename = NULL;
  }
  if( p->server ) { free(p->server); p->server = NULL; }
  if( p->tag ) { free(p->tag); p->tag = NULL; }
  if( p->sysName ) { free(p->sysName); p->sysName = NULL; }
  if( p->cmName ) { free(p->cmName); p->cmName = NULL; }
  if( p->config ) { free(p->config); p->config = NULL; }
  if( p->configBuilt ) { free(p->configBuilt); p->configBuilt = NULL; }
  if( p->str ) { free(p->str); p->str = NULL; }
  if( p->schedAffinity ) { free(p->schedAffinity); p->schedAffinity = NULL; }
  p->uiInfo = CfgMsgFree( p->uiInfo );
  if( p->userActivation ) { 
    free( p->userActivation ); p->userActivation = NULL; 
  }
  pthread_mutex_destroy( &p->uiMutex );
#ifdef Linux
  if(p->procFd ) { free(p->procFd); p->procFd = NULL; }
#endif
  memset((char *)p, 0, sizeof(CfgData_t));  
  free(p);
  CbfCleanDone( "CfgDataFree", p );
  if( p == CfgData ) CfgData = (CfgData_t*)NULL;
  /*---------------------------------------------------------------------------*/
  return((CfgData_t*)NULL);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------- CfgDataLogFilenameInit */
/*-----------------------------------------------------------------------------*/
int CfgDataLogFilenameInit( CfgData_t *p ) {
  char path[MAXPATHLEN];
  char startTimeStr[50];

  if( !p || !p->cmName ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( p->logFilename ) p->logFilename[0] = '\0';
  strftime( startTimeStr, 50, CFG_TIME_FMT, gmtime(&p->startTime));
  getcwd( path, sizeof path );
  p->logFilename = CfgStrFmtCat( p->logFilename, CfgFalse,
                                      "%s/%s_%s.log",
                                      path, p->cmName?p->cmName:"",
                                      startTimeStr );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgDataLogPrintv */
/*-----------------------------------------------------------------------------*/
int CfgDataLogFPrintv( CfgData_t *p, char *fmt, va_list args ) {
  int rtn;
  FILE *fp;
  
  if( !p || pthread_equal(pthread_self(), p->tid) == 0 ) return( CFG_FAIL );
  /*---------------------------------------------------------------------------*/
  fp = p->logStream;
  if( !fp ) fp = stderr;
  fprintf(fp, "%s>", CfgDataGetStrTime(p) );  
  rtn = vfprintf( fp, fmt, args );
  fflush( fp );
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgDataLogPrint */
/*-----------------------------------------------------------------------------*/
int CfgDataLogFPrint( CfgData_t *p, char *fmt, ... ) {
  int rtn;
  va_list args;
  
  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgDataLogFPrintv( p, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}



/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgDataMsgAddFmtv */
/*-----------------------------------------------------------------------------*/
int CfgDataMsgAddFmtv( CfgData_t *p, int errLvl, int debug, char *tag, 
		       char *fmt, va_list args ) {
  char *strFmt;
  CfgMsg_t *msg;
  CfgBool_t noFmt, noTag;
  CfgMsgR_t *msgR;
  CfgMsg_t *uiInfo;
  pthread_t tid;
  
  /*---------------------------------------------------------------------------*/
  if( !p || !(msgR = p->msgR)) {
    CfgMsgPrintv( errLvl, fmt, args );
    if( errLvl == CFG_FATAL)  { 
      CbfTerminate(); 
      exit(EXIT_FAILURE);
    }
    return(CFG_OK);
  }
  if( debug > p->debug ) return(CFG_OK);
  /*---------------------------------------------------------------------------*/
  noFmt = (!fmt || !fmt[0]);
  noTag = (!tag || !tag[0]);
  if( noFmt && noTag ) return(CFG_FAIL);
  /*-------------------------------------------------------- Build the message */
  if( noFmt ) strFmt = tag; else strFmt = fmt;
  if( !(msg = CfgMsgNewArgs( errLvl, strFmt, args )) ) return(CFG_FAIL);
  /*---------------------------------------------------------- Add the message */
  if( errLvl == CFG_USER_INFO ) {
    /*-------------------------------------------------------------------------*/
    pthread_mutex_lock( &p->uiMutex );
    uiInfo = p->uiInfo;
    p->uiInfo = msg;
    pthread_mutex_unlock( &p->uiMutex );    
    if( uiInfo ) uiInfo = CfgMsgFree( uiInfo );
  } else {
    /*-------------------- Add the message into the message reporter CfgMsgR_t */
    if( strFmt == tag ) tag = CfgMsgGetTxt(msg);   /*-- The message is the Tag */
    CfgMsgRAddMsgTag( msgR, msg, tag );
    pthread_mutex_lock( &p->uiMutex );
    if( debug < p->msgDbgMin ) p->msgDbgMin = debug;
    if( p->msgDbgMin == CFGSEND || errLvl == CFG_FATAL) p->flush = 1;
    pthread_mutex_unlock( &p->uiMutex );
    /*-------------------------------------------------------------------------*/
    if( errLvl == CFG_FATAL ) {
      CbfTerminate();
      tid = pthread_self();
      if( !pthread_equal( tid, p->tid ) ) pthread_exit(NULL) /* pthread_cancel(tid)*/;
    }
  }
  if( p->flush ) CfgDataMsgRSend( p, 0. );
  /*---------------------------------------------------------------------------*/
  return( CFG_OK );
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgDataMsgAddFmt */
/*-----------------------------------------------------------------------------*/
int CfgDataMsgAddFmt( CfgData_t *p, int errLvl, int debug, char *tag,
		      char *fmt, ... ) {
  va_list args;
  
  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  CfgDataMsgAddFmtv( p, errLvl, debug, tag, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------- CfgDataMsgLogReporCloset */
/*-----------------------------------------------------------------------------*/
int CfgDataMsgLogReportClose( CfgData_t *p ) {
  
  /*---------------------------------------------------------------------------*/
  return(CfgDataCloseLogStream(p, CfgFalse));
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------- CfgDataMsgLogReportOpen */
/*-----------------------------------------------------------------------------*/
int CfgDataMsgLogReportOpen( CfgData_t *p ) {
  const char *name;
  
  if( !p || !(name = p->logFilename) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( !p->logFlag || (!access(name, F_OK) && p->logStream) )
    return(CFG_OK);
  CfgDataMsgAddFmt( p, CFG_INFO, 0, NULL, 
		    "CfgMsgLogReportOpen> try to open log stream");
  /*------------------------------------------------ Initialize logFile report */
  CfgDataCloseLogStream( p, CfgTrue );
  if( CfgDataOpenLogStream( p ) == CFG_FAIL ) {
    CfgDataMsgAddFmt( p, CFG_WARNING, 0, NULL, "Unable to open logFile %s", 
		      name );
    return(CFG_OK);
  }
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CfgDataMsgSendInit */
/*-----------------------------------------------------------------------------*/
int CfgDataMsgSendInit( CfgData_t *p, const CfgMsg_t *msg ) {
  char *fmt;
  int cnt;

  if( !p ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( p->str ) { p->str[0] = '\0';  }
  if( (cnt = CfgMsgGetCount(msg)) ) fmt = "%s@%s>%s%s (%d times)\n";
  else fmt = "%s@%s>%s%s\n";
  cnt += 1;
  strftime( p->strTime, sizeof p->strTime, CFG_TIME_FMT,
	   CfgMsgGetTimeStampTm(msg) );

  if( (p->msgLength = snprintf( p->str, p->strSize, fmt,
				p->cmName, p->strTime,
				CfgMsgGetLevelStr(msg), CfgMsgGetTxt(msg),
				 cnt )) == -1 ) {
    fprintf( (p->logStream ? p->logStream : stderr),
	     "CfgDataMsgSendInit> snprintf failed ");
    return(CFG_FAIL);
  }
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgDataMsgSendLog */
/*-----------------------------------------------------------------------------*/
int CfgDataMsgSendLog( CfgData_t *p, const CfgMsg_t *msg ) {
  int length;
   
  if( !p ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( !p->logStream || p->msgLength <= 0 || 
      CfgMsgGetLevel(msg) == CFG_USER_INFO ) return(CFG_OK);
  /*---------------------------------------------------------------------------*/
  length = p->msgLength;
  p->str[length-1] = '\n';
  length -= p->cmNameSize;
  if( write( fileno(p->logStream), p->str + p->cmNameSize, length) < 0 ) {
    CfgDataCloseLogStream( p, CfgTrue );
  } else fflush(p->logStream);
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataMsgSendStdout */
/*-----------------------------------------------------------------------------*/
int CfgDataMsgSendStdout( CfgData_t *p, const CfgMsg_t *msg ) {
  int level, length, cLength, diff;
  static int prevLength = 0, prevLevel = -1;

  if( !p ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( !p->stdoutFlag || p->msgLength <= 0 ) return(CFG_OK);
  /*---------------------------------------------------------------------------*/
  length = p->msgLength;
  level  = CfgMsgGetLevel(msg);
  if( prevLevel == CFG_USER_INFO && (diff = prevLength - length ) > 0 ) {
    memset( p->str + length - 1, ' ', diff );
    cLength = prevLength;
    p->str[cLength - 1] = '\n';
  } else cLength = length;

  /*---------------------------------------------------------------------------*/
  if( level == CFG_USER_INFO ) p->str[cLength-1]= '\r';
  if( write( fileno(stdout), p->str, cLength) < 0 ) {
    CfgDataSetRFlag( p, -1, CfgFalse, -1 );
  } else fflush(stdout);

  /*---------------------------------------------------------------------------*/
  prevLength = length;
  prevLevel  = level;
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgDataRMsgSend */
/*-----------------------------------------------------------------------------*/
int CfgDataMsgRSend( CfgData_t *p, double tmo ) {
  size_t size, freeMem;
  int msgLvl,  flag;
  CfgMsg_t *uiInfo;
  double diff, lTmo;
  pthread_t tid;
  int  rtn, inc; 
  CfgMsg_t *msg;
  CmConnectCondition condition;
  const char *tmp;

  tid = pthread_self();
  if( !p || !pthread_equal(tid, p->tid) ) { 
    /*---------------------------------------------------- Not the Main thread */
    /*-------------------------------------------------------------------------*/
    return(CFG_OK);
  }
  
  /*------------------------ Main thread --------------------------------------*/
  /*---------------------------------------------------- Update CfgData fields */
  rtn = CFG_OK;
  flag = 0;
  CfgDataGetStrTime( p );
  /*------------------------------------- Check the time since the last report */
  diff = difftime( p->currentTime, p->lastTime );
  if( diff >= p->periodTime )  { 
    flag = -1;
    /*------------------------------------------------------- Used Memory size */
    size = CfgGetDataSize( p->procFd, p->pid, &freeMem );
    if( size > p->memSize ) {
      inc = size - p->memSize;
      p->memSize = size;
      msg = CfgMsgNewFmt( CFG_INFO, 
			  "Memory Used increase, cur. %.2f(KB), inc. %.2f(KB)",
			  size/1024., inc/1024. );
      if( msg ) CfgMsgRAddMsg( p->msgR, msg );
    }
  }
  /*---------------------------------------------------------------------------*/
  lTmo  = fabs(tmo);

  /*---------------------------------------------------------------------------*/
  pthread_mutex_lock( &p->uiMutex );
  p->msgDbgMin = CFGDBGMAX;
  if( p->flush ) {
    flag = 1;
    p->flush = 0;
  }
  pthread_mutex_unlock( &p->uiMutex );

  /*---------------------------------------------------------------------------*/
  if( p->debug == CFGSEND ) flag = 2; 

  if( flag || tmo < 0.)  { 
    if( flag >= 1 && lTmo == 0. ) lTmo = 0.005;
   /*------------------------------------------------ Extract the User's info */
    pthread_mutex_lock( &p->uiMutex );
    uiInfo = p->uiInfo;
    p->uiInfo = (CfgMsg_t*)NULL;
    pthread_mutex_unlock( &p->uiMutex );
    /*------------------------------------------------------- Message reporter */
    msgLvl = CfgDataSetPrnMsgLvl( p, CFG_CLINFO );    
    /*------ Build the message for UIClient, Sent the message to LogFile, .... */
    rtn = CfgMsgRSend(p->msgR, uiInfo );
    /*-------------------------------------------------------------------------*/
    CfgDataSetPrnMsgLvl( p, msgLvl );
    /*------------------------------------------------ Clear the UIInfo if any */
    uiInfo = CfgMsgFree( uiInfo );
    time( &p->lastTime );   /*-------------------- Update the last report time */
  }
  if( rtn < 0 ) lTmo = 0.1;  /*--------------------------------- Case of FATAL */
  /*---------------------------------------------------------------------------*/
  if( !CmConnectGetDomain() ) CfgWaitWithTimeout( tmo );
  else {
    condition = -1;
    if( lTmo == 0. ) condition = CmMessageCheck(); 
    else condition = CmMessageWaitWithTimeout(lTmo);
    if( p->debug  > 2 ) {
      tmp = "INFO...-";
      CfgDataLogFPrint( p, "%sCfgDataMsgRSend>flag %d - lTmo %g(%g) "
			"- condition %d, nbMgs %d n",
			tmp, flag, lTmo, tmo, condition, rtn );
    }
  }
  /*---------------------------------------------------------------------------*/
  if( rtn < 0 ) exit( EXIT_FAILURE );
  /*---------------------------------------------------------------------------*/
  return(rtn);
}
   

/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgDataInit */
/*-----------------------------------------------------------------------------*/
CfgData_t *CfgDataNew( int argc, char *argv[], int msgNb ) {
  CfgData_t *p;
  char host[256], tag[MAXPATHLEN];
  struct hostent *hostent;
  size_t freeMem;

  /*---------------------------------------------------------------------------*/
  if( !(p = (CfgData_t*) malloc(sizeof(CfgData_t))) ) return(p);
  memset((char*)p, 0, sizeof(CfgData_t));
  /*---------------------------------------------------------------------------*/
  p->cfgPrn = CfgPrinterNew( CFG_MSG_BUF_SIZE );
  /*------------------------------------------------- install cleanup function */
  CbfCleanAddRoot( "CfgDataFree", (CbfCleanFnt_t)CfgDataFree, p);
  /*---------------------------------------------------------- Signal Handling */
/*   for( sig=SIGHUP; sig<=SIGUSR2; sig++ ) { */
/*     if( (sigHdlr = signal(sig, SIG_DFL)) == SIG_IGN ) { */
/*       psignal(sig, "Handler SIG_IGN to SIG_DFL"); */
/*     } */
/*   } */

  /*-------------------------------------------- message report initialization */
  if ( msgNb > 0 )   p->msgR = CfgMsgRNew( msgNb );
  else p->msgR = (CfgMsgR_t*)NULL;
  p->msgDbgMin = CFGDBGMAX;


  time(&p->startTime);
  p->argc=argc;
  p->argv=argv;
  p->pid=getpid();
  p->tid=pthread_self();
  p->prio=getpriority(PRIO_PROCESS, 0);
  gethostname(host, sizeof host );
  hostent = gethostbyname( host );
  strncpy( p->hostname, hostent->h_name, sizeof p->hostname );
/*   printf("CfgDataNew> hostent %s - host %s\n", p->hostname, host ); */

  /*---------------------------------------------------------- activation path */
  CfgFindPathExec(p->activation, MAXPATHLEN-1, (argc ? argv[0] : NULL) );
  p->server=( p->activation ? 
	      strdup((const char *)CfgBasename(p->activation)) : NULL );
  /*----------------------------------------------------------------- tag path */
  /* p->tag=((argc > 1) ? strdup((const char *)argv[1]) : NULL ); */
  p->tag = NULL;
  if( argc > 1 ) { 
    CfgFindPath(tag, MAXPATHLEN-1, argv[1], F_OK );
    p->tag = strdup( tag );
  }
  p->config=NULL;
  p->configLen=0;
  p->cmName=CfgCmName(argc, argv);
  if (p->cmName) p->cmNameSize = strlen(p->cmName)+1;
  if( (p->sysName = CfgSysName(argc, argv)) ) {
    p->noDbSave   = 0;        /*----------------------- Set to commit to Db ON */
    p->noFileSave = 0;        /*--------------------- Set to commit to file ON */
  } else {
    p->noDbSave   = 1;        /*---------------------- Set to commit to Db OFF */
    p->noFileSave = 1;        /*-------------------- Set to commit to file OFF */
  }
  p->stdoutFlag = CfgFalse; /*------------------------------------ Stdout flag */
  p->logFlag    = CfgFalse; /*----------------------------------- Logfile flag */
  p->cmTrace    = 0;        /*--------------------------------- Cm trace level */

  /*-------------------------------------------- Fill the CfgData strTime field */
  strftime( p->strTime, sizeof p->strTime, CFG_TIME_FMT,
	    gmtime(&p->startTime) );

  p->strSize = CFG_MSG_BUF_SIZE-1+p->cmNameSize+20;
  p->str = (char*)malloc(p->strSize);
  /*-------------------- Build the logfilename = CWD + <cmName>_<startingTime> */
  CfgDataLogFilenameInit( p );

  /*----------------- Initialize the period field for a no CFG_UI_INFO message */
  p->lastTime      = p->startTime;
  p->periodTime    = CFG_PERIOD;

  if( 
#ifdef RIO806X
  pthread_mutex_init(&p->uiMutex, pthread_mutexattr_default)
#else
  pthread_mutex_init(&p->uiMutex, NULL) 
#endif
  == -1 ) {
    return(CfgDataFree(p));
  }

#ifdef Linux
  p->procFd = CfgStrFmtCat( NULL, CfgFalse, "/proc/%d/statm",
				 p->pid );
#endif
  /*--------------------------------------------------------------------------*/
  p->memSize      = CfgGetDataSize( p->procFd, p->pid, &freeMem );
  p->prnMsgLvl    = CFG_INFO;
  p->cmCleanupTmo = 2.;
  /*--------------------------------------------------------------------------*/
  return(p);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataOpenLogStream */
/*-----------------------------------------------------------------------------*/
int CfgDataOpenLogStream( CfgData_t *p ) {
  char *sysName, *strTime;

  if( !p || !p->logFilename ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  p->logStream = fopen( p->logFilename,  "a+" );
  if( !p->logStream )  return(CFG_FAIL);
  /*---------------------------------------------------------- Starting string */
  sysName = p->sysName;
  strTime = CfgDataGetStrTime(p);
  fprintf( p->logStream, "Open at %s - Host %s\n# Server %s\n"
	   "# configTag %s - sysName %s - pid %d\n",
           strTime, p->hostname, p->activation, p->tag,
	   (sysName ? sysName : "Undefined"), p->pid);
  if( !p->config || !p->config[0] ) return(CFG_OK);
  /*------------------------------------------------------ print config buffer */
  fprintf( p->logStream, "%s> Current config :\n%s\n", strTime, p->config );
  fflush( p->logStream );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgDataReachState */
/*-----------------------------------------------------------------------------*/
int CfgDataReachState( CfgData_t *p, CfgServerState_t state ) {
  CfgServerState_t cState;
  int rtn;
  CfgStateReachFnt_t *fnt, f;
  void **arg, *q;

  if( !CbfCleanFind("CfgDataFree", p) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  p->reqState = state;
  cState = p->state;
  fnt    = p->stateReachFnt;
  arg    = p->stateReachArg;
  rtn  = CFG_OK;
  if( cState == CfgServerError || /*--------------------------- CfgServerError */
      state == CfgServerError ) {
    p->state = state;
    rtn = CFG_OK;
  } else if( state == CfgServerEnd ) { /*------------------------ CfgServerEnd */
    f = fnt[CfgServerEnd - 1];
    q = arg[CfgServerEnd - 1];
    if( f ) rtn = f( q ); else rtn = CFG_OK;
    p->state = CfgServerIdle;
    exit(EXIT_SUCCESS);
  } else if( state > cState &&  
	     state >= CfgServerIdle && state <= CfgServerGolden ) {
    /*-------------- CfgServerConfigure -> CfgServerActive -> CfgServerGolden */
    rtn = CFG_OK;
    for( ; cState < state; cState = p->state ) {
      f = fnt[cState];
      q = arg[cState];
      if( f ) rtn = f( q ); else rtn = CFG_OK; 
      /*-----------------------------------------------------------------------*/
      if( rtn == CFG_OK )  p->state++;
      else {
	p->state = CfgServerError;
	break;
      }
    }
  } else if( state <= cState && state >= CfgServerActive  ) {
    /*------------------------------------ CfgServerGolden -> CfgServerActive */
    p->state = state;
    rtn = CFG_OK;
    
  } else {
    CfgDataMsgAddFmt( p,  CFG_WARNING, 0, NULL,  "CfgDataReachState> "
		      "current state %s - ask to reach %s state - "
		      "operation not allowed", CfgServerStateStr[cState],
		      CfgServerStateStr[state]);
    rtn = CFG_FAIL;
  }
  /*---------------------------------------------------------------------------*/
  return( rtn );
}



/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------- CfgDataSetCmCleanupTmo */
/*-----------------------------------------------------------------------------*/
int CfgDataSetCmCleanupTmo( CfgData_t *p, double tmo ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  CfgDataMsgAddFmt( p, CFG_INFO, 0, NULL, "CfgDataSetCmCleanupTmo> "
		    "from %g to %g", p->cmCleanupTmo, tmo );
  p->cmCleanupTmo = tmo;
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------- CfgDataSetCmDomainName */
/*-----------------------------------------------------------------------------*/
int CfgDataSetCmDomainName( CfgData_t *p, char *cmDomain ) {

  CmConnectSetDomain( cmDomain );
  /*---------------------------------------------------------------------------*/
  CfgDataMsgAddFmt( p, CFG_INFO, 0, NULL, "CfgDataSetCmDomainName> "
		    "to %s", (cmDomain ? cmDomain : "NULL") );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgDataSetConfig */
/*-----------------------------------------------------------------------------*/
int CfgDataSetConfig( CfgData_t *p, char *buffer ) {
  
  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( p->config ) { free( p->config ); p->config = NULL; }
  p->config = buffer;
  if( buffer && buffer[0] ) p->configLen = strlen( buffer );
  time(&p->configTime); 
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CfgDataSetConfigFnt */
/*-----------------------------------------------------------------------------*/
CfgConfigFnt_t CfgDataSetConfigFnt( CfgData_t *p, CfgConfigFnt_t configFnt ) {
  CfgConfigFnt_t f=(CfgConfigFnt_t)NULL;
  
  if( !CbfCleanFind( "CfgDataFree", p ) ) return(f);
  /*---------------------------------------------------------------------------*/
  f = p->configFnt;
  p->configFnt = configFnt;
  /*---------------------------------------------------------------------------*/
  return(f);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgDataSetDebug */
/*-----------------------------------------------------------------------------*/
int CfgDataSetDebug( CfgData_t *p, int debug ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( debug > -1 )  {
    p->debug = debug;
    CfgDataMsgAddFmt( p, CFG_INFO, 0, NULL, 
		      "CfgDataSetDebug> to %d", p->debug );
  }
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CfgDataSetNoDbSave */
/*-----------------------------------------------------------------------------*/
int CfgDataSetNoDbSave( CfgData_t *p, int val ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  p->noDbSave = val;
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataSetNoFileSave */
/*-----------------------------------------------------------------------------*/
int CfgDataSetNoFileSave( CfgData_t *p, int val ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  p->noFileSave = val;
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataSetPeriodTime */
/*-----------------------------------------------------------------------------*/
int CfgDataSetPeriodTime( CfgData_t *p, double period ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  p->periodTime = (period > 0 ? period : CFG_PERIOD );
  CfgDataMsgAddFmt( p, CFG_INFO, 0, NULL, "CfgDataSetPeriodTime> "
		    " to %gs", p->periodTime );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CfgDataSetPrio */
/*-----------------------------------------------------------------------------*/
int CfgDataSetPrio( CfgData_t *p, int prio ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  setpriority( PRIO_PROCESS, 0, prio );
  p->prio = getpriority(PRIO_PROCESS, 0);
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgDataSetRFlag */
/*-----------------------------------------------------------------------------*/
int CfgDataSetRFlag( CfgData_t *p, int log, int stdOut, int cmTrace ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( !p->msgR ) return(CFG_OK);
  if( log > -1 )    p->logFlag    = ( log    ? CfgTrue : CfgFalse );
  if( stdOut > -1 ) p->stdoutFlag = (stdOut ? CfgTrue : CfgFalse );
  if( cmTrace > -1 ) p->cmTrace = cmTrace;
  CfgDataMsgAddFmt( p, CFG_INFO, 0, NULL, "CfgDataSetRFlag> logFlag %d -"
		    " stdoutFlag %d - cmTrace %d", p->logFlag, p->stdoutFlag,
		    p->cmTrace );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------- CfgDataSetSchedAffinity */
/*-----------------------------------------------------------------------------*/
/*--------------------- /sys/devices/system/cpu/online to have the list of cpu */
int CfgDataSetSchedAffinity( CfgData_t *p, char *schedAffinity ) {
  cpu_set_t cpuset;
  int i, j , k, rtn;
  char *str, *c, *n, *o;

  if( !CbfCleanFind( "CfgDataFree", p )  || !schedAffinity ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( p->schedAffinity ) { free(p->schedAffinity); p->schedAffinity = NULL; }
  p->schedAffinity = strdup( schedAffinity );
  /*---------------------------------------------------------------------------*/
  /*---------------------------------------- CPU list as 0,2,4,5-9,11,13,16-20 */
  CPU_ZERO( &cpuset ); 
  for( str = NULL, n = c = schedAffinity; n != NULL;  ) {
    n = strchr( c,  ',' ); 
    if( n ) n[0] = '\0';
    /*------------------------------------------------------------------------*/
    i = atoi(c);
    if( (o = strchr( c, '-')) ) j = atoi(o+1);
    else j = i;
    /*------------------------------------------------------------------------*/
    if( n ) { n[0] = ',';  c = n+1; }
    /*------------------------------------------------------------------------*/
    for( k = i; k <= j; k++ ) { 
      CPU_SET( k, &cpuset );
      str = CfgStrFmtCat( str, CfgFalse, "%d;", k );
    }
  }
  CfgDataMsgAddFmt( p, CFG_INFO, 1, NULL,
		    "CfgDatSetSchedAffinity> \"%s\" - %s",
		    p->schedAffinity, str);
  if( str ) free(str);
  /*---------------------------------------------------------------------------*/
  if( sched_setaffinity( p->pid, sizeof(cpu_set_t), &cpuset ) ) {
    rtn = CFG_FAIL;
    CfgDataMsgAddFmt( p, CFG_WARNING, 0, NULL,
		      "CfgDatSetSchedAffinity> \"%s\" - Fail",
		      p->schedAffinity );
  } else {
    rtn = CFG_OK;
    CfgDataMsgAddFmt( p, CFG_INFO, 0, NULL,
		      "CfgDatSetSchedAffinity> \"%s\" - Ok",
		      p->schedAffinity );
  }
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CfgDataSetStateFnt */
/*-----------------------------------------------------------------------------*/
int CfgDataSetStateFnt( CfgData_t *p, int cState, int rState,
			CfgStateReachFnt_t f, void *q ) {
  CfgStateReachFnt_t *fnt;
  void **arg;

  if( !CbfCleanFind( "CfgDataFree", p ) || !f ||
      rState <= cState ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  fnt = p->stateReachFnt;
  arg = p->stateReachArg;
  fnt[rState-1] = f;
  arg[rState-1] = q;						
  CfgDataMsgAddFmt( p, CFG_INFO, 0, NULL, 
		    "CfgDataSetStateFnt> Transition  %s to %s installed",
		    CfgServerStateStr[cState], CfgServerStateStr[rState] );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------ CfgDataSetStateStrFnt */
/*-----------------------------------------------------------------------------*/
void CfgDataSetStateStrFnt( CfgData_t *p, CfgStateStrFnt_t stateFnt ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return;
  /*---------------------------------------------------------------------------*/
  if( stateFnt ) p->stateStrFnt = stateFnt;
  /*---------------------------------------------------------------------------*/
  return;
}



/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------ CfgDataSetStateStrFnt */
/*-----------------------------------------------------------------------------*/
int CfgDataSetUserActivationFnt( CfgData_t *p, 
				 CfgUserActivationGetFnt_t userActiGet,
				 void *param ) {

  if( !CbfCleanFind( "CfgDataFree", p ) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  p->getActivation = userActiGet;
  p->getActiParam  = param;
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}



/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CfgDataMsgPrinterv */
/*-----------------------------------------------------------------------------*/
int CfgDataMsgPrinterv( CfgData_t *cfg, char *format, va_list args ) {
  int size, rtn;
  CfgPrinter_t *p;


  if( !cfg ) return(CFG_FAIL);  
  /*---------------------------------------------------------------------------*/
  if( !(p = cfg->cfgPrn)  ) return( vprintf( format, args) );
  /*---------------------------------------------------------------------------*/
  size = vsnprintf( p->pBuf, p->pSize, format, args );
  p->pBuf  += size;
  p->pSize -= size;
  if( p->pSize > 0 && p->pBuf[-1] != '\n' ) return(size);
  /*---------------------------------------------------------- End of the line */
  if( p->pSize <= 0 ) {
    p->pBuf = p->buf + p->size;
    p->pBuf[0] = '\0';
  } else p->pBuf[-1] = '\0';
  rtn = CfgMsgRAddMsg( cfg->msgR, CfgMsgNew( cfg->prnMsgLvl, p->buf ) );
  /*----------------------------------------------- Reset the global varaibles */
  p->buf[0] = '\0';
  p->pBuf   = p->buf;
  p->pSize  = p->size;
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


