/* SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Cfg - A library for Virgo process implementation
 *
 * Copyright © 2001-2021 Laboratoire de physique des particules d'Annecy - CNRS
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Fatih Bellachia <fatih.bellachia@lapp.in2p3.fr>
 *   Laurent Fournier <laurent.fournier@lapp.in2p3.fr>
 *   Alain Masserot <alain.masserot@lapp.in2p3.fr>
 */

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <sys/types.h>
#include <CfgList.h>

#ifdef USE_MW
#include <memwatch.h>
#endif /* USE_MW */

void CfgNameListPrint(CfgNameList_t *names) {
  fprintf(stderr, "CfgNameListPrint:");
  if (names) {
    int nNum;
    for (nNum = 0 ; nNum < names->size ; nNum++)
      fprintf(stderr, "\n\t%s 0x%p %ld %g",
        CfgNameListName(names, nNum), CfgNameListVData(names, nNum),
        CfgNameListIData(names, nNum), CfgNameListDData(names, nNum));

    fprintf(stderr, "\n");
  }
  else
    fprintf(stderr, " NULL name list\n");
} /* CfgNameListPrint */

/**************************************************************/
/* name list functions */
CfgNameList_t *CfgNameListNew() {
  CfgNameList_t *names = (CfgNameList_t *)malloc(sizeof(CfgNameList_t));
  CfgNameListInit(names);
  return(names);
} /* CfgNameListNew */

/**************************************************************/
CfgNameList_t *CfgNameListInit(CfgNameList_t *names) {
  if (names)
    memset((char *)names, 0, sizeof(CfgNameList_t));
  return(names);
} /* CfgNameListInit */

/**************************************************************/
CfgNameList_t *CfgNameListDone(CfgNameList_t *names) {
  if (names) {
    while (names->size)
      CfgNameListNRemove(names, names->size - 1);
    memset((char *)names, 0, sizeof(CfgNameList_t));
  }
  return(names);
} /* CfgNameListDone */

/**************************************************************/
CfgNameList_t *CfgNameListFree(CfgNameList_t *names) {
  CfgNameListDone(names);
  if(names) { free(names); names = NULL; }
  return(names);
} /* CfgNameListFree */

/**************************************************************/
/* add a name to list - no duplicates allowed */
int CfgNameListSetFreeFnc(CfgNameList_t *names, CfgNameFreeFnc_t freeFnc) {
  if (names) {
    names->freeFnc = freeFnc;
    return(CFG_OK);
  }
  return(CFG_FAIL);
} /* CfgNameListSetFreeFnc */

/**************************************************************/
int CfgNameListSetSortOption(CfgNameList_t *names, int sort) {
  if (names) {
    names->sort = sort;
    return(CFG_OK);
  }
  return(CFG_FAIL);
} /* CfgNameListSetSortOption */

/**************************************************************/
/* add a name to list - no duplicates allowed */
int CfgNameListAdd(CfgNameList_t *names, char *name) {
  int nNum = CFG_FAIL;

  if ((names) && (name)) {
    nNum = CfgNameListSearchName(names, name);
    if (nNum != CFG_FAIL) {
      names->list[nNum].overwrites++;
      return(nNum);
    }
    for (nNum = 0 ; nNum < names->size ; nNum++)
      if (!(names->list[nNum].name)) {
        names->list[nNum].name = strdup(name);
        return(nNum);
    }
    names->size++;
    names->list = realloc(names->list, names->size * sizeof(CfgName_t));
    memset((char *)&names->list[names->size - 1], 0, sizeof(CfgName_t));
    nNum = CfgNameListSetName(names, names->size - 1, name);
  }

  return(nNum);
} /* CfgNameListAdd */

/**************************************************************/
/* add a name and integer data to list */
int CfgNameListAddInt(CfgNameList_t *names, char *name, int iData) {
  int nNum = CfgNameListAdd(names, name);
  return(CfgNameListSetInt(names, nNum, iData));
} /* CfgNameListAddInt */

/**************************************************************/
/* add a name and integer data to list */
int CfgNameListAddDouble(CfgNameList_t *names, char *name, double dData) {
  int nNum = CfgNameListAdd(names, name);
  return(CfgNameListSetDouble(names, nNum, dData));
} /* CfgNameListAddDouble */

/**************************************************************/
int CfgNameListSetName(CfgNameList_t *names, int nNum, char *name) {
  if ((names) && (nNum >= 0) && (nNum < names->size) && (name)) {
    CfgName_t *cfgName = &names->list[nNum];
    if(cfgName->name){ free(cfgName->name); cfgName->name = NULL; }
    cfgName->name = strdup(name);
    
    if (names->sort == CFG_NAME_ASORT_NAME || names->sort == CFG_NAME_DSORT_NAME) {
      CfgNameListSort(names, names->sort);
      return(CfgNameListSearchName(names, name));
    }
    else
      return(nNum);
  }
  return(CFG_FAIL);
} /* CfgNameListSetName */

/**************************************************************/
int CfgNameListSetData(CfgNameList_t *names, int nNum, void *vData) {

  if ((names) && (nNum >= 0) && (nNum < names->size)) {
    CfgName_t *cfgName = &names->list[nNum];
    cfgName->vData = vData;
    return(nNum);
  }

  return(CFG_FAIL);
} /* CfgNameListSetdata */

/**************************************************************/
int CfgNameListSetInt(CfgNameList_t *names, int nNum, int iData) {
  if ((names) && (nNum >= 0) && (nNum < names->size)) {
    CfgName_t *cfgName = &names->list[nNum];
    cfgName->iData = iData;
    if (names->sort == CFG_NAME_ASORT_IDATA || names->sort == CFG_NAME_DSORT_IDATA) {
      CfgNameListSort(names, names->sort);
      nNum = CfgNameListSearchInt(names, iData);
    }
    return(nNum);
  }
  return(CFG_FAIL);
} /* CfgNameListSetInt */

/**************************************************************/
int CfgNameListSetDouble(CfgNameList_t *names, int nNum, double dData) {
  if ((names) && (nNum >= 0) && (nNum < names->size)) {
    CfgName_t *cfgName = &names->list[nNum];
    cfgName->dData = dData;
    if (names->sort == CFG_NAME_ASORT_DDATA || names->sort == CFG_NAME_DSORT_DDATA) {
      CfgNameListSort(names, names->sort);
      nNum = CfgNameListSearchDouble(names, dData);
    }
    return(nNum);
  }
  return(CFG_FAIL);
} /* CfgNameListSetDouble */

/**************************************************************/
/* remove item from list and fill empty space if needed */
int CfgNameListNRemove(CfgNameList_t *names, int nNum) {
  int cNum;

  if ((names) && (nNum >= 0) && (nNum < names->size)) {
    CfgName_t *cfgName = &names->list[nNum];
    
/*     fprintf(stderr, "CfgNameListNRemove: %p %d [%s], data = 0x%x %p\n",  */
/* 	    names, nNum, cfgName->name, cfgName->vData, names->freeFnc );  */
    if ((cfgName->vData) && (names->freeFnc)) {
      cfgName->vData = names->freeFnc(cfgName->vData);
    }
    if(cfgName->name){ free(cfgName->name); cfgName->name = NULL; }

    /* shift records and reallocate space */
    names->size--;
    for (cNum = nNum; cNum < names->size; cNum++)
      names->list[cNum] = names->list[cNum + 1];
    names->list = realloc(names->list, names->size * sizeof(CfgName_t));
    CfgNameListSort(names, names->sort);

/*    CfgNameListPrint(names); */
    return(nNum);
  }

  return(CFG_FAIL);
} /* CfgNameListNRemove */

/**************************************************************/
/* remove name from list and fill empty space if needed */
int CfgNameListRemoveName(CfgNameList_t *names, char *name) {
  return(CfgNameListNRemove(names, CfgNameListSearchName(names, name)));
} /* CfgNameListRemoveName */

/**************************************************************/
/* remove void data from list and fill empty space if needed */
int CfgNameListRemoveData(CfgNameList_t *names, void *vData) {
  return(CfgNameListNRemove(names, CfgNameListSearchData(names, vData)));
} /* CfgNameListRemoveData */

/**************************************************************/
/* remove integer from list and fill empty space if needed */
int CfgNameListRemoveInt(CfgNameList_t *names, int iData) {
  return(CfgNameListNRemove(names, CfgNameListSearchInt(names, iData)));
} /* CfgNameListRemoveInt */

/**************************************************************/
/* remove double from list and fill empty space if needed */
int CfgNameListRemoveDouble(CfgNameList_t *names, double dData) {
  return(CfgNameListNRemove(names, CfgNameListSearchDouble(names, dData)));
} /* CfgNameListRemoveDouble */

/**************************************************************/
/* comparison on name for ascending order */
static int CfgNameACompare(CfgName_t *item1, CfgName_t *item2) {
  return(strcmp(item1->name, item2->name));
} /* CfgNameACompare */

/* comparison on name for descending order */
static int CfgNameDCompare(CfgName_t *item1, CfgName_t *item2) {
  return(strcmp(item2->name, item1->name));
} /* CfgNameDCompare */

/* comparison on iData for ascending order */
static int CfgNameIDataACompare(CfgName_t *item1, CfgName_t *item2) {
  return(item1->iData < item2->iData ? -1 : item1->iData > item2->iData ? 1 : 0);
} /* CfgNameIDataACompare */

/* comparison on iData for descending order */
static int CfgNameIDataDCompare(CfgName_t *item1, CfgName_t *item2) {
  return(item2->iData < item1->iData ? -1 : item2->iData > item1->iData ? 1 : 0);
} /* CfgNameIDataDCompare */

/* comparison on iData for ascending order */
static int CfgNameDDataACompare(CfgName_t *item1, CfgName_t *item2) {
  return(item1->dData < item2->dData ? -1 : item1->dData > item2->dData ? 1 : 0);
} /* CfgNameDDataACompare */

/* comparison on iData for descending order */
static int CfgNameDDataDCompare(CfgName_t *item1, CfgName_t *item2) {
  return(item2->dData < item1->dData ? -1 : item2->dData > item1->dData ? 1 : 0);
} /* CfgNameDDataDCompare */

/**************************************************************/
int CfgNameListSort(CfgNameList_t *names, int sort) {
  int itemNumber = names->size;
  int itemWidth = sizeof(CfgName_t);
  void *items = (void *)names->list;

  if ((items) && (itemNumber))
    switch (sort) {
      case CFG_NAME_NO_SORT:
        break;
      case CFG_NAME_ASORT_NAME:
        qsort(items, itemNumber, itemWidth,
          (int (*)(const void *, const void *)) CfgNameACompare);
        break;
      case CFG_NAME_DSORT_NAME:
        qsort(items, itemNumber, itemWidth,
          (int (*)(const void *, const void *)) CfgNameDCompare);
        break;
      case CFG_NAME_ASORT_IDATA:
        qsort(items, itemNumber, itemWidth,
          (int (*)(const void *, const void *)) CfgNameIDataACompare);
        break;
      case CFG_NAME_DSORT_IDATA:
        qsort(items, itemNumber, itemWidth,
          (int (*)(const void *, const void *)) CfgNameIDataDCompare);
        break;
      case CFG_NAME_ASORT_DDATA:
        qsort(items, itemNumber, itemWidth,
          (int (*)(const void *, const void *)) CfgNameDDataACompare);
        break;
      case CFG_NAME_DSORT_DDATA:
        qsort(items, itemNumber, itemWidth,
          (int (*)(const void *, const void *)) CfgNameDDataDCompare);
        break;
      default:
        break;
    }

  return(CFG_OK);
} /* CfgNameListSort */

/**************************************************************/
int CfgNameListSearchName(CfgNameList_t *names, char *name) {
  if (names && name) {
    int nNum;
    for (nNum = 0 ; nNum < names->size ; nNum++)
      if ((names->list[nNum].name)
       && (strcmp(names->list[nNum].name, name) == 0))
        return(nNum);
  }
  return(CFG_FAIL);
} /* CfgNameListSearchName */

/**************************************************************/
int CfgNameListSearchData(CfgNameList_t *names, void *vData) {
  if (names) {
    int nNum;
    for (nNum = 0 ; nNum < names->size ; nNum++)
      if (names->list[nNum].vData == vData)
        return(nNum);
  }
  return(CFG_FAIL);
} /* CfgNameListSearchData */

/**************************************************************/
int CfgNameListSearchInt(CfgNameList_t *names, int iData) {
  if (names) {
    int nNum;
    for (nNum = 0 ; nNum < names->size ; nNum++)
      if (names->list[nNum].iData == iData)
        return(nNum);
  }
  return(CFG_FAIL);
} /* CfgNameListSearchInt */

/**************************************************************/
int CfgNameListSearchDouble(CfgNameList_t *names, double dData) {
  if (names) {
    int nNum;
    for (nNum = 0 ; nNum < names->size ; nNum++)
      if (names->list[nNum].dData == dData)
        return(nNum);
  }
  return(CFG_FAIL);
} /* CfgNameListSearchDouble */
