/* SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Cfg - A library for Virgo process implementation
 *
 * Copyright © 2001-2021 Laboratoire de physique des particules d'Annecy - CNRS
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Fatih Bellachia <fatih.bellachia@lapp.in2p3.fr>
 *   Laurent Fournier <laurent.fournier@lapp.in2p3.fr>
 *   Alain Masserot <alain.masserot@lapp.in2p3.fr>
 */

#include <CfgHeader.h>
#ifdef USE_MW
#include <memwatch.h>
#endif /* USE_MW */

/**************************************************************/
#ifndef CFGLIST_H
#define CFGLIST_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**************************************************************/
/* sort function definition */
typedef int (*CfgNameSortFnc_t)(const void *, const void *);
typedef void *(*CfgNameFreeFnc_t)(void *);

typedef enum CfgNameSort {
  CFG_NAME_SORT_FIRST=-3,
  CFG_NAME_DSORT_DDATA=-3,  /* sort on dData in descending order */
  CFG_NAME_DSORT_IDATA=-2,  /* sort on iData in descending order */
  CFG_NAME_DSORT_NAME=-1,   /* sort on name in descending order */
  CFG_NAME_NO_SORT=0,       /* no sort */
  CFG_NAME_ASORT_NAME=1,    /* sort on name in ascending order */
  CFG_NAME_ASORT_IDATA=2,   /* sort on iData in ascending order */
  CFG_NAME_ASORT_DDATA=3,   /* sort on dData in ascending order */
  CFG_NAME_SORT_LAST=3
} CfgNameSort_t;

/* array size */
#define CFG_NAME_SORT_NUMBER CFG_NAME_SORT_LAST - CFG_NAME_SORT_FIRST +1

/* any list of names adc and sms tags */
typedef struct CfgName {
  /* private data */
  char *name;               /* indexed name list */
  int overwrites;           /* overwrite counter */
  /* public data */
  void *vData;              /* user optional void data */
  long int iData;           /* user optional integer */
  double dData;             /* user optional double */
} CfgName_t;

typedef struct CfgNameList {
  int size;                 /* number of names */
  CfgName_t *list;          /* name list */
  CfgNameSort_t sort;       /* sort option when adding or removing items */
  CfgNameFreeFnc_t freeFnc; /* free vData function when removing items */
} CfgNameList_t;

/* field access */
#define CfgNameListSize(names)       (names)->size
#define CfgNameListSortOption(names) (names)->sort

#define CfgNameListName(names, num)  (names)->list[(num)].name
#define CfgNameListVData(names, num) (names)->list[(num)].vData
#define CfgNameListIData(names, num) (names)->list[(num)].iData
#define CfgNameListDData(names, num) (names)->list[(num)].dData

/**************************************************************/
/* global variables */

/**************************************************************/
/* function prototypes */

/**************************************************************/
/* name list functions */
CfgNameList_t *CfgNameListNew();
CfgNameList_t *CfgNameListInit(CfgNameList_t *names);
CfgNameList_t *CfgNameListDone(CfgNameList_t *names);
CfgNameList_t *CfgNameListFree(CfgNameList_t *names);
int CfgNameListSetFreeFnc(CfgNameList_t *names, CfgNameFreeFnc_t freeFnc);
int CfgNameListSetSortOption(CfgNameList_t *names, int sort);

int CfgNameListAdd(CfgNameList_t *names, char *name);
int CfgNameListAddInt(CfgNameList_t *names, char *name, int iData);
int CfgNameListAddDouble(CfgNameList_t *names, char *name, double dData);
int CfgNameListSetName(CfgNameList_t *names, int nNum, char *name);
int CfgNameListSetData(CfgNameList_t *names, int nNum, void *vData);
int CfgNameListSetInt(CfgNameList_t *names, int nNum, int iData);
int CfgNameListSetDouble(CfgNameList_t *names, int nNum, double dData);

int CfgNameListNRemove(CfgNameList_t *names, int nNum);
int CfgNameListRemoveName(CfgNameList_t *names, char *name);
int CfgNameListRemoveData(CfgNameList_t *names, void *vData);
int CfgNameListRemoveInt(CfgNameList_t *names, int iData);
int CfgNameListRemoveDouble(CfgNameList_t *names, double dData);
#define CfgNameListRemove CfgNameListRemoveName /* for compatibility */
#define CfgNameListNumber(names) ((names) ? (names)->size : CFG_FAIL)
#define CfgNameListEmpty(names) ((names) && ((names)->size) ? 0 : 1)

int CfgNameListSort(CfgNameList_t *names, int sort);
int CfgNameListSearchName(CfgNameList_t *names, char *name);
int CfgNameListSearchData(CfgNameList_t *names, void *vData);
int CfgNameListSearchInt(CfgNameList_t *names, int iData);
int CfgNameListSearchDouble(CfgNameList_t *names, double dData);


#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif
