/* SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Cfg - A library for Virgo process implementation
 *
 * Copyright © 2001-2021 Laboratoire de physique des particules d'Annecy - CNRS
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Fatih Bellachia <fatih.bellachia@lapp.in2p3.fr>
 *   Laurent Fournier <laurent.fournier@lapp.in2p3.fr>
 *   Alain Masserot <alain.masserot@lapp.in2p3.fr>
 */

#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <signal.h>
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <sys/types.h>

#include <CbfClean.h>
#include <CfgMsg.h>
#include <CfgCommon.h>
#include <CfgSnprintf.h>

/* linked without -static on LINUX */
/* valgrind --tool=memcheck --leak_check=full -v <exec> <parameter> */
 
/*----------------------------------------------------------------------------*/
/* #define DEBUG(m) printf m */
#define DEBUG(m)

/*----------------------------------------------------------------------------*/
struct CfgMsg {
  char *txt;                /* message string */
  int  level;               /* error level */
  unsigned int  count;      /* repeat counter */
  CbfHEntry_t *hEntry;      /* hash table correspoonding entry */
  time_t stamp;
};


/*----------------------------------------------------------------------------*/
struct CfgMsgR {
  int nbMsg;                /* number of elements inside the queue */
  pthread_mutex_t mutex;    /* mutex for data exchanges */
  CbfH_t *hTable;           /* hash table for tagged messages */
  int overflow;
  /*--------------------------------------------------------------------------*/
  CfgMsg_t **msgArray;      /* array of messages */
  int msgIn;                /* number of next input message */
  int msgOut;               /* number of next sent message */
  /*--------------------------------------------------------------------------*/
  int nbPre;                /* Number of message preSend function */
  CfgMsgFnt_t **fntPre;     /* Message preSend function stack */
  void **paramPre;          /* Message preSend parameters stack */
  /*--------------------------------------------------------------------------*/
  int nbSend;                 /* Number of message sent function */
  CfgMsgSendFnt_t **fntSend;  /* Message send function stack */
  void **paramSend;           /* Message send parameters stack */
  /*--------------------------------------------------------------------------*/
  int nbPost;               /* Number of message postSend function */
  CfgMsgFnt_t **fntPost;    /* Message postSend function stack */
  void **paramPost;         /* Message postSent parameters stack */
  /*--------------------------------------------------------------------------*/
/*   CfgMsg_t **msgArrayOld;    /\* array of messages *\/ */
/*   int msgInOld; */
/*   int msgOutOld; */
/*   int nbMsgOld; */
};


static char *CfgMsgLevelStr[CFG_MSGLVL_MAX] = {
  "INFO...-", "WARNING-", "ERROR..-", "FATAL..-", "INFO...-"
};
static CfgMsgPrinter_t CfgMsgPrinter = vprintf;

static void CfgMsgDoPrint( char *format, ... );
static inline CfgMsg_t **CfgMsgRArrayInc( CfgMsgR_t *p );

/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CbfMsgDoPrint */
/*----------------------------------------------------------------------------*/
static void CfgMsgDoPrint( char *format, ... ) {
  va_list args; 

  if( !CfgMsgPrinter || !format ) return;
  /*--------------------------------------------------------------------------*/
  va_start( args, format );
  CfgMsgPrinter( format, args );
  va_end( args );
  /*--------------------------------------------------------------------------*/
  return;
}


/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgMsgRArrayInc */
/*----------------------------------------------------------------------------*/
static inline CfgMsg_t **CfgMsgRArrayInc( CfgMsgR_t *p ) {
  CfgMsg_t **pMsg, **array;;
  int id, rest, nbMsg;
  
 
  if( !p ) return((CfgMsg_t**)NULL);
  /*--------------------------------------------------------------------------*/
  p->overflow++;
  nbMsg = p->nbMsg;
  array = p->msgArray;
  if( !(p->msgArray = (CfgMsg_t**)malloc(2*nbMsg*sizeof(CfgMsg_t*))) ) {
    return((CfgMsg_t**)NULL);
  }
  /*--------------------------------------------------------------------------*/
  pMsg = p->msgArray; 
  id   = p->msgIn;
  rest = nbMsg - id;
  memcpy( pMsg, array + id, rest*sizeof(CfgMsg_t*));
  if( id ) memcpy( pMsg +  rest, array, id*sizeof(CfgMsg_t*));
  memset((char*)(pMsg + nbMsg), 0, nbMsg*sizeof(CfgMsg_t*));
  /*--------------------------------------------------------------------------*/
  p->msgOut = 0;
  p->msgIn  = nbMsg;
  p->nbMsg *= 2;
  free( array ); 
  /*--------------------------------------------------------------------------*/
  return( p->msgArray + p->msgIn );
}


/*----------------------------------------------------------------------------*/
/*------------------------------------------------------ CfgMsgInstallPrinter */
/*----------------------------------------------------------------------------*/
CfgMsgPrinter_t CfgMsgInstallPrinter( CfgMsgPrinter_t printer ) {
  CfgMsgPrinter_t old;

  /*--------------------------------------------------------------------------*/
  old = CfgMsgPrinter;
  CfgMsgPrinter = printer;
  /*--------------------------------------------------------------------------*/
  return(old);
}


/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgMsgGetCount */
/*----------------------------------------------------------------------------*/
int CfgMsgGetCount( const CfgMsg_t *p ) {

  if( !p ) return(CFG_FAIL);
  /*--------------------------------------------------------------------------*/
  /*--------------------------------------------------------------------------*/
  return(p->count);
}


/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgMsgGetLevel */
/*----------------------------------------------------------------------------*/
int CfgMsgGetLevel( const CfgMsg_t *p ) {

  if( !p ) return(CFG_FAIL);
  /*--------------------------------------------------------------------------*/
  /*--------------------------------------------------------------------------*/
  return(p->level);
}


/*----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CfgMsgGetLevelStr */
/*----------------------------------------------------------------------------*/
const char *CfgMsgGetLevelStr( const CfgMsg_t *p ) {
  int level;

  if( !p ) return(NULL);
  /*--------------------------------------------------------------------------*/
  level = p->level;
  if( level < 0 || level >= CFG_MSGLVL_MAX ) level = CFG_INFO;
  /*--------------------------------------------------------------------------*/
  return(CfgMsgLevelStr[level]);
}


/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CfgMsgGetTimeStamp */
/*----------------------------------------------------------------------------*/
time_t CfgMsgGetTimeStamp( const CfgMsg_t *p ) {

  if( !p ) return((time_t)NULL);
  /*--------------------------------------------------------------------------*/
  /*--------------------------------------------------------------------------*/
  return(p->stamp);
}


/*----------------------------------------------------------------------------*/
/*------------------------------------------------------ CfgMsgGetTimeStampTm */
/*----------------------------------------------------------------------------*/
const struct tm *CfgMsgGetTimeStampTm( const CfgMsg_t *p ) {

  if( !p ) return((struct tm*)NULL);
  /*--------------------------------------------------------------------------*/
  /*--------------------------------------------------------------------------*/
  return(gmtime(&p->stamp));
}


/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgMsgGetTxt */
/*----------------------------------------------------------------------------*/
char *CfgMsgGetTxt( const CfgMsg_t *p ) {

  if( !p ) return(NULL);
  /*--------------------------------------------------------------------------*/
  /*--------------------------------------------------------------------------*/
  return(p->txt);
}


/*----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgMsgFree */
/*----------------------------------------------------------------------------*/
CfgMsg_t *_CfgMsgFree( CfgMsg_t *p ) {

  if( !p ) return(p);
  /*--------------------------------------------------------------------------*/
  if( p->txt ) { free(p->txt); p->txt = NULL; }
  p->hEntry = NULL;
  p->count = p->level = 0;
  p->stamp = 0;
  free(p);
  /*--------------------------------------------------------------------------*/
  return((CfgMsg_t*)NULL);
}


/*----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgMsgFree */
/*----------------------------------------------------------------------------*/
CfgMsg_t *CfgMsgFree( CfgMsg_t *p ) {

  if( !p ) return(p);
  /*--------------------------------------------------------------------------*/
  if( p->hEntry ) CbfHEntryRemove(p->hEntry);
  /*--------------------------------------------------------------------------*/
  return(_CfgMsgFree(p));
}


/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CfgMsgNew */
/*----------------------------------------------------------------------------*/
CfgMsg_t *CfgMsgNew( int level, const char *txt ) {
  CfgMsg_t *p;
  
  if( !txt ) return((CfgMsg_t*)NULL);
  /*--------------------------------------------------------------------------*/
  if( !(p = (CfgMsg_t*)malloc(sizeof(CfgMsg_t)) ) ) return(p);
  memset((char*)p, 0, sizeof(CfgMsg_t));
  /*--------------------------------------------------------------------------*/
  p->level = level;
  time( &p->stamp );
  if( !(p->txt = strdup(txt)) ) {    
    CfgMsgDoPrint("CfgMsgNew> malloc failed\n");
    p = CfgMsgFree(p);
  }
  /*--------------------------------------------------------------------------*/
  return(p);
}


/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CfgMsgNewArgs */
/*----------------------------------------------------------------------------*/
CfgMsg_t *CfgMsgNewArgs( int level, char *fmt, va_list args ) {
  CfgMsg_t *p;
  char *last;
  int rtn;
  
  if( !fmt ) return((CfgMsg_t*)NULL);
  /*--------------------------------------------------------------------------*/
  if( !(p = (CfgMsg_t*)malloc(sizeof(CfgMsg_t)) ) ) return(p);
  memset((char*)p, 0, sizeof(CfgMsg_t));
  p->level = level;
  time( &p->stamp );
  /*--------------------------------------------------------------------------*/
  if( !(p->txt = (char*)malloc(CFG_MSG_BUF_SIZE)) ) {
    CfgMsgDoPrint( "\nCfgMsgNewArgs> malloc failed\n" );
    return(CfgMsgFree(p));
  }
  if( (rtn = vsnprintf( p->txt, CFG_MSG_BUF_SIZE-1, fmt, args )) < 0  ) {
    CfgMsgDoPrint( "CfgMsgNewArgs> vsnprintf failed\n" );
    return(CfgMsgFree(p));    
  }
  if( rtn > 1 ) {
    last  = p->txt + rtn -1;
    if( last[0] == '\n' || last[0] == '\r' ) last[0] = '\0';
  } else p = CfgMsgFree(p);
  /*--------------------------------------------------------------------------*/
  return(p);
}


/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgMsgNewFmt */
/*----------------------------------------------------------------------------*/
CfgMsg_t *CfgMsgNewFmt( int level, char *fmt, ... ) {
  CfgMsg_t *p;
  va_list args;

  if( !fmt ) return((CfgMsg_t*)NULL);
  /*--------------------------------------------------------------------------*/
  va_start( args, fmt );
  p = CfgMsgNewArgs( level, fmt, args );
  va_end( args );
  /*--------------------------------------------------------------------------*/
  return(p);
}


/*----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgMsgPrint */
/*----------------------------------------------------------------------------*/
void CfgMsgPrintv( int level, char *fmt, va_list args ) { 
  char *lFmt, strTime[50];
  time_t current;

  /*--------------------------------------------------------------------------*/
  time( &current );
  strftime( strTime, sizeof strTime, CFG_TIME_FMT, gmtime(&current));
  lFmt = CfgStrFmtCat( NULL, CfgFalse, "@%s>%s%s\n", strTime, 
		       CfgMsgLevelStr[level], fmt);
  vprintf( lFmt, args );
  free(lFmt);
  /*--------------------------------------------------------------------------*/
  return; 
}


/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CfgMsgRAddFnt */
/*----------------------------------------------------------------------------*/
int CfgMsgRAddFnt( CfgMsgR_t *p, CfgMsgSendFnt_t *fSend, void *q, 
		   CfgMsgFnt_t *fPre, CfgMsgFnt_t *fPost) {
  int nb;

  if( !CbfCleanFind( "CfgMsgRFree", p ) ) return(CFG_FAIL);
  /*------------------------------------------ Message preSend function stack */
  if( fPre ) {
    nb = p->nbPre;
    p->fntPre   = (CfgMsgFnt_t**)realloc(p->fntPre,(nb+1)*sizeof(CfgMsgFnt_t*));
    p->paramPre = (void**)realloc(p->paramPre, (nb+1)*sizeof(void*));
    if( !p->fntPre || !p->paramPre ) {
      CfgMsgDoPrint( "CfgMsgRAddFnt> fntPre realloc failed\n" );
      return(CFG_FAIL);
    }
    /*------------------------------------------------------------------------*/
    p->fntPre[nb]   = fPre;
    p->paramPre[nb] = q;
    p->nbPre++;
  }
  /*--------------------------------------------- Message Send function stack */
  if( fSend ) {
    nb = p->nbSend;
    p->fntSend   = (CfgMsgSendFnt_t**)realloc(p->fntSend,
					      (nb+1)*sizeof(CfgMsgSendFnt_t*));
    p->paramSend = (void**)realloc(p->paramSend, (nb+1)*sizeof(void*));
    if( !p->fntSend || !p->paramSend ) {
      CfgMsgDoPrint( "CfgMsgRAddFnt> fntSend realloc failed\n" );
      return(CFG_FAIL);
    }
    /*------------------------------------------------------------------------*/
    p->fntSend[nb]   = fSend;
    p->paramSend[nb] = q;
    p->nbSend++;
  }
  /*----------------------------------------- Message postSend function stack */
  if( fPost ) {
    nb = p->nbPost;
    p->fntPost   = (CfgMsgFnt_t**)realloc(p->fntPost,
					  (nb+1)*sizeof(CfgMsgFnt_t*));
    p->paramPost = (void**)realloc(p->paramPost, (nb+1)*sizeof(void*));
    if( !p->fntPost || !p->paramPost ) {
      CfgMsgDoPrint( "CfgMsgRAddFnt> fntPost realloc failed\n" );
      return(CFG_FAIL);
    }
    /*------------------------------------------------------------------------*/
    p->fntPost[nb]   = fPost;
    p->paramPost[nb] = q;
    p->nbPost++;
  }
  /*--------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CfgMsgRAddMsg */
/*----------------------------------------------------------------------------*/
int CfgMsgRAddMsg( CfgMsgR_t  *p, CfgMsg_t *msg ) {
  CfgMsg_t **pMsg;

  if( !p || !msg ) return(CFG_FAIL);
  /*--------------------------------------------------------------------------*/
  pthread_mutex_lock( &p->mutex );
  /*------------------------------------ Look for the first element available */
  pMsg  = p->msgArray + p->msgIn;
  /*------------------------------------- No more space - Increase the buffer */
  if( pMsg[0] && !(pMsg = CfgMsgRArrayInc( p )) ) {
    pthread_mutex_unlock( &p->mutex );
    return(CFG_FAIL);
  }
    
  /*--------------------------------------------------------------------------*/
  pMsg[0] = msg;
  DEBUG(("CfgMsgRAddMsg>%d %p - %d msg \"%s\" cnt %d - %s %p\n",
	 p->msgIn, iMsg, iMsg->errLvl, iMsg->msg, iMsg->count, iMsg->hEntry->key, 
	 iMsg->hEntry ));
  p->msgIn++; p->msgIn %= p->nbMsg;
  pthread_mutex_unlock( &p->mutex );
  /*--------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgMsgRAddMsgTag */
/*----------------------------------------------------------------------------*/
int CfgMsgRAddMsgTag( CfgMsgR_t  *p, CfgMsg_t *iMsg, char *tag ) {
  CbfHEntry_t *entry;
  CfgMsg_t **pMsg, *msg;
  char *tmp;

  if( !p || !iMsg ) return(CFG_FAIL);
  /*--------------------------------------------------------------------------*/
  if( !tag ) return(CfgMsgRAddMsg( p, iMsg ));
  /*-------------------------------------------------- Start critical section */
  pthread_mutex_lock( &p->mutex );
  /*------------------------------------------------------------------------- */
  if( (entry = CbfHFind( p->hTable, tag )) ) {
    msg  = (CfgMsg_t*)entry->data;
    /*------------------------------------------------- Tag already available */
    msg->count += 1;
    /*-------------------------------------------- Update the message content */
    tmp         = msg->txt;
    msg->txt    = iMsg->txt;
    iMsg->txt   = tmp;  
    DEBUG(("CfgMsgRAddMsgTag>%p %p - %d msg \"%s\" cnt %d - %s %p\n", pMsg, iMsg,
	   msg->errLvl, msg->msg, msg->count, msg->hEntry->key,
	   msg->hEntry ));
    /*-------------------------------------------------- End critical section */
    pthread_mutex_unlock( &p->mutex );
    /*------------------------------------------------------------------------*/
    iMsg = CfgMsgFree( iMsg );
  } else if( !iMsg->hEntry )  { /*---------- Message without tag to be tagged */
    /*------------------------------------------------------------------------*/
    pMsg  = p->msgArray + p->msgIn;
    /*----------------------------------- No more space - Increase the buffer */
    if( pMsg[0] && !(pMsg =  CfgMsgRArrayInc( p )) ) {
      pthread_mutex_unlock( &p->mutex );
      return(CFG_FAIL);
    }
    
    /*-------------------------------------------------------------------------*/
    iMsg->hEntry = CbfHEnter( p->hTable, tag, (void*)iMsg );
    pMsg[0] = iMsg;
    DEBUG(("CfgMsgRAddMsgTag>End %d %p - %d msg \"%s\" cnt %d - %s %p\n",
	   p->msgIn, iMsg, iMsg->errLvl, iMsg->msg, iMsg->count, iMsg->hEntry->key, 
	   iMsg->hEntry ));
    p->msgIn++; p->msgIn %= p->nbMsg;
    /*-------------------------------------------------- End critical section */
    pthread_mutex_unlock( &p->mutex );
  } else {
    /*-------------------------------------------------- End critical section */
    pthread_mutex_unlock( &p->mutex );
    CfgMsgDoPrint("CfgMsgRAddMsgTag> Reject iTag %s %p - %d msg \"%s\""
		  " cnt %d - %p\n", tag,
		  iMsg, iMsg->level, iMsg->txt, iMsg->count, iMsg->hEntry );
  }
  /*--------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgMsgRFree */
/*----------------------------------------------------------------------------*/
CfgMsgR_t *CfgMsgRFree( CfgMsgR_t *p ) {
  CfgMsg_t **pMsg;
  int i, nb;

  if( !CbfCleanFind( "CfgMsgRFree", p) ) return(p);
  /*--------------------------------------------------------------------------*/
  pthread_mutex_lock( &p->mutex );
  for( pMsg = p->msgArray, i = 0, nb = p->nbMsg; i < nb; i++, pMsg++ ) {
    if( !pMsg[0] ) continue;
    if( pMsg[0]->txt ) {
      CfgMsgDoPrint("CfgMsgRFree> %d/%d msg: %p, txt %s\n",
		    i, nb, pMsg[0],  pMsg[0]->txt);
    } else {
    CfgMsgDoPrint("CfgMsgRFree> %d/%d msg: %p, txt %p level %d entry %p\n",
		  i, nb, pMsg[0],
		  pMsg[0]->txt, pMsg[0]->level, pMsg[0]->hEntry);
    }
    pMsg[0] = CfgMsgFree( pMsg[0] );
  }
  CfgMsgPrinter = vprintf;
  if( p->msgArray ) { free(p->msgArray); p->msgArray = NULL; }
  if( p->hTable) p->hTable = CbfHFree(p->hTable);
  if( p->fntPre ) { free(p->fntPre); p->fntPre = NULL; }
  if( p->paramPre ) { free(p->paramPre); p->paramPre = NULL; }
  if( p->fntSend ) { free(p->fntSend); p->fntSend = NULL; }
  if( p->paramSend ) { free(p->paramSend); p->paramSend = NULL; }
  if( p->fntPost ) { free(p->fntPost); p->fntPost = NULL; }
  if( p->paramPost ) { free(p->paramPost); p->paramPost = NULL; }
  pthread_mutex_unlock( &p->mutex );
  pthread_mutex_destroy( &p->mutex );
  /*--------------------------------------------------------------------------*/
  memset((char*)p, 0, sizeof(CfgMsgR_t));
  CbfCleanDone("CfgMsgRFree", p);
  free(p);
  /*--------------------------------------------------------------------------*/
  return((CfgMsgR_t*)NULL);
}


/*----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgMsgRNew */
/*----------------------------------------------------------------------------*/
CfgMsgR_t *CfgMsgRNew( int nbMsg ) { 
  CfgMsgR_t *p = (CfgMsgR_t*)NULL;

  if( nbMsg <= 0 ) return(p);
  /*--------------------------------------------------------------------------*/
  if( !(p = (CfgMsgR_t*)malloc( sizeof(CfgMsgR_t) )) ) {
    CfgMsgDoPrint( "CfgMsgRNew> malloc failed" );
    return(p);
  }
  memset((char*)p, 0, sizeof(CfgMsgR_t));
  CbfCleanAddRoot( "CfgMsgRFree", (CbfCleanFnt_t)CfgMsgRFree, p);

  /*--------------------------------------------------------------------------*/
  p->nbMsg = nbMsg;
  if( !(p->msgArray = (CfgMsg_t**)malloc(nbMsg*sizeof(CfgMsg_t*))) ) {
    CfgMsgDoPrint( "CfgMsgRNew> malloc failed"  );
    return(CfgMsgRFree(p));
  }
  memset((char*)p->msgArray, 0, nbMsg*sizeof(CfgMsg_t*));
  if( 
#ifdef RIO806X
      pthread_mutex_init(&p->mutex, pthread_mutexattr_default) == -1
#else
    pthread_mutex_init(&p->mutex, NULL) == -1
#endif
      ) {
    CfgMsgDoPrint( "CfgMsgRNew> pthread_mutex_init failed" );
    return(CfgMsgRFree(p));
  }
  if( !(p->hTable = CbfHNew(nbMsg)) ) {
    CfgMsgDoPrint( "CfgMsgRNew> CbfHNew failed" );
    return(CfgMsgRFree(p));
  }
  pthread_mutex_unlock( &p->mutex );
  /*--------------------------------------------------------------------------*/
  return(p);
}
  

/*----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgMsgRRmFnt */
/*----------------------------------------------------------------------------*/
int CfgMsgRRmFnt( CfgMsgR_t *p, CfgMsgSendFnt_t *f, void *q, 
		   CfgMsgFnt_t *fPre, CfgMsgFnt_t *fPost) {
  int i, nb;
  CfgMsgSendFnt_t **iF;
  CfgMsgFnt_t **jF;
  void **iQ;
  
  
  if( !CbfCleanFind( "CfgMsgRFree", p) ) return(CFG_FAIL);
  /*--------------------------------------------------------------------------*/
  if( fPre ) {
    jF = p->fntPre;
    nb = p->nbPre;
    for( i = 0; i < nb; i++, jF++ ) {
      if( jF[0] == fPre ) { p->nbPre--;  break; }
    }
    for( i = i; i < nb-1; i++)  jF[i] = jF[i+1];
  }
  /*--------------------------------------------------------------------------*/
  if( f ) { 
    iF = p->fntSend;
    iQ = p->paramSend;
    nb = p->nbSend;
    for( i = 0; i < nb; i++, iF++, iQ++ ) {
      if( iF[0] == f && iQ[0] == q ) { p->nbSend--;  break; }
    }
    for( i = i; i < nb-1; i++) {
      iF[i] = iF[i+1];
      iQ[i] = iQ[i+1];
    }
  }
  /*--------------------------------------------------------------------------*/
  if( fPost ) {
    jF = p->fntPost;
    nb = p->nbPost;
    for( i = 0; i < nb; i++, jF++ ) {
      if( jF[0] == fPost ) { p->nbPost--;  break; }
    }
    for( i = i; i < nb-1; i++)  jF[i] = jF[i+1];
  }
  /*--------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgMsgRSend */
/*----------------------------------------------------------------------------*/
 int CfgMsgRSend( CfgMsgR_t *p, const CfgMsg_t *uiMsg ) {
   int i, id, nbMsg, j , nb, fatal;
  CfgMsg_t *msg, **pMsg ;
  CfgMsgFnt_t **e;
  CfgMsgSendFnt_t **f;
  void **q;

  if( !p ) return(CFG_FAIL);
  /*------------------------------------------------------- Pre send function */
  nb = p->nbPre;
  e  = p->fntPre;
  q  = p->paramPre;
  for( j = 0; j < nb; j++, q++, e++ ) e[0]( q[0] );
  

  /*--------------------------------------------------------------------------*/
  msg   = (CfgMsg_t *)NULL;
  /*-------------------------------------------------- Start critical section */
  pthread_mutex_lock( &p->mutex );
  /* output only the current message -----------------------------------------*/
  /* not the one added while sending the others------------------------------ */
  nbMsg = p->msgIn - p->msgOut;
  if( nbMsg < 0 ) nbMsg += p->nbMsg;
  for( pMsg = p->msgArray + p->msgOut, fatal = i = 0; 
/*      pMsg[0] != NULL  && i < p->nbMsg; */
       pMsg[0] != NULL  && i < nbMsg;   
       i++, pMsg = p->msgArray + p->msgOut  ) {
    /*--------------------------------------------------- Extract the message */
    msg       = pMsg[0]; 
    pMsg[0]   = (CfgMsg_t *)NULL;
    /*----------------------------------------- Increase the mgsOut reference */
    id        = p->msgOut;
    p->msgOut = (id+1)%p->nbMsg;
    /*------------------------------------- Remove the tag for the Hash table */
    if( msg->hEntry ) { CbfHEntryRemove(msg->hEntry); msg->hEntry = NULL; }

    /*-------------------------------------------------- End critical section */
    pthread_mutex_unlock( &p->mutex );
    if( msg->level == CFG_FATAL ) fatal++;
    /*------------------------------------------ Call the Sent function stack */
    DEBUG(("CfgMsgRSend> id %d - %d msg \"%s\" cnt %d\n", id, msg->level,
	   msg->txt, msg->count));
    nb = p->nbSend;
    f  = p->fntSend;
    q  = p->paramSend;
    for( j = 0; j < nb; j++, q++, f++ ) f[0]( q[0], msg );

    /*-------------------------------------------- Release the CfgMsg message */
    msg = CfgMsgFree( msg );

    /*------------------------------------------------ Start critical section */
    pthread_mutex_lock( &p->mutex );
 }
  msg = (CfgMsg_t*)NULL;
  if( p->overflow ) {
    msg = CfgMsgNewFmt( CFG_INFO, "CfgMsgSendR> mailbox set to %d - overflow %d",
			p->nbMsg, p->overflow );
    p->overflow = 0;    
  }
  /*---------------------------------------------------- End critical section */
  pthread_mutex_unlock( &p->mutex );

  /*------------------------------------------------------- Extra Cfg message */
  if( msg ) { /*--------------------------------------------- MailBox oveflow */
    nb = p->nbSend;
    f  = p->fntSend;
    q  = p->paramSend;
    for( j = 0; j < nb; j++, q++, f++ ) f[0]( q[0], msg );
    i++;
    msg = CfgMsgFree( msg );
  }
  if( uiMsg ) { /*------------------------------------------------ User Info  */
    nb = p->nbSend;
    f  = p->fntSend;
    q  = p->paramSend;
    for( j = 0; j < nb; j++, q++, f++ ) f[0]( q[0], uiMsg );
    i++;
  }

  /*------------------------------------------------------ Post send function */
  /*-------------------------------------- Reverse order as Pre send function */
  nb = p->nbPost - 1;
  e  = p->fntPost + nb ;
  q  = p->paramPost + nb ;
  for( j = nb; j >= 0 ; j--, q--, e-- ) e[0]( q[0] );
  
  DEBUG(("CfgMsgRSend> %d message(s) sent\n", i));
  /*--------------------------------------------------------------------------*/
  if( fatal ) return( -1*i ); else return(i);
}




