/* SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Cfg - A library for Virgo process implementation
 *
 * Copyright © 2001-2021 Laboratoire de physique des particules d'Annecy - CNRS
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Fatih Bellachia <fatih.bellachia@lapp.in2p3.fr>
 *   Laurent Fournier <laurent.fournier@lapp.in2p3.fr>
 *   Alain Masserot <alain.masserot@lapp.in2p3.fr>
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <setjmp.h>
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <string.h>
#include <strings.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <CbfHash.h>
#include <CbfClean.h>
#include <CfgList.h>
#include <CfgOut.h>
#include <Reg.h>


#ifdef USE_MW
#include <memwatch.h>
#endif /* USE_MW */
#ifdef USE_MPATROL
#include <mpatrol.h> 
#endif /* USE_MPATROL */

#define CFGOUT_MIN_PRIO      0
#define CFGOUT_MAX_PRIO      10
#define CFGOUT_WATCHDOG      2.
#define CFGOUT_MAX_CONSUMER  5
#define CFGOUT_MAX_GROUP     10
#define CFGOUT_THRESHOLD     -1
#define CFGOUT_DEBUG         0
#define CFGOUT_NO_SOCKID     -1
#define CFGOUT_DEFAULT_NAME  "default"
#define CFGOUT_MAGIC_CHAR    " |*[?+^{("
#define CFGOUT_CON_GRANTED   0xC000BABEL
#define CFGOUT_CON_DENIED    0xDEADBABEL

typedef struct CfgOut{
  int debug; /*--------------------------------- Debug level to show messages */
  int threshold; /*------ Threshold of priority needed to activate a consumer */
  int nbConsumer; /*--------------------------------------------- # consumers */
  int maxConsumer; /*---------------------------- Maximum number of consumers */
  int maxGroup; /*---------------------------------- Maximum number of groups */
  time_t start; /*---------------------------------- Start time (in seconds ) */
  int nsPeriod; /*------------- Period for requesting infos to the NameServer */
  CfgBool_t check; /*- Tell if connection check to the consumers is requiered */
  CfgBool_t statistic; /*--------------------- Tell if statistic is requiered */
  CfgNameList_t *csList; /*--------------------- Pointer of consumer nameList */
  CfgNameList_t *csOnReqList; /*------ Pointer of onRequest consumer nameList */
  char group[CFG_MAX_NAME]; /*------------------------- Group ID of consumers */
  CfgOutBuildMsg_t buildMsg; /*------------------------------ User's function */
  void *arg; /*---------------------------------------------- User's argument */
  /*=================================================== PRIVATE INFORMATIONS =*/
  CfgNameList_t *gpList; /*------------------------ Pointer of group nameList */
  CfgNameList_t *dmList; /*----------------------- Pointer of domain nameList */
  int index; /*--------------------------------------- Last index of nameList */
} CfgOut_t;

typedef struct CfgOutAuthentication{
  char owner[CFG_MAX_NAME]; /*----------------------------- Owner of consumer */
  char host[CFG_MAX_NAME]; /*--------------------------- Hostname of consumer */
  CfgBool_t accepted; /*------------------------ Tell if consumer is accepted */
  RegExp reg; /*-------------------------------- Regular Expression structure */
  struct CfgOutAuthentication *next; /*------------- Pointer to the next item */
  /*=================================================== PRIVATE INFORMATIONS =*/
  CfgOut_t *cfgOut; /*---------------------- Keep pointer of CfgOut structure */
  int nbGroups; /*---------------- # groups post message to the same consumer */
  int retry; /*----------- Current maximum number of retries before remove it */
  int priority; /*--------------------------------- Current consumer priority */
  CfgOutRequest_t request; /*------------------ Current consumer request flag */
} CfgOutAuthentication_t;

/*========================================= Define for future implementation =*/
typedef struct CfgOutStat{
  double min; /*----------------------------------------- Compute the minimum */
  double avg; /*----------------------------------------- Compute the average */
  double max; /*----------------------------------------- Compute the maximum */
} CfgOutStat_t;

typedef struct CfgOutInfo{
  int sended; /*----------------------------------- Number of sended messages */
  CfgOutStat_t size; /*--------------------------- Statistics of message size */
  CfgOutStat_t time; /*-------------------------- Statistics of transfer time */
} CfgOutInfo_t;

typedef struct CfgConsumer{
  char name[CFG_MAX_NAME]; /*------------------------------- Name of consumer */
  int retry; /*------------------- Maximum number of retries before remove it */
  CfgOutRequest_t request; /*-------------------------- Consumer request flag */
  CmMessage message; /*---------------------------------- CmMessage structure */
  CfgOutInfo_t info; /*--------------------------- Information about consumer */
  CfgOutStatus_t status; /*------------------------------- Status of consumer */
  time_t start; /*---------------------------------- Start time (in seconds ) */
  /*=================================================== PRIVATE INFORMATIONS =*/
  CmMessageState state; /*---------------------------------- Cm message state */
  CmMessageIterator iterator; /*--------------------------- CmMessageIterator */
  int period; /*------ Period between two connection when host is unreachable */
  int counter; /*----------------------------------------- Counter of retries */
  CfgBool_t complete; /*------------------------- Tell if message is complete */
  CfgBool_t alive; /*------------------------------ Tell if consumer is alive */
  CfgBool_t remove; /*---------------------- Tell if consumer must be removed */
  CfgBool_t connectDone; /*---------- Tell if connection to consumer has done */
  CfgOut_t *cfgOut; /*---------------------- Keep pointer of CfgOut structure */
  char host[CFG_MAX_NAME]; /*--------------------------- Hostname of consumer */
  int port; /*--------------------------------------------------- Port number */
  int sid; /*------------------------------------------------------ Socket ID */
  CfgOutAuthentication_t *authp; /*------------ Consumer owner authentication */
} CfgConsumer_t;

char *CfgOutCmState[] = {
  "CmMessage dead", "CmMessage not ready", "CmMessage new",
  "CmMessage not empty", "CmMessage closed", "CmMessage sending"
};

char *CfgOutHostState[] = {
  "None", "unknown by NS", "connect in progress", "host unreachable",
  "host alive", "sending", "message new"
};

static jmp_buf         CfgOutEnv, *CfgOutEnvp = (jmp_buf *)NULL;
static CbfH_t          *CfgOutHashTable = (CbfH_t *)NULL;
static CfgBool_t       CfgOutDone = CfgFalse;
static CfgBool_t       CfgOutTimeout = CfgFalse;
static char            CfgOutDefaultGroup[256] = { CFGOUT_DEFAULT_NAME };
static CfgOutPrinter_t CfgOutPrinter = (CfgOutPrinter_t)vprintf;

#ifndef __linux__
extern int  sethostent(int);
#endif /* __linux__ */
extern void endhostent(void);

extern void CfgNameListPrint(CfgNameList_t *);

/******************************************************************************/
/*                       PRIVATE FUNCTIONS PROTOTYPES                         */
/******************************************************************************/

static int CfgOutActiveWatchdog(double, struct sigaction *);
static void CfgOutAlarmHandler(int);
static int CfgOutCheckGroupSet(void *, CfgBool_t, char *);
static CmMessageStatus CfgOutCmCsmAdd(CmMessage, char *, char *);
static CmMessageStatus CfgOutCmCsmRemove(CmMessage, char *, char *);
static CmMessageStatus CfgOutCmGroupAdd(CmMessage, char *, char *);
static CmMessageStatus CfgOutCmGroupRemove(CmMessage, char *, char *);
static CmMessageStatus CfgOutCmNSAddress(CmMessage, char *, char *);
static CmMessageStatus CfgOutCmNSNames(CmMessage, char *, char *);
static CfgBool_t CfgOutConnectionCheck(char *, char *, double);
static int CfgOutConnectionClose(CfgConsumer_t *);
static void CfgOutConnectionHandler(CmConnect);
static int CfgOutConnectionOpen(char *, char *);
static void _CfgOutConnectionOpen(char *, double);
static int CfgOutConsumerAdd(void *, char *, int, int, char *, CfgOutRequest_t,
                             char *, char *);
static int CfgOutConsumerGetNames(CfgOut_t *, char *);
static CmConnect CfgOutConsumerGetReference(char *, char *, double);
static CfgConsumer_t *CfgOutConsumerFree(CfgConsumer_t *);
static CfgBool_t CfgOutConsumerIsAlive(char *, char *, CfgBool_t);
static CfgBool_t CfgOutConsumerIsServer(char *);
static int CfgOutConsumerListAdd(CfgOut_t *, char *, int, int, CfgOutRequest_t,
                                 char *, char *);
static CfgConsumer_t *CfgOutConsumerSearch(char *, char *);
static CfgConsumer_t *CfgOutConsumerSearchByMsg(CmMessage);
static void CfgOutConsumerState(CfgConsumer_t *, char *, char *, CfgBool_t);
static void CfgOutCsmDisconnectionHandler(CmConnect);
static CfgOut_t *CfgOutCurrentPointer(CfgOut_t *);
static int CfgOutDebugSet(void *, int, char *);
static CfgOut_t *_CfgOutFree(CfgOut_t *);
static CfgOut_t *CfgOutGetOut(char *);
static int CfgOutGroupPrint(char *, int, char *, ...);
static CbfH_t *CfgOutHFree(CbfH_t *);
static int CfgOutMaxConsumerSet(void *, int, char *);
static int CfgOutMaxGroupSet(void *, int, char *);
static int CfgOutNsPeriodSet(void *, int, char *);
static CmMessageStatus CfgOutPostHandler(CmMessage, char *, char *);
static void CfgOutPrintMsgError(CfgConsumer_t *);
static int CfgOutReleaseWatchdog(struct sigaction *);
static int CfgOutRemoveEntryList(CfgOut_t *, char *, int);
static CfgOut_t *CfgOutSearchByGroup(char *);
static int CfgOutStatSet(void *, CfgBool_t, char *);
static int CfgOutThresholdSet(void *, int, char *);
static void CfgOutTagServer(char *);

CfgBool_t CfgOutIsConnectionGranted(char *);
void CfgOutRegisterDisconnection(char *);

/******************************************************************************/
/*                           PRIVATE FUNCTIONS                                */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*- CfgOutActiveWatchdog -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutActiveWatchdog(double watchdog, struct sigaction *oact){
struct itimerval itimer;
struct sigaction act;

  /*---------------------------------------------- Setup the signal handler */
  act.sa_handler = CfgOutAlarmHandler;
  sigemptyset(&act.sa_mask);
  act.sa_flags = 0;
  if (sigaction(SIGALRM, &act, oact) == -1) {
    fprintf(stderr, "CfgOutActiveWatchdog: sigaction: %s\n", strerror(errno));
    return(CFG_FAIL);
  }

  memset((char *)&itimer, 0, sizeof(struct itimerval));

  itimer.it_value.tv_sec  = (time_t)watchdog;
  itimer.it_value.tv_usec = (time_t)((watchdog - (time_t)watchdog) * 1e6);

  if(setitimer(ITIMER_REAL, &itimer, (struct itimerval *)NULL) == -1){
    fprintf(stderr, "CfgOutActiveWatchdog: setitimer: %s\n", strerror(errno));
    return(CFG_FAIL);
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutAlarmHandler -------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void CfgOutAlarmHandler(int signo){

  CfgOutGroupPrint((char *)NULL, 5,
                   "CfgOut> Receive signal ALARM (timeout expired)...\n");

  CfgOutTimeout = CfgTrue;

  if(CfgOutEnvp != (jmp_buf *)NULL){ longjmp(CfgOutEnv, CFG_FAIL); }

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCheckGroupSet ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutCheckGroupSet(void *arg, CfgBool_t check, char *group){
register CfgOut_t *cfgOut;

  if((cfgOut = CfgOutGetOut(group)) != (CfgOut_t *)NULL){
    cfgOut->check = check;
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCmCsmAdd -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CmMessageStatus CfgOutCmCsmAdd(CmMessage message, char *sender,
                                      char *server){
char *group;
char *consumer;
int  retry, priority;
char name[256];

  retry    = CmMessageGetInt(message);
  priority = CmMessageGetInt(message);
  group    = CmMessageGetText(message);
  consumer = CmMessageGetText(message); 

  if(consumer == (char *)NULL){
    if(sender == (char *)NULL){ return(CmMessageOk); }
    consumer = name;
    strcpy(consumer, sender);
  }

  CfgOutGroupPrint(group, 0,
                   "CfgOut> receive request ADD from %s for group %s.\n",
                   consumer, group? group: CfgOutGroupDefaultGet());

  {
    CmConnect cmConnect = CmMessageGetConnect(message);
    char *owner = CmConnectGetOwner(cmConnect);
    char *host  = CmConnectGetHost(cmConnect);

    CfgOutGroupConsumerAdd(group, consumer, retry, priority, CfgOutDynamic,
                           owner, host);
  }

  return(CmMessageOk);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCmCsmRemove --------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CmMessageStatus CfgOutCmCsmRemove(CmMessage message, char *sender,
                                         char *server){
char *group, *consumer;
char name[256];

  consumer = CmMessageGetText(message);
  group    = CmMessageGetText(message);

  if(consumer == (char *)NULL){
    if(sender == (char *)NULL){ return(CmMessageOk); }

    consumer = name;
    strcpy(consumer, sender);
  }

  CfgOutGroupPrint(group, 0, "CfgOut> "
                   "receive request REMOVE from %s for group %s.\n",
                   consumer, group? group: CfgOutGroupDefaultGet());

  CfgOutGroupConsumerRemove(group, consumer);

  return(CmMessageOk);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCmGroupAdd ---------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CmMessageStatus CfgOutCmGroupAdd(CmMessage message, char *sender,
                                        char *server){
char *domain;
char *group;

  group  = CmMessageGetText(message);
  domain = CmMessageGetText(message);

  CfgOutGroupPrint(domain, 0,
                   "CfgOut> receive request from %s ADD GROUP %s to %s.\n",
                   sender,
                   group? group: CfgOutGroupDefaultGet(),
                   domain? domain: CfgOutGroupDefaultGet());

  CfgOutDomainGroupAdd(domain, group);

  return(CmMessageOk);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCmGroupRemove ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CmMessageStatus CfgOutCmGroupRemove(CmMessage message, char *sender,
                                           char *server){
char *domain;
char *group;

  group  = CmMessageGetText(message);
  domain = CmMessageGetText(message);

  CfgOutGroupPrint(domain, 0,
                   "CfgOut> receive request from %s REMOVE GROUP %s from %s.\n",
                   sender,
                   group? group: CfgOutGroupDefaultGet(),
                   domain? domain: CfgOutGroupDefaultGet());

  CfgOutDomainGroupRemove(domain, group);

  return(CmMessageOk);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCmNSAddress --------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CmMessageStatus CfgOutCmNSAddress(CmMessage message, char *sender,
                                         char *server){
char* name;

  name = CmMessageGetText(message);

  if(CmMessageGetItemType(message) == CmMessageItemText){
    char* host;
    int port;
    int multiple;
    char* owner;
    CmConnect cmConnect/* , cmServer */;

    host     = CmMessageGetText(message);
    port     = CmMessageGetInt(message);
    multiple = CmMessageGetInt(message);
    owner    = CmMessageGetText(message);

/*     cmServer  = CmConnectGetServer(CmMessageGetConnect(message)); */
    cmConnect = CmConnectGetReference(name);
    if(cmConnect == NULL){
      cmConnect = CmConnectNewWithSocket(name, -1);
      CmConnectSetHost(cmConnect, host);
      CmConnectSetPort(cmConnect, port);
      CmConnectSetOwner(cmConnect, owner);
      if(multiple){ CmConnectSetMultiple(cmConnect); }
    }
  }

  if(CmMessageGetItemType(message) == CmMessageItemInt){
    int transaction = CmMessageGetInt(message);
    CmTerminateTransaction(transaction);
  }

  return(CmMessageOk);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCmNSNames ----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CmMessageStatus CfgOutCmNSNames(CmMessage message, char *sender,
                                       char *server){
CfgOut_t *cfgOut = CfgOutCurrentPointer((CfgOut_t *)NULL);

  while(CmMessageGetItemType(message) == CmMessageItemText){
    char* name = CmMessageGetText(message);

    if(CfgNameListSearchName(cfgOut->csList, name) == CFG_FAIL){
      CbfHEntry_t *itemp;

      itemp = CbfHFind(CfgOutHashTable, "regular exp");
      if(itemp != (CbfHEntry_t *)NULL){
        CfgOutAuthentication_t *authp = (CfgOutAuthentication_t *)itemp->data;

        for(; authp != (CfgOutAuthentication_t *)NULL; authp = authp->next){
          if(RegExpMatch(authp->reg, name)){
            if(cfgOut->nbConsumer < cfgOut->maxConsumer){
              CfgOutConsumerListAdd(cfgOut, name, authp->retry, authp->priority,
                                  authp->request, authp->owner, authp->host);
            }
            else{
              CfgOutGroupPrint(cfgOut->group, 0,
                               "CfgOut> Cannot add %s to group %s,"
                               " too many consumers [%d > %d]\n",
                               name, cfgOut->group,
                               cfgOut->nbConsumer+1,
                               cfgOut->maxConsumer);

              /*--------- Here # max consumers is reached, refused connection */
              if(authp->request == CfgOutOnRequest){
                CmConnect cmConnect = CmConnectGetReference(name);

                if(cmConnect != (CmConnect)NULL){ CmConnectKill(cmConnect); }
              }
            }
          }
	}
      }
    }
  }

  CfgOutDone = CfgTrue;

  return(CmMessageBreak);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConnectionCheck ----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgBool_t CfgOutConnectionCheck(char *group, char *name, double timeout){
register CfgConsumer_t *cfgConsumer;

  cfgConsumer = CfgOutConsumerSearch(name, group);
  if(cfgConsumer == (CfgConsumer_t *)NULL){ return(CfgFalse); }

  if(cfgConsumer->sid != CFGOUT_NO_SOCKID){
    int sval;
    fd_set rfds, wfds, efds;
    struct timeval tmo, *tmop = (struct timeval *)NULL;

    FD_ZERO(&rfds);
    FD_ZERO(&wfds);
    FD_ZERO(&efds);
    FD_SET(cfgConsumer->sid, &rfds);
    FD_SET(cfgConsumer->sid, &wfds);
    FD_SET(cfgConsumer->sid, &efds);

    if(timeout >= 0.){
      tmo.tv_sec  = (int)timeout;
      tmo.tv_usec = (int)((timeout - (int)timeout) * 1e6);
      tmop        = &tmo;
    }

    sval = select(cfgConsumer->sid+1, &rfds, &wfds, &efds, tmop);
    if(sval > 0){
      CfgBool_t alive = CfgUnknown;

      if(FD_ISSET(cfgConsumer->sid, &rfds) ||
         FD_ISSET(cfgConsumer->sid, &efds)){
        if(cfgConsumer->request == CfgOutDynamic){
          cfgConsumer->connectDone = CfgFalse;
        }
        alive               = CfgFalse;
        cfgConsumer->status = CfgOutHostUnreachable;
      }
      else{
        if(FD_ISSET(cfgConsumer->sid, &wfds)){
          if(cfgConsumer->connectDone == CfgFalse){
            CmConnect cmConnect, cmServer;

            alive     = CfgFalse;
            cmServer  = CmConnectWhoAmI();
            cmConnect = CmConnectNewWithAddress(cmServer, name,
                                                cfgConsumer->port,
                                                cfgConsumer->host);
  
            if(cfgConsumer->authp->accepted == CfgTrue){
              if(cmConnect != (CmConnect)NULL){
                alive               = CfgTrue;
                cfgConsumer->status = CfgOutHostAlive;
              }
              cfgConsumer->connectDone = CfgTrue;
            }
          }
          else{
            alive               = CfgTrue;
            cfgConsumer->status = CfgOutHostAlive;
          }
        }
      }

      CfgOutConnectionClose(cfgConsumer);

      return(alive);
    }

    cfgConsumer->status = (cfgConsumer->alive == CfgTrue)? \
                          CfgOutConnectInProgress: \
                          cfgConsumer->status;

    return((cfgConsumer->alive == CfgTrue)? CfgUnknown: cfgConsumer->alive);
  }

  return(cfgConsumer->alive);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConnectionClose ----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutConnectionClose(CfgConsumer_t *cfgConsumer){

  if(cfgConsumer->sid != CFGOUT_NO_SOCKID){
    shutdown(cfgConsumer->sid, 2);
    close(cfgConsumer->sid);
    cfgConsumer->sid = CFGOUT_NO_SOCKID;
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConnectionHandler --------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void CfgOutConnectionHandler(CmConnect cmConnect){
CbfHEntry_t *itemp;
char *name, *owner, *host;

  name  = CmConnectGetName(cmConnect);
  owner = CmConnectGetOwner(cmConnect);
  host  = CmConnectGetHost(cmConnect);

  if((itemp = CbfHFind(CfgOutHashTable, name)) != (CbfHEntry_t *)NULL){
    CfgOutAuthentication_t *authp = (CfgOutAuthentication_t *)itemp->data;

    for(; authp != (CfgOutAuthentication_t *)NULL; authp = authp->next){
      if(((strcmp(authp->owner, owner) != 0) &&
          (strcmp(authp->owner, "<everyone>") != 0)) ||
         ((strcmp(authp->host, host) != 0) &&
          (strcmp(authp->host, "<any host>") != 0))){

        CfgOutGroupPrint(authp->cfgOut->group, 2,
                         "CfgOut> <%s> Connection from %s"
                         " -> %s@%s refused, permission denied.\n",
                         authp->cfgOut->group, name, owner, host);
 
        CmConnectKill(cmConnect);
      }
      else{
        CfgOutGroupPrint(authp->cfgOut->group, 2,
                         "CfgOut> <%s> Connection from %s -> %s@%s granted.\n",
                         authp->cfgOut->group, name, owner, host);

        authp->accepted = CfgTrue;
      }
    }
  }
  else{
    itemp = CbfHFind(CfgOutHashTable, "regular exp");
    if(itemp != (CbfHEntry_t *)NULL){
      CfgOutAuthentication_t *authp = (CfgOutAuthentication_t *)itemp->data;

      for(; authp != (CfgOutAuthentication_t *)NULL; authp = authp->next){
        if(RegExpMatch(authp->reg, name)){
          if(((strcmp(authp->owner, owner) != 0) &&
              (strcmp(authp->owner, "<everyone>") != 0)) ||
             ((strcmp(authp->host, host) != 0) &&
              (strcmp(authp->host, "<any host>") != 0))){

            CfgOutGroupPrint(authp->cfgOut->group, 2,
                             "CfgOut> <%s> (regExp) Connection from %s"
                             " -> %s@%s refused, permission denied.\n",
                             authp->cfgOut->group, name, owner, host);

            CmConnectKill(cmConnect);
          }
          else{
            CfgOutGroupPrint(authp->cfgOut->group, 2,
                             "CfgOut> <%s> (regExp) Connection from %s"
                             " -> %s@%s granted.\n",
                             authp->cfgOut->group, name, owner, host);

            authp->accepted = CfgTrue;
          }
        }
      }
    }
  }

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConnectionOpen -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutConnectionOpen(char *group, char *name){
struct sockaddr_in servAddr;
register struct hostent *hostp;
register CmConnect cmConnect;
register CfgConsumer_t *cfgConsumer;
register int sid, rc;
int on = 1;

  cfgConsumer = CfgOutConsumerSearch(name, group);
  if(cfgConsumer == (CfgConsumer_t *)NULL){ return(CFG_FAIL); }

  if(cfgConsumer->sid != CFGOUT_NO_SOCKID){
    return(CFGOUT_INPROGRESS);
  }

  cmConnect = CfgOutConsumerGetReference(name, group, CFGOUT_WATCHDOG);
  if(cmConnect == (CmConnect)NULL){
    cfgConsumer->status = CfgOutUnknownByNS;
    return(CFG_FAIL);
  }

  if((hostp = gethostbyname(cfgConsumer->host)) == (struct hostent *)NULL){
    perror("CfgOutConnectionOpen: gethostbyname()");
    return(CFG_FAIL);
  }

  memset((char *)&servAddr, 0, sizeof servAddr);
  memcpy((char *)&servAddr.sin_addr, (char *)hostp->h_addr, hostp->h_length);
  servAddr.sin_family = hostp->h_addrtype;
  servAddr.sin_port   = htons(cfgConsumer->port);

  if((sid = socket(AF_INET, SOCK_STREAM, 0)) == -1){
    perror("CfgOutConnectionOpen: socket()");
    return(CFG_FAIL);
  }

#ifndef __linux__
  {
    int option = 1;

    rc = setsockopt(sid, SOL_SOCKET, SO_REUSEPORT, (void *)&option,
                  sizeof option);
    if(rc == -1){
      perror("CfgOutConnectionOpen: setsockopt(SO_REUSEPORT)");
      return(CFG_FAIL);
    }
  }
#endif /* __linux__ */

  if(ioctl(sid, FIONBIO, &on) == -1){
    close(sid);
    perror("CfgOutConnectionOpen: ioctl(FIONBIO)");
    return(CFG_FAIL);
  }

  rc = connect(sid, (struct sockaddr *)&servAddr, sizeof servAddr);
  if(rc == -1){
    if(errno != EINPROGRESS){
      perror("CfgOutConnectionOpen: connect()");
      close(sid);
      return(CFG_FAIL);
    }
  }

  cfgConsumer->sid = sid;

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- _CfgOutConnectionOpen ----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void _CfgOutConnectionOpen(char *group, double timeout){
CfgOut_t *cfgOut;
struct timespec tmo;

  if((cfgOut = CfgOutSearchByGroup(group)) != (CfgOut_t *)NULL){
    int index;

    if(cfgOut->check == CfgFalse){ return; }

    for(index=0; index<CfgNameListSize(cfgOut->csList); index++){
      CfgConsumer_t *cfgConsumer;

      cfgConsumer = CfgNameListVData(cfgOut->csList, index);
      if(cfgConsumer != (CfgConsumer_t *)NULL){
        if(CfgOutConnectionOpen(group, cfgConsumer->name) == CFG_FAIL){
          cfgConsumer->alive = CfgFalse;
        }
      }
    }
  }

  /*------------------------------------------------ Wait on connection reply */
  tmo.tv_sec  = (long)timeout;
  tmo.tv_nsec = (long)((timeout - (long)timeout) * 1e9);
  nanosleep(&tmo, (struct timespec *)NULL);

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConsumerAdd --------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutConsumerAdd(void *arg, char *name, int retry, int priority,
                             char *group, CfgOutRequest_t request, char *owner,
                             char *host){
register int rtn;

  rtn = CfgOutGroupConsumerAdd(group, name, retry, priority, request, owner,
                               host);

  return(rtn);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConsumerGetNames ---------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutConsumerGetNames(CfgOut_t *cfgOut, char *names){
volatile int rtn = CFG_OK;

  if((cfgOut == (CfgOut_t *)NULL) || (names == (char *)NULL)){
    return(CFG_FAIL);
  }

  if(strpbrk(names, CFGOUT_MAGIC_CHAR) != (char *)NULL){
    struct sigaction oact;
    CmMessage message = CmMessageNew();
    char *nameserver  = CmConnectGetName(CmConnectGetNameServer());
         
    CmMessageInstallHandler(CfgOutCmNSNames, "NSNames");
         
    CfgOutCurrentPointer(cfgOut);
    CfgOutDone = CfgFalse;
  
    CmMessageSetType(message, "NSGetNames");
    CmMessagePutText(message, names);

    if(setjmp(CfgOutEnv) == 0){
      CfgOutEnvp = &CfgOutEnv;
      /*---------------------------------------------------- Fire on watchdog */
      CfgOutActiveWatchdog(CFGOUT_WATCHDOG, &oact);
      CmMessageSend(message, nameserver);
      /*--------------------------------------------------- Fire off watchdog */
    }
    else{
      CfgOutTimeout = CfgFalse;
      CfgOutEnvp    = (jmp_buf *)NULL;
      CfgOutDone    = CfgTrue;
      rtn           = CFG_FAIL;
    }

    CfgOutReleaseWatchdog(&oact);
                                     
    CmMessageDelete(message);
    while(CfgOutDone == CfgFalse){ CmMessageWait(); } 

    CfgOutCurrentPointer((CfgOut_t *)NULL);
  
    CmMessageUninstallHandler("NSNames");
  }

  if(rtn == CFG_FAIL){ errno = ETIMEDOUT; }

  return(rtn);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConsumerGetReference -----------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CmConnect CfgOutConsumerGetReference(char *name, char *group,
                                            double timeout){
register CmConnect cmConnect;
register CfgConsumer_t *cfgConsumer = CfgOutConsumerSearch(name, group);

  if(cfgConsumer == (CfgConsumer_t *)NULL){ return((CmConnect)NULL); }

  if((cmConnect = CmConnectGetReference(name)) == (CmConnect)NULL){
    CfgOut_t *cfgOut = cfgConsumer->cfgOut;
    time_t now = time((time_t *)NULL);
    double period = difftime(now, cfgConsumer->start);

    if((period >= 0.) && (period < (double)cfgOut->nsPeriod)){
      return((CmConnect)NULL);
    }
    else{
      cfgConsumer->start = (period >= (double)cfgOut->nsPeriod)? \
                           now: cfgConsumer->start;
    }

    if(!CfgOutConsumerIsServer(name)){
      struct sigaction oact;
      CmMessageHandler oldHandler;

      cfgConsumer->connectDone = CfgFalse;

      oldHandler = CmMessageInstallHandler(CfgOutCmNSAddress, "NSAddress");
      /*-------------------------------------------------- Fire on watchdog */
      CfgOutActiveWatchdog(timeout, &oact);
      cmConnect = CmConnectNew(name);
      /*------------------------------------------------- Fire off watchdog */
      CfgOutReleaseWatchdog(&oact);
      CmMessageInstallHandler(oldHandler, "NSAddress");
    }
  }

  if(cmConnect != (CmConnect)NULL){
    cfgConsumer->port = CmConnectGetPort(cmConnect);
    strcpy(cfgConsumer->host, CmConnectGetHost(cmConnect));
  }

  if(CfgOutTimeout){
    CfgOutTimeout = CfgFalse;
    errno         = ETIMEDOUT;
  }

  return(cmConnect);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConsumerFree -------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgConsumer_t *CfgOutConsumerFree(CfgConsumer_t *cfgConsumer){
char key[256];
CbfHEntry_t *itemp;
register int nbGroups = 0;

  if(cfgConsumer == (CfgConsumer_t *)NULL){
    fprintf(stderr, "CfgOutConsumerFree: invalid pointer argument\n");
    return(cfgConsumer);
  }
  
  if(cfgConsumer->request != CfgOutOnRequest){
    /*---------------------------------------------------- Update group count */
    itemp = CbfHFind(CfgOutHashTable, cfgConsumer->name);
    if(itemp != (CbfHEntry_t *)NULL){
      CfgOutAuthentication_t *authp = (CfgOutAuthentication_t *)itemp->data;

      nbGroups = authp->nbGroups - 1;

      for(; authp != (CfgOutAuthentication_t *)NULL; authp = authp->next){
	authp->nbGroups = nbGroups;
      }
    }
  }

  /*------------------------------------------- Remove authentification entry */
  itemp = CbfHFind(CfgOutHashTable, cfgConsumer->name);
  if(itemp != (CbfHEntry_t *)NULL){
    if(itemp->data == (void *)cfgConsumer->authp){
      CbfHEntryRemove(itemp);
      if(cfgConsumer->authp->next != (CfgOutAuthentication_t *)NULL){
        CbfHEnter(CfgOutHashTable, cfgConsumer->name,
                  (void *)cfgConsumer->authp->next);
      }
    }
    else{
      CfgOutAuthentication_t *authp = (CfgOutAuthentication_t *)itemp->data;

      for(; authp != (CfgOutAuthentication_t *)NULL; authp = authp->next){
        if(authp->next == cfgConsumer->authp){
          authp->next = cfgConsumer->authp->next;
          break;
        }
      }
    }
  }

  if(cfgConsumer->authp) { free(cfgConsumer->authp); cfgConsumer->authp = NULL; }
  sprintf(key, "%p", cfgConsumer->message);
  CbfHEntryRemove(CbfHFind(CfgOutHashTable, key));
  CmMessageIteratorDelete(cfgConsumer->iterator);
  CmMessageDelete(cfgConsumer->message);
  if(nbGroups == 0){ CmConnectKill(CmConnectGetReference(cfgConsumer->name)); }

  cfgConsumer->cfgOut->nbConsumer--;

  memset((char *)cfgConsumer, 0, sizeof(CfgConsumer_t));
  if(cfgConsumer){ free(cfgConsumer); cfgConsumer = NULL; }

  return((CfgConsumer_t *)NULL);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConsumerIsAlive ----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgBool_t CfgOutConsumerIsAlive(char *name, char *group,
                                       CfgBool_t useNameServer){
register CfgBool_t     alive = CfgFalse;
register CfgConsumer_t *cfgConsumer;
struct sigaction       oact;

  cfgConsumer = CfgOutConsumerSearch(name, group);
  if(cfgConsumer != (CfgConsumer_t *)NULL){
    CfgOut_t *cfgOut = cfgConsumer->cfgOut;

    if(cfgOut->check == CfgFalse){
      if(!CfgOutConsumerIsServer(name)){
        if(useNameServer == CfgTrue){
          CmConnect cmConnect;

          if((cmConnect = CmConnectGetReference(name)) == (CmConnect)NULL){
            time_t now = time((time_t *)NULL);
            double period = difftime(now, cfgConsumer->start);

            if((period >= 0.) && (period < (double)cfgOut->nsPeriod)){
              CfgOutGroupPrint(group, 4,
                               "CfgOut> <%s> Dont't ask to the Nameserver"
                               " about %s [%g s/%d s]\n",
                               cfgOut->group, name, period, cfgOut->nsPeriod);
              return(cfgConsumer->alive);
            }
            else{
              CfgOutGroupPrint(group, 3,
                               "CfgOut> <%s> Try to ask to the Nameserver"
                               " about %s [%g s/%d s]\n",
                               cfgOut->group, name, period, cfgOut->nsPeriod);

              cfgConsumer->start = (period >= (double)cfgOut->nsPeriod)? \
                                   now: cfgConsumer->start;
            }
          }

          CfgOutTimeout = CfgFalse;

          /*------------------------------------------------ Fire on watchdog */
          CfgOutActiveWatchdog(CFGOUT_WATCHDOG, &oact);
          alive = CmMessageAreYouThere(name);
          if(CfgOutTimeout == CfgFalse){
            cfgConsumer->status = alive ? CfgOutHostAlive: CfgOutUnknownByNS;
          }
          else{
            time_t now = time((time_t *)NULL);

            cfgConsumer->status = CfgOutHostUnreachable;
            cfgConsumer->start  = now;

            CfgOutGroupPrint(group, 3,
                             "CfgOut> <%s> Connection to %s failed, "
                             "timeout expired.\n", cfgOut->group, name);
          }

          /*----------------------------------------------- Fire off watchdog */
          CfgOutReleaseWatchdog(&oact);
        }
        else{
          if(CmConnectGetReference(name) == (CmConnect)NULL){
            cfgConsumer->status = CfgOutUnknownByNS;
          }
          else{ alive = CfgTrue; }
        }
      }
      else{ alive = CfgTrue; }
    }
    else{
      alive = CfgOutConnectionCheck(group, name, 0.);
    }
  }

  return(alive);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConsumerIsServer ---------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgBool_t CfgOutConsumerIsServer(char *name){
register char *server = CmConnectGetName(CmConnectWhoAmI());

  return(!strcmp(name, server));
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConsumerListAdd ----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutConsumerListAdd(CfgOut_t *cfgOut, char *name, int retry,
                                 int priority, CfgOutRequest_t request,
                                 char *owner, char *host){
CfgConsumer_t *cfgConsumer;
CfgBool_t update;
register int index;
CfgNameList_t *nlistp;

  if(request == CfgOutOnRequest){ nlistp = cfgOut->csOnReqList; }
  else{ nlistp = cfgOut->csList; }

  if((index = CfgNameListAdd(nlistp, name)) == CFG_FAIL){ return(CFG_FAIL); }

  update = CfgFalse;

  if((cfgConsumer = CfgNameListVData(nlistp, index)) == (CfgConsumer_t *)NULL){
    char key[256];
    int nbGroups = 0;
    CbfHEntry_t *itemp;
    CfgOutAuthentication_t *authp;

    cfgConsumer = (CfgConsumer_t *)calloc(1, sizeof(CfgConsumer_t));
    if(cfgConsumer == (CfgConsumer_t *)NULL){
      perror("CfgOutConsumerListAdd: calloc(cfgConsumer)");
      return(CFG_FAIL);
    }

    cfgConsumer->sid     = CFGOUT_NO_SOCKID;
    cfgConsumer->alive   = CfgUnknown;
    cfgConsumer->status  = CfgOutNone;
    cfgConsumer->cfgOut  = cfgOut;
    cfgConsumer->request = request;
    cfgConsumer->period  = 1;
    cfgConsumer->message = CmMessageNew();
    cfgConsumer->status  = CfgOutMsgNew;
    strcpy(cfgConsumer->name, name);

    sprintf(key, "%p", cfgConsumer->message);
    CbfHEnter(CfgOutHashTable, key, (void *)cfgConsumer);

    authp = (CfgOutAuthentication_t *)calloc(1, sizeof(CfgOutAuthentication_t));
    if(authp == (CfgOutAuthentication_t *)NULL){
      perror("CfgOutConsumerListAdd: calloc(authp)");
      cfgConsumer = CfgOutConsumerFree(cfgConsumer);
      return(CFG_FAIL);
    }

    cfgConsumer->authp = authp;
    authp->cfgOut      = cfgOut;
    authp->request     = request;
    strcpy(authp->owner, owner);
    strcpy(authp->host, host);

    if((itemp = CbfHFind(CfgOutHashTable, name)) == (CbfHEntry_t *)NULL){
      CbfHEnter(CfgOutHashTable, name, (void *)authp);
    }
    else{
      CfgOutAuthentication_t *This = (CfgOutAuthentication_t *)itemp->data;

      nbGroups    = This->nbGroups;
      authp->next = This;
      itemp->data = (void *)authp;
    }

    if(cfgConsumer->request != CfgOutOnRequest){
      for(; authp != (CfgOutAuthentication_t *)NULL; authp = authp->next){
	authp->nbGroups = nbGroups + 1;
      }
    }

    if(CfgNameListSetData(nlistp, index, cfgConsumer) == CFG_FAIL){
      cfgConsumer = CfgOutConsumerFree(cfgConsumer);
      return(CFG_FAIL);
    }

    cfgOut->nbConsumer++;
  }
  else{
    update = CfgTrue;
  }

  cfgConsumer->retry           = retry;
  cfgConsumer->authp->retry    = retry;
  cfgConsumer->authp->priority = priority;
  cfgConsumer->remove          = CfgFalse;

  if(cfgConsumer->request == CfgOutDynamic){
    cfgConsumer->connectDone = CfgTrue;
  }

  if(CfgNameListSetInt(nlistp, index, priority)  == CFG_FAIL){
    return(CFG_FAIL);
  }

  CfgOutGroupPrint(cfgOut->group, 0,
                   "CfgOut> %s %s [%d, %d, %s, %s, %s, %s].\n",
                   update? "<Update>": "<Add>",
                   name,
                   cfgConsumer->retry,
                   priority,
                   cfgOut->group,
                   (cfgConsumer->request == CfgOutStatic)? "static": \
                   (cfgConsumer->request == CfgOutDynamic)? "dynamic": \
                   "onRequest",
                   cfgConsumer->authp->owner,
                   cfgConsumer->authp->host );

  if(cfgOut->debug > 3){ CfgNameListPrint(nlistp); }

  return(update? CFG_CONTINUE: CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConsumerSearch -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgConsumer_t *CfgOutConsumerSearch(char *name, char *group){
register CfgOut_t *cfgOut;

  if(name == (char *)NULL){ return((CfgConsumer_t *)NULL); }

  if((cfgOut = CfgOutSearchByGroup(group)) != (CfgOut_t *)NULL){
    int index;

    if((index = CfgNameListSearchName(cfgOut->csList, name)) != CFG_FAIL){
      return(CfgNameListVData(cfgOut->csList, index));  
    }
  }

  return((CfgConsumer_t *)NULL);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConsumerSearchByMsg ------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgConsumer_t *CfgOutConsumerSearchByMsg(CmMessage message){
CbfHEntry_t *itemp;
char        key[16];

  if(message == (CmMessage)NULL){  return((CfgConsumer_t *)NULL); }

  sprintf(key, "%p", message);
  itemp = CbfHFind(CfgOutHashTable, key);
  if(itemp == (CbfHEntry_t *)NULL){ return((CfgConsumer_t *)NULL); }

  return((CfgConsumer_t *)itemp->data);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutConsumerState ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void CfgOutConsumerState(CfgConsumer_t *cfgConsumer, char *name,
                                char *group, CfgBool_t useNameServer){

  int state = CmMessageGetState(cfgConsumer->message);
  cfgConsumer->alive = CfgOutConsumerIsAlive(name, group, useNameServer);

  if(cfgConsumer->status == CfgOutHostAlive){
    cfgConsumer->status = (state == CmMessageIsNew)? \
                          cfgConsumer->status: CfgOutMsgSending;
  }

  if(state == CmMessageIsNew){ cfgConsumer->complete = CfgFalse; }

  if(cfgConsumer->alive == CfgFalse){
    CmMessageReset(cfgConsumer->message);
    /*----------------- Remove consumer from list if it was added dynamically */
    cfgConsumer->remove = (cfgConsumer->request == CfgOutDynamic)? \
                          CfgTrue: CfgFalse;
  }

  CfgOutGroupPrint(group, 3,
                   "CfgOut> [%s] message state %s (%s), consumer is %s.\n",
                   name,
                   CfgOutCmState[state],
                   cfgConsumer->complete? "complete": "free",
                   cfgConsumer->alive? \
                   ((cfgConsumer->alive == CfgTrue)? "alive": \
                                                     "unreachable"): "dead");

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCsmDisconnectionHandler --------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void CfgOutCsmDisconnectionHandler(CmConnect cmConnect){

  if(cmConnect != (CmConnect)NULL){
    char *name;
    char key[128];

    if((name = CmConnectGetName(cmConnect)) != (char *)NULL){
      CbfHEntry_t *itemp;

      sprintf(key, "#%s#", name);

      if((itemp =  CbfHFind(CfgOutHashTable, key)) != (CbfHEntry_t *)NULL){
        itemp->data = (void *)CFGOUT_CON_DENIED;
      }
    }
  }

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCurrentPointer -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgOut_t *CfgOutCurrentPointer(CfgOut_t *cfgOut){
static CfgOut_t *currentp = (CfgOut_t *)NULL;

  return(currentp = (cfgOut? cfgOut: currentp));
}

/*----------------------------------------------------------------------------*/
/*- CfgOutDebugSet -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutDebugSet(void *arg, int debug, char *group){

  CfgOutGroupDebugSet(group, debug);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutDisconnectionHandler -----------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void CfgOutDisconnectionHandler(CmConnect cmConnect){
CbfHEntry_t *itemp;
char *name, *owner, *host;
/* int port; */

  name  = CmConnectGetName(cmConnect);
  owner = CmConnectGetOwner(cmConnect);
  host  = CmConnectGetHost(cmConnect);
/*   port  = CmConnectGetPort(cmConnect); */
 
  
  if((itemp = CbfHFind(CfgOutHashTable, name)) != (CbfHEntry_t *)NULL){
    CfgOutAuthentication_t *authp = (CfgOutAuthentication_t *)itemp->data;

    for(; authp != (CfgOutAuthentication_t *)NULL; authp = authp->next){
      if(((strcmp(authp->owner, owner) == 0) ||
	  (strcmp(authp->owner, "<everyone>") == 0)) &&
         ((strcmp(authp->host, host) == 0) ||
	  (strcmp(authp->host, "<any host>") == 0))){
        authp->accepted = CfgFalse;
      }
    }
  }
  
  return;
}

/*----------------------------------------------------------------------------*/
/*- _CfgOutFree --------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgOut_t *_CfgOutFree(CfgOut_t *cfgOut){

  if(cfgOut == (CfgOut_t *)NULL){ return(cfgOut); }

  if(!CbfCleanFind("_CfgOutFree", (void *)cfgOut)){ return((CfgOut_t *)NULL); }

  CbfHEntryRemove(CbfHFind(CfgOutHashTable, cfgOut->group));
  cfgOut->csList      = CfgNameListFree(cfgOut->csList);
  cfgOut->csOnReqList = CfgNameListFree(cfgOut->csOnReqList);
  cfgOut->gpList      = CfgNameListFree(cfgOut->gpList);
  cfgOut->dmList      = CfgNameListFree(cfgOut->dmList);
  memset((char *)cfgOut, 0, sizeof(CfgOut_t));
  CbfCleanDone("_CfgOutFree", (void *)cfgOut);
  if(cfgOut) { free(cfgOut); cfgOut = NULL; }

  return((CfgOut_t *)NULL);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGetOut -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgOut_t *CfgOutGetOut(char *group){
CfgOut_t *cfgOut;

  if((cfgOut = CfgOutSearchByGroup(group)) == (CfgOut_t *)NULL){ 
    int rtn;

    rtn = CfgOutGroupNew(CfgOutGroupDefaultGet(), (CfgParse_t *)NULL,
                         CFGOUT_MAX_CONSUMER, CFGOUT_MAX_GROUP,
                         CFGOUT_THRESHOLD, CFGOUT_DEBUG);

    if(rtn == CFG_OK){ cfgOut = CfgOutSearchByGroup(group); }
  }

  return(cfgOut);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupPrint ---------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutGroupPrint(char *group, int level, char *format, ...){
va_list args;
register int rtn = 0;
CfgOut_t *cfgOut;

  if((cfgOut = CfgOutSearchByGroup(group)) == (CfgOut_t *)NULL){ return(0); }

  if(cfgOut->debug >= level){
    if(CfgOutPrinter != (CfgOutPrinter_t)NULL){
      va_start(args, format);
      rtn = CfgOutPrinter(format, args);
      va_end(args);
    }
  }

  return(rtn);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutHFree --------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CbfH_t *CfgOutHFree(CbfH_t *hTable){
CbfHEntry_t *itemp;

  if(!CbfCleanFind("CfgOutHFree", (void *)hTable)){ return((void *)NULL); }

  if((itemp = CbfHFind(CfgOutHashTable, "regular exp")) != (CbfHEntry_t *)NULL){
    CfgOutAuthentication_t *authp;
  
    for(authp = (CfgOutAuthentication_t *)itemp->data;
        authp != (CfgOutAuthentication_t *)NULL; authp = authp->next){
      RegExpDelete(authp->reg);
      free(authp);
    }

    CbfHEntryRemove(itemp);
  }

  CbfCleanDone("CfgOutHFree", (void *)hTable);

  return(CfgOutHashTable = CbfHFree(hTable));
}

/*----------------------------------------------------------------------------*/
/*- CfgOutMaxConsumerSet -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutMaxConsumerSet(void *arg, int maxConsumer, char *group){
CfgOut_t *cfgOut;

  if((cfgOut = CfgOutGetOut(group)) != (CfgOut_t *)NULL){ 
    cfgOut->maxConsumer = maxConsumer;
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutMaxGroupSet --------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutMaxGroupSet(void *arg, int maxGroup, char *domain){
CfgOut_t *cfgOut;

  if((cfgOut = CfgOutGetOut(domain)) != (CfgOut_t *)NULL){ 
    cfgOut->maxGroup = maxGroup;
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutNsPeriodSet --------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutNsPeriodSet(void *arg, int period, char *group){
  return(CfgOutGroupNsPeriodSet(group, period));
}

/*----------------------------------------------------------------------------*/
/*- CfgOutPostHandler --------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CmMessageStatus CfgOutPostHandler(CmMessage message, char *sender,
                                         char *server){
register CfgConsumer_t *cfgConsumer;

  CmMessageReset(message);

  cfgConsumer = CfgOutConsumerSearchByMsg(message);

  if(cfgConsumer != (CfgConsumer_t *)NULL){

    cfgConsumer->status = CfgOutMsgNew;

  }

  return(CmMessageOk);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutPrintMsgError ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void CfgOutPrintMsgError(CfgConsumer_t *cfgConsumer){

  switch(cfgConsumer->status){
    case CfgOutUnknownByNS:
      CfgOutGroupPrint(cfgConsumer->cfgOut->group, 2, "CfgOut> "
                       "Consumer %s unknown by the `Cm' nameServer [%d/%d].\n",
                       cfgConsumer->name,
                       cfgConsumer->counter,
                       cfgConsumer->retry);
      break;
    case CfgOutConnectInProgress:
      CfgOutGroupPrint(cfgConsumer->cfgOut->group, 2, "CfgOut> "
                       "Cannot send message to %s, "
                       "connection in progress [%d/%d].\n",
                       cfgConsumer->name,
                       cfgConsumer->counter,
                       cfgConsumer->retry);
      break;
    case CfgOutHostUnreachable:
      CfgOutGroupPrint(cfgConsumer->cfgOut->group, 2, "CfgOut> "
                       "Cannot send message to %s, host unreachable [%d/%d].\n",
                       cfgConsumer->name,
                       cfgConsumer->counter,
                       cfgConsumer->retry);
      break;
    case CfgOutMsgSending:
      CfgOutGroupPrint(cfgConsumer->cfgOut->group, 2,
                       "CfgOut> Message to %s is sending [%d/%d].\n",
                       cfgConsumer->name,
                       cfgConsumer->counter,
                       cfgConsumer->retry);
      break;
    default:
      break;
  }

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutReleaseWatchdog ----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutReleaseWatchdog(struct sigaction *act){
struct itimerval itimer;
sigset_t mask;

  /*------------------------------------------------------- Mask signal ALARM */
  sigemptyset(&mask);
  sigaddset(&mask, SIGALRM);
  sigprocmask(SIG_SETMASK, &mask, (sigset_t *)NULL);

  /*-------------------------------------------------------------- Stop Timer */
  memset((char *)&itimer, 0, sizeof(struct itimerval));

  if(setitimer(ITIMER_REAL, &itimer, (struct itimerval *)NULL) == -1){
    fprintf(stderr, "CfgOutReleaseWatchdog: setitimer: %s\n", strerror(errno));
    return(CFG_FAIL);
  }

  /*----------------------------------------------- Restore old signal action */
  if(sigaction(SIGALRM, act, (struct sigaction *)NULL) == -1) {
    fprintf(stderr, "CfgOutReleaseWatchdog: sigaction: %s\n", strerror(errno));
    return(CFG_FAIL);
  }

  /*----------------------------------------------------- Unmask signal ALARM */
  sigdelset(&mask, SIGALRM);
  sigprocmask(SIG_SETMASK, &mask, (sigset_t *)NULL);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutRemoveEntryList ----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutRemoveEntryList(CfgOut_t *cfgOutGrp, char *group, int index){
CfgOut_t *cfgOutDm = CfgNameListVData(cfgOutGrp->dmList, index);

  /*----------------------------------- Remove `Group' entry from Domain list */
  CfgNameListNRemove(cfgOutDm->gpList,
                     CfgNameListSearchName(cfgOutDm->gpList, group)); 
  /*----------------------------------- Remove `Domain' entry from Group list */
  CfgNameListNRemove(cfgOutGrp->dmList, index);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutStatSet ------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutStatSet(void *arg, CfgBool_t statistic, char *group){
CfgOut_t *cfgOut;

  if((cfgOut = CfgOutGetOut(group)) != (CfgOut_t *)NULL){ 
    cfgOut->statistic = statistic;
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutSearchByGroup ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgOut_t *CfgOutSearchByGroup(char *group){
CbfHEntry_t *itemp;

  itemp = CbfHFind(CfgOutHashTable, group? group: CfgOutGroupDefaultGet());
  if(itemp == (CbfHEntry_t *)NULL){ return((CfgOut_t *)NULL); }

  return((CfgOut_t *)itemp->data);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutThresholdSet -------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgOutThresholdSet(void *arg, int threshold, char *group){

  CfgOutGroupThresholdSet(group, threshold);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutTagServer ----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void CfgOutTagServer(char *server){
CbfHEntry_t *itemp;          
char key[128];

  sprintf(key, "#%s#", server);

  if((itemp = CbfHFind(CfgOutHashTable, key)) != (CbfHEntry_t *)NULL){
    itemp->data = (void *)CFGOUT_CON_GRANTED;
  }
    
  return;
}

/******************************************************************************/
/*                            PUBLIC FUNCTIONS                                */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*- CfgOutCmEnd --------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
void CfgOutCmEnd(void){

  CmMessageUninstallHandler(CFGOUT_CSM_ADD_TYPE);
  CmMessageUninstallHandler(CFGOUT_CSM_RM_TYPE);
  CmMessageUninstallHandler(CFGOUT_GRP_ADD_TYPE);
  CmMessageUninstallHandler(CFGOUT_GRP_RM_TYPE);

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCmInit -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
void CfgOutCmInit(void){

  CmMessageInstallHandler(CfgOutCmCsmAdd,      CFGOUT_CSM_ADD_TYPE);
  CmMessageInstallHandler(CfgOutCmCsmRemove,   CFGOUT_CSM_RM_TYPE);
  CmMessageInstallHandler(CfgOutCmGroupAdd,    CFGOUT_GRP_ADD_TYPE);
  CmMessageInstallHandler(CfgOutCmGroupRemove, CFGOUT_GRP_RM_TYPE);

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutDomainGroupAdd -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutDomainGroupAdd(char *domain, char *group){
CfgOut_t *cfgOutDm, *cfgOutGrp;
register int index;

  group  = group? group: CfgOutGroupDefaultGet();
  domain = domain? domain: CfgOutGroupDefaultGet();

  if(strcmp(group, domain) == 0){ return(CFG_OK); }

  if((cfgOutDm = CfgOutSearchByGroup(domain)) == (CfgOut_t *)NULL){
    return(CFG_FAIL);
  }

  if(CfgNameListSize(cfgOutDm->gpList) >= cfgOutDm->maxGroup){ return(CFG_OK); }

  if((cfgOutGrp = CfgOutSearchByGroup(group)) == (CfgOut_t *)NULL){
    int rtn;

    CfgOutGroupPrint(domain, 0, "CfgOut> domain %s - group %s [%d, %d, %d].\n",
                     cfgOutDm->group,
                     group,
                     cfgOutDm->maxConsumer,
                     cfgOutDm->threshold,
                     cfgOutDm->debug);

    rtn = CfgOutGroupNew(group, (CfgParse_t *)NULL, cfgOutDm->maxConsumer,
                         cfgOutDm->maxGroup, cfgOutDm->threshold,
                         cfgOutDm->debug);

    if(rtn == CFG_FAIL){ return(CFG_FAIL); }

    cfgOutGrp = CfgOutSearchByGroup(group);
  }

  /*-------------------------------------- Add `Group' entry into Domain list */
  if((index = CfgNameListAdd(cfgOutDm->gpList, group)) != CFG_FAIL){
    if(CfgNameListSetData(cfgOutDm->gpList, index, cfgOutGrp) == CFG_FAIL){
      return(CFG_FAIL);
    }
  }

  /*-------------------------------------- Add `Domain' entry into Group list */
  if((index = CfgNameListAdd(cfgOutGrp->dmList, domain)) != CFG_FAIL){
    if(CfgNameListSetData(cfgOutGrp->dmList, index, cfgOutDm) == CFG_FAIL){
      return(CFG_FAIL);
    }
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutDomainGroupRemove --------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutDomainGroupRemove(char *domain, char *group){
CfgOut_t *cfgOutGrp;
register int index;

  group  = group? group: CfgOutGroupDefaultGet();
  domain = domain? domain: CfgOutGroupDefaultGet();

  if(strcmp(group, domain) == 0){ return(CFG_OK); }

  if((cfgOutGrp = CfgOutSearchByGroup(group)) == (CfgOut_t *)NULL){
    return(CFG_FAIL);
  }

  domain = domain? domain: CfgOutGroupDefaultGet();

  if(strcmp(domain, "all") == 0){
    index = 0;

    while(index < CfgNameListSize(cfgOutGrp->dmList)){
      CfgOutRemoveEntryList(cfgOutGrp, group, index);
      index++;
    }
  }
  else{
    if((index = CfgNameListSearchName(cfgOutGrp->dmList, domain)) != CFG_FAIL){
      CfgOutRemoveEntryList(cfgOutGrp, group, index);
    }   
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutInstallPrinter -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgOutPrinter_t CfgOutInstallPrinter(CfgOutPrinter_t printer){
CfgOutPrinter_t oprinter = CfgOutPrinter;

  CfgOutPrinter = printer;

  return(oprinter);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupConsumerAdd ---------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupConsumerAdd(char *group, char *name, int retry, int priority,
                           CfgOutRequest_t request, char *owner, char *host){
CfgOut_t *cfgOut;
CfgConsumer_t *cfgConsumer;
register int rtn = CFG_OK;

  if(name == (char *)NULL){ return(CFG_FAIL); }

  if(host != (char *)NULL){
    struct hostent *hostp;

    sethostent(1);
    if((hostp = gethostbyname(host)) == (struct hostent *)NULL){
      fprintf(stderr, "CfgOutGroupConsumerAdd: unknown host %s\n", host);
      return(CFG_OK);
    }
    host = hostp->h_name;
    endhostent();
  }
  else{
    host = "<any host>";
  }

  owner = owner? owner: "<everyone>";

  if((cfgOut = CfgOutSearchByGroup(group)) == (CfgOut_t *)NULL){
    int rtn;

    /*--------------------------------------------- Search default parameters */
    if((cfgOut = CfgOutGetOut(CfgOutGroupDefaultGet())) == (CfgOut_t *)NULL){
      return(CFG_FAIL);
    }

    CfgOutGroupPrint(group, 0, "CfgOut> default paramters [%d, %d, %d]"
                     " (reference group: %s).\n",
                     cfgOut->maxConsumer, cfgOut->threshold, cfgOut->debug,
                     CfgOutGroupDefaultGet());

    rtn = CfgOutGroupNew(group, (CfgParse_t *)NULL, cfgOut->maxConsumer,
                         cfgOut->maxGroup, cfgOut->threshold, cfgOut->debug);
    if(rtn == CFG_FAIL){ return(CFG_FAIL); }

    cfgOut = CfgOutSearchByGroup(group);
  }

  /*------------------------------- Check if static consumer is already exist */
  cfgConsumer = CfgOutConsumerSearch(name, group);
  if((cfgConsumer != (CfgConsumer_t *)NULL) &&
     (cfgConsumer->request == CfgOutStatic) && (request == CfgOutDynamic)){

    CmConnect cmConnect = CmConnectGetReference(name);

    if(cmConnect != (CmConnect)NULL){
      CfgOutGroupPrint(group, 0,
                       "CfgOut> `dynamic' consumer %s has been rejected,\n"
                       "        `static' consumer %s already exist.\n",
                       name, name);
      CmConnectKill(cmConnect);
    }

    return(CFG_CONTINUE);
  }

  if(cfgOut->nbConsumer < cfgOut->maxConsumer){
    int index;

    priority = (priority > CFGOUT_MAX_PRIO)? CFGOUT_MAX_PRIO: priority;
    priority = (priority < CFGOUT_MIN_PRIO)? CFGOUT_MIN_PRIO: priority;

    if(strpbrk(name, CFGOUT_MAGIC_CHAR) != (char *)NULL){
      CfgOutAuthentication_t *authp;
      CbfHEntry_t *itemp;

      authp = calloc(1, sizeof(CfgOutAuthentication_t));
      if(authp == (CfgOutAuthentication_t *)NULL){ return(CFG_FAIL); }
       
      authp->cfgOut   = cfgOut;
      authp->retry    = retry;
      authp->priority = priority;
      authp->request  = request;
      authp->reg      = RegExpNew(name);
      strcpy(authp->owner, owner);
      strcpy(authp->host, host);

      itemp = CbfHFind(CfgOutHashTable, "regular exp");
      if(itemp == (CbfHEntry_t *)NULL){
        CbfHEnter(CfgOutHashTable, "regular exp", authp);
      }
      else{
        CfgOutAuthentication_t *This = (CfgOutAuthentication_t *)itemp->data;

        authp->next = This; 
        itemp->data = (void *)authp;
      }

      if(request == CfgOutStatic){
        if((index = CfgNameListAdd(cfgOut->csList, name)) != CFG_FAIL){
          CfgNameListSetInt(cfgOut->csList, index, priority);
        }
      }
    }
    else{
      rtn = CfgOutConsumerListAdd(cfgOut, name, retry, priority, request,
                                  owner, host);
    }
  }
  else{
    CfgOutGroupPrint(group, 0, "CfgOut> Cannot add %s to group %s,"
                     " too many consumers [%d > %d]\n",
                     name, cfgOut->group, cfgOut->nbConsumer+1,
                     cfgOut->maxConsumer);
    /*------------------- Here # max consumers is reached, refused connection */
    if(request == CfgOutDynamic){
      CmConnect cmConnect = CmConnectGetReference(name);

      if(cmConnect != (CmConnect)NULL){ CmConnectKill(cmConnect); }
    }
    rtn = CFG_CONTINUE;
  }

  return(rtn);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupConsumerDataAdd -----------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupConsumerDataAdd(char *group, char *name, Ulong data){
register CfgOut_t *cfgOut;
register int index;

  if((cfgOut = CfgOutSearchByGroup(group)) == (CfgOut_t *)NULL){
    CfgOutGroupPrint(group, 0,
                     "CfgOut> CfgOutGroupConsumerDataAdd: "
                     "group %s unknown.\n",
                     group? group: CfgOutGroupDefaultGet());

    return(CFG_FAIL);
  }

  if((index = CfgNameListSearchName(cfgOut->csList, name)) == CFG_FAIL){
    CfgOutGroupPrint(group, 0,
                     "CfgOut> CfgOutGroupConsumerDataAdd: "
                     "Consumer %s is not in group %s.\n",
                     name, group? group: CfgOutGroupDefaultGet());

    return(CFG_FAIL);
  }

  CfgNameListSetDouble(cfgOut->csList, index, (double)data);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupConsumerPost --------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupConsumerPost(char *group, char *name, CfgOutBuildMsg_t buildMsg,
                            void *arg){
register CfgConsumer_t *cfgConsumer;
register CfgOut_t *cfgOut;
register int priority;

  if(buildMsg == (CfgOutBuildMsg_t)NULL){
    CfgOutGroupPrint(group, 0,
                     "CfgOut> CfgOutGroupConsumerPost: "
                     "Invalid argument \"null\" pointer passed.\n");

    return(CFG_FAIL);
  }

  cfgConsumer = CfgOutConsumerSearch(name, group);
  if(cfgConsumer == (CfgConsumer_t *)NULL){
    CfgOutGroupPrint(group, 0,
                     "CfgOut> CfgOutGroupConsumerPost: "
                     "Consumer %s is not in group %s.\n",
                     name, group? group: CfgOutGroupDefaultGet());

    return(CFG_FAIL);
  }

  cfgOut = cfgConsumer->cfgOut;

  if(cfgConsumer->start == 0){
    cfgConsumer->start = time((time_t *)NULL) - cfgOut->nsPeriod;
  }

  /*-------------------------------- Remove previous consumers that requested */
  if((cfgConsumer->remove == CfgTrue) &&
     (CmMessageGetState(cfgConsumer->message) == CmMessageIsNew)){

    CfgOutGroupPrint(group, 0,
                     "CfgOut> %s removed from group %s.\n",
                     cfgConsumer->name,
                     cfgOut->group);

    CfgNameListRemoveName(cfgOut->csList, name);

    return(CFG_OK);
  }

  priority = CfgNameListIData(cfgOut->csList,
                              CfgNameListSearchName(cfgOut->csList, name));

  if(priority < cfgOut->threshold){ return(CFG_OK); }

  _CfgOutConnectionOpen(group, 50e-3);
  /*--------------------------------------------------- Get state of consumer */
  CfgOutConsumerState(cfgConsumer, name, group, CfgTrue);

  if((cfgConsumer->complete == CfgFalse) && (cfgConsumer->alive == CfgTrue)){
    int rc;
  
    if(cfgOut->buildMsg != (CfgOutBuildMsg_t)NULL){
      rc = cfgOut->buildMsg(cfgConsumer->message,
                            cfgOut->arg,
                            cfgConsumer->name,
                            cfgOut->group);
    }
    else{
      rc = buildMsg(cfgConsumer->message,
                    arg,
                    cfgConsumer->name,
                    cfgOut->group);
    }

    switch(rc){
      case CFG_FAIL:
        fprintf(stderr,
                "CfgOutGroupConsumerPost: Cannot build message for %s\n",
                cfgConsumer->name);
        return(rc);
        break;
      case CFG_OK:
        cfgConsumer->complete = CfgTrue;
        break;
      case CFG_CONTINUE:
        return(rc);
        break;
    }

    if(cfgConsumer->complete == CfgTrue){
      /*------------------------------------------- Free previous Cm iterator */
      CmMessageIteratorDelete(cfgConsumer->iterator);

      /*----------------------------------------------------- Post Cm message */
      cfgConsumer->iterator = CmMessagePost(cfgConsumer->message,
                                            cfgConsumer->name,
				            CfgOutPostHandler);

      /*------------------ if Consumer is server itself then send message now */
      if(CfgOutConsumerIsServer(cfgConsumer->name)){
        CmMessageIteratorForth(cfgConsumer->iterator);
      }

      cfgConsumer->counter = 0;

      CfgOutGroupPrint(group, 1, "CfgOut> (%2d/%2d) Message is sent to %s.\n",
                       CfgNameListSearchName(cfgOut->csList, name), 
                       cfgOut->maxConsumer, cfgConsumer->name);
    }
  }
  else{
    /*--------------------------- Here message is blocked or consumer is dead */
    cfgConsumer->counter++;
    CfgOutPrintMsgError(cfgConsumer);
  }

  /*----------- Remove consumer if reach maximum of retry or consumer is dead */
  if(((cfgConsumer->counter > cfgConsumer->retry) &&
      (cfgConsumer->retry >= 0)) ||
     ((cfgConsumer->alive == CfgFalse) && (cfgConsumer->remove == CfgTrue))){

    CfgOutGroupPrint(group, 0, "CfgOut> remove %s from group %s [%d/%d].\n",
                cfgConsumer->name,
                cfgOut->group,
                cfgConsumer->counter,
                cfgConsumer->retry);

    CfgNameListRemoveName(cfgOut->csList, name);
  }

  /*--------------- Remove current group that is not `default' if no consumer */
  if(cfgOut && (CfgNameListSize(cfgOut->csList) == 0) &&
     ((group != (char *)NULL) &&
      (strcmp(group, CfgOutGroupDefaultGet()) != 0))){

    CfgOutGroupPrint(group, 0, "CfgOut> group %s is removed.\n", cfgOut->group);

    CfgOutDomainGroupRemove("all", group);
    cfgOut = _CfgOutFree(cfgOut);
  }

  return((cfgConsumer->alive == CfgTrue)? CFG_OK: CFG_FAIL);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupConsumerRemove ------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupConsumerRemove(char *group, char *name){
CfgOut_t *cfgOut;

  if(name == (char *)NULL){ return(CFG_FAIL); }

  if((cfgOut = CfgOutSearchByGroup(group)) != (CfgOut_t *)NULL){
    int index;

    if((index = CfgNameListSearchName(cfgOut->csList, name)) != CFG_FAIL){
      CfgConsumer_t *cfgConsumer = CfgNameListVData(cfgOut->csList, index);

      /*---------------------------------------- Tag consumer must be removed */
      cfgConsumer->remove = CfgTrue;
      CfgOutGroupPrint(group, 0,
                       "CfgOut> %s from group %s is tagged has to be removed\n",
		       cfgConsumer->name, cfgOut->group);
    }
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupConsumerRequest -----------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupConsumerRequest(char *group, char *name,
                               CfgOutBuildMsg_t buildMsg, void *arg,
                               double timeout){
register CfgBool_t granted;
        
  if(name == (char *)NULL){ return(CFG_FAIL); }

  CfgOutRegisterDisconnection(name);

  if(CfgOutGroupConsumerPost(group, name, buildMsg, arg) == CFG_FAIL){
    return(CFG_FAIL);
  }

  /*------------------- Send request if consumer is there or message is build */
  CmMessageWaitWithTimeout(timeout);

  granted = CfgOutIsConnectionGranted(name);

  return((granted == CfgTrue)? CFG_OK: CFG_FAIL);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupConsumerStatusGet ---------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgOutStatus_t CfgOutGroupConsumerStatusGet(char *group, char *name){
CfgConsumer_t *cfgConsumer;
        
  cfgConsumer = CfgOutConsumerSearch(name, group);
  if(cfgConsumer == (CfgConsumer_t *)NULL){ return(CfgOutNone); }

  CfgOutConsumerState(cfgConsumer, name, group, CfgFalse);
  
  return(cfgConsumer->status);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupDebugSet ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupDebugSet(char *group, int debug){
CfgOut_t *cfgOut;

  if((cfgOut = CfgOutSearchByGroup(group)) != (CfgOut_t *)NULL){ 
    cfgOut->debug = debug;
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupDefaultGet ----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
char *CfgOutGroupDefaultGet(void){ return(CfgOutDefaultGroup); }

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupDefaultSet ----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
void CfgOutGroupDefaultSet(char *group){
  strcpy(CfgOutDefaultGroup, group? group: CfgOutGroupDefaultGet());
  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupFree ----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
void CfgOutGroupFree(char *group){ _CfgOutFree(CfgOutSearchByGroup(group)); }

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupInstallBuilder ------------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgOutBuildMsg_t CfgOutGroupInstallBuilder(char *group,
                                           CfgOutBuildMsg_t buildMsg,
                                           void *arg){
CfgOut_t *cfgOut;
CfgOutBuildMsg_t oBuildMsg = (CfgOutBuildMsg_t)NULL;

  if((cfgOut = CfgOutSearchByGroup(group)) != (CfgOut_t *)NULL){ 
    oBuildMsg        = cfgOut->buildMsg;
    cfgOut->buildMsg = buildMsg;
    cfgOut->arg      = arg;
  }

  return(oBuildMsg);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupNbConsumerGet -------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupNbConsumerGet(char *group ) {
CfgOut_t *cfgOut;
int nbConsumer;

 if( !(cfgOut = CfgOutSearchByGroup(group)) ) return(CFG_FAIL);
 /*--------------------------------------------------------------------------*/
 nbConsumer = CfgNameListSize(cfgOut->csList);

 return(nbConsumer);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupNew -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupNew(char *group, CfgParse_t *cfgParse, int maxConsumer,
                   int maxGroup, int threshold, int debug){
register int rtn = CFG_OK;
CfgOut_t *cfgOut;

  if(CfgOutHashTable == (CbfH_t *)NULL){
    if((CfgOutHashTable = CbfHNew(256)) == (CbfH_t *)NULL){
      return(CFG_FAIL);
    }

    CbfCleanAddRoot( "CfgOutHFree", (CbfCleanFnt_t)CfgOutHFree, 
		     CfgOutHashTable);

    CmConnectInitialize();
    CmConnectInstallConnectionHandler(CfgOutConnectionHandler);
    CmConnectInstallDisconnectionHandler(CfgOutDisconnectionHandler);
  }

  /*----------------------------------------------------- Group already exist */
  if((cfgOut = CfgOutSearchByGroup(group)) != (CfgOut_t *)NULL){
    return(CFG_OK);
  }

  if((cfgOut = (CfgOut_t *)calloc(1, sizeof(CfgOut_t))) == (CfgOut_t *)NULL){
    fprintf(stderr, "CfgOutGroupNew: calloc: %s\n", strerror(errno));
    return(CFG_FAIL);
  }

  if((cfgOut->csList = CfgNameListNew()) == (CfgNameList_t *)NULL){
    fprintf(stderr,
            "CfgOutGroupNew: Cannot get Consumer `nameList' descriptor\n");
    _CfgOutFree(cfgOut);
    return(CFG_FAIL);
  }

  if((cfgOut->csOnReqList = CfgNameListNew()) == (CfgNameList_t *)NULL){
    fprintf(stderr,
            "CfgOutGroupNew: Cannot get onRequest Consumer `nameList' "
            "descriptor\n");
    _CfgOutFree(cfgOut);
    return(CFG_FAIL);
  }

  if((cfgOut->gpList = CfgNameListNew()) == (CfgNameList_t *)NULL){
    fprintf(stderr, "CfgOutGroupNew: Cannot get Group `nameList' descriptor\n");
    _CfgOutFree(cfgOut);
    return(CFG_FAIL);
  }

  if((cfgOut->dmList = CfgNameListNew()) == (CfgNameList_t *)NULL){
    fprintf(stderr,
            "CfgOutGroupNew: Cannot get Domain `nameList' descriptor\n");
    _CfgOutFree(cfgOut);
    return(CFG_FAIL);
  }

  rtn |= CfgNameListSetSortOption(cfgOut->csList, CFG_NAME_DSORT_IDATA);
  rtn |= CfgNameListSetFreeFnc(cfgOut->csList,
                               (CfgNameFreeFnc_t)CfgOutConsumerFree);
  rtn |= CfgNameListSetSortOption(cfgOut->csOnReqList, CFG_NAME_DSORT_IDATA);
  rtn |= CfgNameListSetFreeFnc(cfgOut->csOnReqList,
                               (CfgNameFreeFnc_t)CfgOutConsumerFree);
  if(rtn == CFG_FAIL){
    _CfgOutFree(cfgOut);
    return(CFG_FAIL);
  }

  cfgOut->threshold   = threshold;
  cfgOut->maxConsumer = maxConsumer;
  cfgOut->maxGroup    = maxGroup;
  cfgOut->debug       = debug;
  strcpy(cfgOut->group, group? group: CfgOutGroupDefaultGet());
  CbfHEnter(CfgOutHashTable, cfgOut->group, (void *)cfgOut);

  if(CfgOutParserInit(cfgParse) == CFG_FAIL){
    _CfgOutFree(cfgOut);
    return(CFG_FAIL);
  }

  CbfCleanAddRoot( "_CfgOutFree", (CbfCleanFnt_t)_CfgOutFree, cfgOut);

  CfgOutGroupPrint(group, 0, "CfgOut> Create a new group %s "
                   "[%d, %d, %d] (initial values).\n",
                   cfgOut->group, cfgOut->maxConsumer,
                   cfgOut->threshold, cfgOut->debug);


  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupNsPeriodSet ---------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupNsPeriodSet(char *group, int period){
CfgOut_t *cfgOut;
  
  if((cfgOut = CfgOutGetOut(group)) != (CfgOut_t *)NULL){ 
    cfgOut->nsPeriod = period;
  }

  return(CFG_OK);
}


/*----------------------------------------------------------------------------*/
/*- CfgOutGroupPost ----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupPost(char *group, CfgOutBuildMsg_t buildMsg, void *arg){
volatile int index, rtn;
register CfgOut_t *cfgOut;

  if(buildMsg == (CfgOutBuildMsg_t)NULL){
    CfgOutGroupPrint(group, 0,
                     "CfgOut> CfgOutGroupPost: "
                     "Invalid argument \"null\" pointer passed.\n");

    return(CFG_FAIL);
  }

  if((cfgOut = CfgOutSearchByGroup(group)) == (CfgOut_t *)NULL){
    return(CFG_OK);
  }

  rtn   = CFG_OK;
  index = 0;

  if(cfgOut->start == 0){
    cfgOut->start = time((time_t *)NULL) - cfgOut->nsPeriod;
  }

  {
    time_t now = time((time_t *)NULL);
    double period = difftime(now, cfgOut->start);

    while(index < CfgNameListSize(cfgOut->csList)){
      CfgConsumer_t *cfgConsumer = CfgNameListVData(cfgOut->csList, index);

      /*---------------------------------- Search consumers from pattern name */
      if(cfgConsumer == (CfgConsumer_t *)NULL){
        if(period >= cfgOut->nsPeriod){
          CfgOutConsumerGetNames(cfgOut,
                                 CfgNameListName(cfgOut->csList, index));
        }
      }
      else{
        /*-------------------------- Remove previous consumers that requested */
        if((cfgConsumer->remove == CfgTrue) &&
           (CmMessageGetState(cfgConsumer->message) == CmMessageIsNew)){
          CfgOutGroupPrint(group, 0,
                           "CfgOut> %s removed from group %s.\n",
                           cfgConsumer->name,
                           cfgOut->group);

          CfgNameListNRemove(cfgOut->csList, index);
          continue;
        }
      }

      index++;
    }

    cfgOut->start = (period >= cfgOut->nsPeriod)? now: cfgOut->start;
  }

  index = 0;

  _CfgOutConnectionOpen(group, 50e-3);

  /*------------------------------------- Send message to consumers of domain */
  while(index < CfgNameListSize(cfgOut->csList)){
    CfgConsumer_t *cfgConsumer = CfgNameListVData(cfgOut->csList, index);

    if(cfgConsumer == (CfgConsumer_t *)NULL){ index++; continue; }

    if(cfgConsumer->start == 0){
      cfgConsumer->start = time((time_t *)NULL) - cfgOut->nsPeriod;
    }

    if(CfgNameListIData(cfgOut->csList, index) < cfgOut->threshold){ break; }

    /*------------------------------------------------- Get state of consumer */
    CfgOutConsumerState(cfgConsumer, cfgConsumer->name, group, CfgTrue);

    if((cfgConsumer->complete == CfgFalse) && (cfgConsumer->alive == CfgTrue)){
      volatile int rc;

      if(cfgOut->buildMsg != (CfgOutBuildMsg_t)NULL){
        rc = cfgOut->buildMsg(cfgConsumer->message,
                              cfgOut->arg,
                              cfgConsumer->name,
                              cfgOut->group);
      }
      else{
        rc = buildMsg(cfgConsumer->message,
                      arg,
                      cfgConsumer->name,
                      cfgOut->group);
      }

      switch(rc){
        case CFG_FAIL:
          rtn = CFG_FAIL;
          fprintf(stderr, "CfgOutGroupPost: Cannot build message for %s\n",
                  cfgConsumer->name);
        case CFG_CONTINUE:
          index++;
          continue;
          break;
        case CFG_OK:
          cfgConsumer->complete = CfgTrue;
          break;
      }

      if(cfgConsumer->complete == CfgTrue){
        /*----------------------------------------- Free previous Cm iterator */
	CmMessageIteratorDelete(cfgConsumer->iterator);

        /*--------------------------------------------------- Post Cm message */
        cfgConsumer->iterator = CmMessagePost(cfgConsumer->message,
                                              cfgConsumer->name,
					      CfgOutPostHandler);

        /*---------------- if Consumer is server itself then send message now */
        if(CfgOutConsumerIsServer(cfgConsumer->name)){
          CmMessageIteratorForth(cfgConsumer->iterator);
        }

        cfgConsumer->counter = 0;

        CfgOutGroupPrint(group, 1, "CfgOut> (%2d/%2d) Message is sent to %s.\n",
                         index, cfgOut->maxConsumer, cfgConsumer->name);
      }
    }
    else{
      /*------------------------- Here message is blocked or consumer is dead */
      cfgConsumer->counter++;
      CfgOutPrintMsgError(cfgConsumer);
    }

    /*-------- Remove consumer if reach maximum of retry or consumer is dead */
    if(((cfgConsumer->counter > cfgConsumer->retry) &&
        (cfgConsumer->retry >= 0)) ||
       ((cfgConsumer->alive == CfgFalse) && (cfgConsumer->remove == CfgTrue))){

      CfgOutGroupPrint(group, 0, "CfgOut> remove %s from group %s [%d/%d].\n",
                       cfgConsumer->name,
                       cfgOut->group,
                       cfgConsumer->counter,
                       cfgConsumer->retry);

      CfgNameListNRemove(cfgOut->csList, index);
      continue;
    }

    index++;
  }

  index = 0;

  /*---------- Send message to consumers of subgroups into the current domain */
  while((cfgOut != (CfgOut_t *)NULL) &&
        !CfgNameListEmpty(cfgOut->gpList) &&
        (index < CfgNameListSize(cfgOut->gpList))){
    char *subgroup = CfgNameListName(cfgOut->gpList, index);

    CfgOutGroupPrint(group, 2,
                     "\t*** Send message of group %s in domain %s ***\n",
                     subgroup, group? group: CfgOutGroupDefaultGet());

    if(CfgOutGroupPost(subgroup, buildMsg, arg) == CFG_FAIL){
      return(CFG_FAIL);
    }

    index++;
  }

  /*--------------- Remove current group that is not `default' if no consumer */
  if(cfgOut && (CfgNameListSize(cfgOut->csList) == 0) &&
     ((group != (char *)NULL) &&
      (strcmp(group, CfgOutGroupDefaultGet()) != 0))){

    CfgOutGroupPrint(group, 0, "CfgOut> group %s is removed.\n", cfgOut->group);

    CfgOutDomainGroupRemove("all", group);
    cfgOut = _CfgOutFree(cfgOut);
  }

  return(rtn);
}



/*----------------------------------------------------------------------------*/
/*- CfgOutGroupStatusGet -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutGroupStatusGet(char *group, char *name, CfgOutStatus_t *status){
CfgOut_t *cfgOut;

  if(name == (char *)NULL){ return(CFG_FAIL); }
  else{ name[0] = '\0'; }

  if(status == (CfgOutStatus_t *)NULL){ return(CFG_FAIL); }
  else{ *status = CfgOutNone; }

  if((cfgOut = CfgOutSearchByGroup(group)) != (CfgOut_t *)NULL){
    int nbConsumer;

    nbConsumer = CfgNameListSize(cfgOut->csList);

    if(cfgOut->index < nbConsumer){
    CfgConsumer_t *cfgConsumer = CfgNameListVData(cfgOut->csList,
                                                  cfgOut->index);

      strcpy(name, cfgConsumer->name);

/*       CfgOutConsumerState(cfgConsumer, name, group, CfgFalse); */

      *status = cfgConsumer->status;

      cfgOut->index++;

      return(CFG_OK);
    }
    else{
      cfgOut->index = 0;
    }
  }

  return(CFG_FAIL);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupThresholdSet --------------------------------------------------*/
/*----------------------------------------------------------------------------*/
void CfgOutGroupThresholdSet(char *group, int threshold){
CfgOut_t *cfgOut;

  if((cfgOut = CfgOutGetOut(group)) != (CfgOut_t *)NULL){
    cfgOut->threshold = threshold;

    if(!CfgNameListEmpty(cfgOut->csList)){
      if(cfgOut->threshold > CfgNameListIData(cfgOut->csList, 0)){
        cfgOut->threshold = CfgNameListIData(cfgOut->csList, 0);
      }
    }
  }

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupUninstallBuilder ----------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgOutBuildMsg_t CfgOutGroupUninstallBuilder(char *group){
CfgOut_t *cfgOut;
CfgOutBuildMsg_t oBuildMsg = (CfgOutBuildMsg_t)NULL;

  if((cfgOut = CfgOutSearchByGroup(group)) != (CfgOut_t *)NULL){ 
    oBuildMsg        = cfgOut->buildMsg;
    cfgOut->buildMsg = (CfgOutBuildMsg_t)NULL;
    cfgOut->arg      = (void *)NULL;
  }

  return(oBuildMsg);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutParserInit ---------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutParserInit(CfgParse_t *cfgParse){
register int rtn = CFG_OK;

  if(cfgParse != (CfgParse_t *)NULL){
    CfgParseSetEnumAttr(cfgParse, "static", CfgOutStatic);
    CfgParseSetEnumAttr(cfgParse, "request", CfgOutOnRequest);

    rtn |= CfgParseFunctionAdd(cfgParse, "CFGOUT_NS_PERIOD",
                               CfgOutNsPeriodSet, (void *)NULL, 2, CfgDec,
                               CfgString);
    rtn |= CfgParseFunctionAdd(cfgParse, "CFGOUT_DEBUG", CfgOutDebugSet,
                               (void *)NULL, 2, CfgDec, CfgString);
    rtn |= CfgParseFunctionAdd(cfgParse, "CFGOUT_THRESHOLD",
                               CfgOutThresholdSet,
                               (void *)NULL, 2, CfgDec, CfgString);
    rtn |= CfgParseFunctionAdd(cfgParse, "CFGOUT_STAT", CfgOutStatSet,
                               (void *)NULL, 2, CfgBoolean, CfgString);
    rtn |= CfgParseFunctionAdd(cfgParse, "CFGOUT_MAX_CONSUMER",
                               CfgOutMaxConsumerSet,
                               (void *)NULL, 2, CfgDec, CfgString);
    rtn |= CfgParseFunctionAdd(cfgParse, "CFGOUT_MAX_GROUP",
                               CfgOutMaxGroupSet,
                               (void *)NULL, 2, CfgDec, CfgString);
    rtn |= CfgParseFunctionAdd(cfgParse, "CFGOUT_CHECK_GROUP",
                               CfgOutCheckGroupSet,
                               (void *)NULL, 2, CfgBoolean, CfgString);
    rtn |= CfgParseFunctionAdd(cfgParse, "CFGOUT_CONSUMER", CfgOutConsumerAdd,
                               (void *)NULL, 7, CfgString, CfgDec,
                               CfgDec, CfgString, CfgEnum, CfgString,
                               CfgString);
  }

  return(rtn);
}

/******************************************************************************/
/*                         MISCELLIOUS FUNCTIONS                              */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*- CfgOutCmReqCsmAdd --------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutCmReqCsmAdd(char *server, int retry, int priority, char *group,
                      char *sender){
register int rc;
CmMessage message;
CbfHEntry_t *itemp;
char key[128];


  if(server == (char *)NULL){
    fprintf(stderr,
            "CfgOutCmReqCsmAdd: `server' invalid pointer argument\n");
    return(CFG_FAIL);
  }

  CfgOutTagServer(server);

  message = CmMessageNew();

  CmMessageSetType(message, CFGOUT_CSM_ADD_TYPE);
  CmMessagePutInt(message, retry);
  CmMessagePutInt(message, priority);
  CmMessagePutText(message, group);
  CmMessagePutText(message, sender);

  rc = CmMessageSend(message, server);

  sprintf(key, "#%s#", server);

  if(!rc && (itemp = CbfHFind(CfgOutHashTable, key)) != (CbfHEntry_t *)NULL){
    itemp->data = (void *)CFGOUT_CON_DENIED;
  }

  CmMessageDelete(message);

  return(rc? CFG_OK: CFG_FAIL);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCmReqCsmRemove -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutCmReqCsmRemove(char *server, char *group, char *sender){
register int rc;
CmMessage message;
CbfHEntry_t *itemp;
char key[128];

  if(server == (char *)NULL){
    fprintf(stderr,
            "CfgOutCmReqCsmRemove: `server' invalid pointer argument\n");
    return(CFG_FAIL);
  }

  CfgOutTagServer(server);

  message = CmMessageNew();

  CmMessageSetType(message, CFGOUT_CSM_RM_TYPE);
  CmMessagePutText(message, sender);
  CmMessagePutText(message, group);

  rc = CmMessageSend(message, server);

  sprintf(key, "#%s#", server);

  if(!rc && (itemp = CbfHFind(CfgOutHashTable, key)) != (CbfHEntry_t *)NULL){
    itemp->data = (void *)CFGOUT_CON_DENIED;
  }

  CmMessageDelete(message);

  return(rc? CFG_OK: CFG_FAIL);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCmReqGroupAdd ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutCmReqGroupAdd(char *server, char *group, char *domain){
register int rc;
CmMessage message;
CbfHEntry_t *itemp;
char key[128];

  if(server == (char *)NULL){
    fprintf(stderr, "CfgOutCmReqGroupAdd: `server' invalid pointer argument\n");
    return(CFG_FAIL);
  }

  if(((group == (char *)NULL) &&
      (domain == (char *)NULL)) ||
     ((group != (char *)NULL) &&
      (domain != (char *)NULL) &&
      (strcmp(group, domain) == 0))){
    return(CFG_OK);
  }

  CfgOutTagServer(server);

  message = CmMessageNew();

  CmMessageSetType(message, CFGOUT_GRP_ADD_TYPE);
  CmMessagePutText(message, group);
  CmMessagePutText(message, domain);

  rc = CmMessageSend(message, server);

  sprintf(key, "#%s#", server);

  if(!rc && (itemp = CbfHFind(CfgOutHashTable, key)) != (CbfHEntry_t *)NULL){
    itemp->data = (void *)CFGOUT_CON_DENIED;
  }

  CmMessageDelete(message);

  return(rc? CFG_OK: CFG_FAIL);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutCmReqGroupRemove ---------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgOutCmReqGroupRemove(char *server, char *group, char *domain){
register int rc;
CmMessage message;
CbfHEntry_t *itemp;
char key[128];

  if(server == (char *)NULL){
    fprintf(stderr,
            "CfgOutCmReqGroupRemove: `server' invalid pointer argument\n");
    return(CFG_FAIL);
  }

  if(((group == (char *)NULL) &&
      (domain == (char *)NULL)) ||
     ((group != (char *)NULL) &&
      (domain != (char *)NULL) &&
      (strcmp(group, domain) == 0))){
    return(CFG_OK);
  }

  CfgOutTagServer(server);

  message = CmMessageNew();

  CmMessageSetType(message, CFGOUT_GRP_RM_TYPE);
  CmMessagePutText(message, group);
  CmMessagePutText(message, domain);

  rc = CmMessageSend(message, server);

  sprintf(key, "#%s#", server);

  if(!rc && (itemp = CbfHFind(CfgOutHashTable, key)) != (CbfHEntry_t *)NULL){
    itemp->data = (void *)CFGOUT_CON_DENIED;
  }

  CmMessageDelete(message);

  return(rc? CFG_OK: CFG_FAIL);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutIsConnectionGranted ------------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgBool_t CfgOutIsConnectionGranted(char *server){
register CfgBool_t granted;
register CbfHEntry_t *itemp;
char key[128];

  sprintf(key, "#%s#", server);

  if((itemp = CbfHFind(CfgOutHashTable, key)) != (CbfHEntry_t *)NULL){
    granted = (itemp->data == (void *)CFGOUT_CON_GRANTED)? CfgTrue: CfgFalse;
  }
  else{ granted = CfgUnknown; }

  return(granted);
}

/*----------------------------------------------------------------------------*/
/*- CfgOutGroupStatReset -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
void CfgOutGroupStatReset(char *group){
register int index;
CfgOut_t *cfgOut;

  if((cfgOut = CfgOutSearchByGroup(group)) == (CfgOut_t *)NULL){
    for(index=0; index<CfgNameListSize(cfgOut->csList); index++){
      CfgConsumer_t *cfgConsumer = CfgNameListVData(cfgOut->csList, index);

      memset((char *)&cfgConsumer->info, 0, sizeof(CfgOutInfo_t));
    }
  }

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgOutRegisterDisconnection ----------------------------------------------*/
/*----------------------------------------------------------------------------*/
void CfgOutRegisterDisconnection(char *server){
CbfHEntry_t *itemp;
char key[128];

  if(server == (char *)NULL){ return; }

  if(CfgOutHashTable == (CbfH_t *)NULL){
    if((CfgOutHashTable = CbfHNew(256)) == (CbfH_t *)NULL){ return; }
    CbfCleanAddRoot( "CfgOutHFree", (CbfCleanFnt_t)CfgOutHFree,
		     CfgOutHashTable);
  }

  sprintf(key, "#%s#", server);

  if((itemp = CbfHFind(CfgOutHashTable, key)) == (CbfHEntry_t *)NULL){
    CbfHEnter(CfgOutHashTable, key, (void *)CFGOUT_CON_GRANTED);
    CmConnectInitialize();
    CmConnectInstallDisconnectionHandler(CfgOutCsmDisconnectionHandler);
  }
  else{
    itemp->data = (void *)CFGOUT_CON_GRANTED;
  }

  return;
}
