/* SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Cfg - A library for Virgo process implementation
 *
 * Copyright © 2001-2021 Laboratoire de physique des particules d'Annecy - CNRS
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Fatih Bellachia <fatih.bellachia@lapp.in2p3.fr>
 *   Laurent Fournier <laurent.fournier@lapp.in2p3.fr>
 *   Alain Masserot <alain.masserot@lapp.in2p3.fr>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <CbfHash.h>
#include <CbfClean.h>
#include <CfgParse.h>

#ifdef USE_MW
#include <memwatch.h>
#endif /* USE_MW */

#ifdef USE_MPATROL
#include <mpatrol.h>
#endif /* USE_MPATROL */

#define CFG_LINELEN  256
/* #define CFG_BUFSIZE  1024 */
#define CFG_BUFSIZE  2048 /*---------------- Updated the 20190726 - BM & AM */
#define CFG_TEXT_LEN 19

extern int vsprintf(char *, const char *, va_list);
extern char *CfgDirname(char *path);

typedef struct CfgStack{
  Ulong array[CFG_MAX_STACK];  /* Used to pass arguments to the user function */
} CfgStack_t;

typedef struct CfgDefine{
  char key[CFG_MAX_NAME];
  /* char string[CFG_MAX_NAME]; */
  char *string;
  int  ncalls;
} CfgDefine_t;

typedef struct CfgDefineStack{
  int nb;
  CfgDefine_t *stack;
} CfgDefineStack_t;

typedef struct CfgInclude{
  char filename[CFG_MAXPATHLEN];
  int  ncalls;
} CfgInclude_t;

typedef struct CfgIncludeStack{
  int nb;
  CfgInclude_t *stack;
} CfgIncludeStack_t;

typedef struct CfgArgs{
  void      *attribute;
  char      *format;
  int       variant;
  CfgBool_t flag;
  CfgType_t type;
  int       (*scan)();
} CfgArgs_t;

struct CfgDataType {
  char       *keyword;
  long       index;
  int        nb;
  int        iarg;
  int        nargs;
  CfgArgs_t  args[CFG_MAX_ATTR];
  CfgBool_t  exec;
  CfgBool_t  countArg;
  void       *farg;
  CfgStack_t stack;
  struct {
    int result;
    int index;
    CfgBool_t iarg_done;
  } user;
};

typedef struct CfgAction{
  int               nb;
  int               rest;
  CfgDataType_t     *previous;
  CfgDataType_t     *array;
  CfgDefineStack_t  define;
  CfgIncludeStack_t include;
  CbfH_t            *hTable;
  CbfH_t            *hEnum;
  struct CfgParse   *cfgParse;
} CfgAction_t;

typedef struct CfgComment{
  CfgBool_t start;
  CfgBool_t end; 
  CfgBool_t C_block;     
  int       count;
  int       lineno;
  int       nspaces;
} CfgComment_t;

struct CfgParse {
  char        *keyword;
  int         nb;
  void        *farg;
  Ulong       dummy[CFG_MAX_STACK];
  struct _dargs{
    CfgType_t type;
    void      *attribute;
  } dargs[CFG_MAX_ATTR];
  CfgAction_t     *cfgAction;
  CbfH_t          *hEnum;
  char            *error;
  int             len;
  CfgBool_t       countArg;
  CfgBool_t       processed;
  CfgBool_t       match;
  CfgBool_t       extract;
  int             iarg;
  char            *buffer;
  char            *config;
  char            *head;
  CfgComment_t    comment;
  char            dirname[CFG_MAXPATHLEN];
  struct CfgParse *next;
};

typedef struct CfgError{
  char *msg;
  int  len;
} CfgError_t;

typedef struct CfgInfo{
  char      padding[8];
  char      filename[CFG_MAXPATHLEN];
  CfgBool_t include;
  CfgBool_t processed;
} CfgInfo_t;

char const *CfgFormat[]={
  "",                                /* Conversion specifier for type CfgNone */
  "",                                /* Conversion specifier for type CfgFlag */
  " %s%n",                    /* Conversion specifier for type CfgConstString */
  " %s%n",                         /* Conversion specifier for type CfgString */
  " %s%n",                        /* Conversion specifier for type CfgBoolean */
  " %lf%n",                          /* Conversion specifier for type CfgReal */
  " %lx%n",                           /* Conversion specifier for type CfgHex */
  " %d%n",                           /*  Conversion specifier for type CfgDec */
  "",                            /* Conversion specifier for type CfgFunction */
  " %s%n",                           /* Conversion specifier for type CfgEnum */
  " %s%n",                     /* Conversion specifier for type CfgProcessing */
  ""                          /* Conversion specifier for type CfgGetFunction */
};

char const *CfgTypeName[] = {
  "None",
  "Flag",
  "Const string",
  "String",
  "Boolean",
  "Real",
  "Hex",
  "Dec",
  "Function",
  "Enum",
  "Processing",
  "GetFunction"
};

static CfgError_t CfgError = { NULL, 1 };
static CfgParsePrinter_t CfgParsePrinter = vprintf;

static void CfgParseDoPrint( char *format, ... );
/******************************************************************************/
/* PRIVATE FUNCTIONS                                                          */
/******************************************************************************/
static CfgAction_t   *CfgActionFree(CfgAction_t *);
static CfgAction_t   *CfgActionNew(CfgParse_t *);
static void          CfgCheckCBlock(char *, int);
static int           CfgCheckLastDataType(CfgAction_t *);
static int           CfgConfigExtSprintf(CfgParse_t *, char *, ...);
static CfgDataType_t *CfgDataTypeFree(CfgDataType_t *, int);
static CfgDataType_t *CfgDataTypeNew(int);
static int           CfgDefaultScan(CfgAction_t *, char *, CfgArgs_t *);
static int           CfgDefineScan(CfgAction_t *, char *, CfgArgs_t *);
static int           CfgDefineStackNew(CfgAction_t *, char *, char *);
static char          *_CfgFileToBuf(char *, CfgParse_t *);
static int           CfgFlagScan(CfgAction_t *, char *, CfgArgs_t *);
static void          _CfgFree(CfgParse_t *);
static int           CfgFunctionExec(CfgParse_t *, CfgDataType_t *);
static int           CfgGetEnumAttr(CbfH_t *, char *, long *);
static char          *CfgGetLine(char *);
static char          *CfgGetLine_r(char *, char**);
static CfgParse_t    *CfgGetNext(CfgParse_t *);
static int           CfgIncludeScan(CfgAction_t *, char *, CfgArgs_t *);
static int           CfgIncludeStackNew(CfgAction_t *, char *);
static int           CfgItemScan(CfgAction_t *, char *, CfgArgs_t *);
static int           CfgNumericalScan(CfgAction_t *, char *, CfgArgs_t *);
static int           CfgParseBuffer_r(CfgParse_t *, char *, char **);
static int           CfgParseLine(CfgAction_t *, char *, int);
static char          *CfgProcessBuffer(char *);
static int           CfgReplaceDefine(CfgAction_t *, char *, int);
static char          *CfgReverseSkipSpace(char *);
static int           CfgScanToken(CfgType_t, char *, char *, char *, int *, int);
static CfgDataType_t *CfgSearchDataType(CfgAction_t *, char *);
static int           CfgSearchKey(CfgParse_t *, char *);
static int           CfgSetBoolean(char *, CfgBool_t *);
static void          CfgSetLineComment(char *, CfgComment_t *, int);
static char          *CfgSkipComment(CfgParse_t *, char *, int);
static char          *CfgSkipSpace(char *);
static int           CfgStringScan(CfgAction_t *, char *, CfgArgs_t *);
static int           CfgExtSprintf(CfgParse_t *, char *, ...);

char *CfgMapFile(char *);

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgParseDoPrint */
/*----------------------------------------------------------------------------*/
static void CfgParseDoPrint( char *format, ... ) {
  va_list args; 

  if( !CfgParsePrinter || !format ) return;
  /*--------------------------------------------------------------------------*/
  va_start( args, format );
  CfgParsePrinter( format, args );
  va_end( args );
  /*--------------------------------------------------------------------------*/
  return;
}



/*----------------------------------------------------------------------------*/
/*- CfgActionFree ------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgAction_t *CfgActionFree(CfgAction_t *cfgAct){
register int i;

  if(cfgAct == (CfgAction_t *)NULL){
    return(cfgAct);
  }

  cfgAct->hTable = CbfHFree(cfgAct->hTable);

  /*------------------------------------------------------ Clear all keywords */
  for(i=0; i<cfgAct->nb; i++){
    if(cfgAct->array[i].keyword != (char *)NULL){
      free((void *)cfgAct->array[i].keyword);
      cfgAct->array[i].keyword = (char *)NULL;
    }
  }

  cfgAct->array = CfgDataTypeFree(cfgAct->array, cfgAct->nb);

  if(cfgAct->define.stack != (CfgDefine_t *)NULL){
    for( i=0; i<cfgAct->define.nb; i++ ) {
      free( cfgAct->define.stack[i].string );
      cfgAct->define.stack[i].string = NULL;
    }
    free(cfgAct->define.stack);
    cfgAct->define.stack = (CfgDefine_t *)NULL;
    cfgAct->define.nb    = 0;
  }

  if(cfgAct->include.stack != (CfgInclude_t *)NULL){
    free(cfgAct->include.stack);
    cfgAct->include.stack = (CfgInclude_t *)NULL;
    cfgAct->include.nb    = 0;
  }

  free((void *)cfgAct);

  return((CfgAction_t *)NULL);
}

/*----------------------------------------------------------------------------*/
/*- CfgActionNew -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgAction_t *CfgActionNew(CfgParse_t *cfgParse){
register int  i, j;
CfgAction_t   *cfgAct;
CfgParse_t    *pt;

  if(cfgParse == (CfgParse_t *)NULL){
    CfgParseDoPrint("CfgActionNew> Invalid argument \"nil\" pointer passed.\n");
    return((CfgAction_t *)NULL);
  }

  if(cfgParse->cfgAction != (CfgAction_t *)NULL){
    if(cfgParse->processed == CfgFalse){ return(cfgParse->cfgAction); }
    else{ cfgParse->cfgAction = CfgActionFree(cfgParse->cfgAction); }
  }

  if((cfgAct = malloc(sizeof(CfgAction_t))) == (CfgAction_t *)NULL){
    CfgParseDoPrint("CfgActionNew>  malloc failed");
    return((CfgAction_t *)NULL);
  }

  memset((char *)cfgAct, 0, sizeof(CfgAction_t));

  cfgAct->cfgParse = cfgParse;

  if(cfgParse->keyword != (char *)NULL){
    for(pt=cfgParse; pt!=(CfgParse_t *)NULL; pt=pt->next){ cfgAct->nb++; }
  }

  if((cfgAct->array = CfgDataTypeNew(cfgAct->nb + 2)) == (CfgDataType_t *)NULL){
    free((void *)cfgAct);
    return((CfgAction_t *)NULL);
  }

  cfgAct->hEnum  = cfgParse->hEnum;
  cfgAct->hTable = CbfHNew(cfgAct->nb);

  if(cfgAct->hTable == (CbfH_t *)NULL){
    free((void *)cfgAct);
    return((CfgAction_t *)NULL);
  }

  for(i=0, pt=cfgParse; i<cfgAct->nb; i++){
    CfgDataType_t *dataType = &cfgAct->array[i];

    dataType->keyword  = strdup(pt->keyword);
    dataType->nb       = pt->nb;
    dataType->farg     = pt->farg;
    dataType->index    = 1;
    dataType->countArg = pt->countArg;
    dataType->iarg     = pt->iarg;

    if(dataType->keyword == (char *)NULL){
      free((void *)cfgAct);
      CfgParseDoPrint("CfgActionNew> strdup(keyword)");
      return((CfgAction_t *)NULL);
    }

    if(pt->nb > CFG_MAX_ATTR){
      CfgActionFree(cfgAct);
      CfgParseDoPrint("CfgActionNew> (%s) Number of attributes > %d\n",
		      pt->keyword,
		      CFG_MAX_ATTR);
      return((CfgAction_t *)NULL);
    }

    for(j=0; j<pt->nb; j++){
      dataType->args[j].attribute = pt->dargs[j].attribute;
      dataType->args[j].type      = pt->dargs[j].type;
      dataType->args[j].format    = strdup(CfgFormat[pt->dargs[j].type]);

      if(dataType->args[j].format == (char *)NULL){
        CfgActionFree(cfgAct);
        CfgParseDoPrint("CfgActionNew> strdup(args:format)");
        return((CfgAction_t *)NULL);
      }

      switch(pt->dargs[j].type){
        case CfgConstString:
        case CfgString:
          /*----------------------------------------------------- String type */
          dataType->args[j].scan    = CfgStringScan;
          dataType->args[j].variant = CFG_KEY_ATTRIBUTE;
          break;
        case CfgBoolean:
        case CfgEnum:
          /*------------------------------------------------------- Item type */
          dataType->args[j].scan    = CfgItemScan;
          dataType->args[j].variant = CFG_KEY_ATTRIBUTE;
          break;
        case CfgReal:
        case CfgHex:
        case CfgDec:
          /*-------------------------------------------------- Numerical type */
          dataType->args[j].scan    = CfgNumericalScan;
          dataType->args[j].variant = CFG_KEY_ATTRIBUTE;
          break;
        case CfgFlag:
          /*------------------------------------------------------- Flag type */
          if(j == 0){
            dataType->args[j].scan    = CfgFlagScan;
            dataType->args[j].variant = CFG_KEY;
          }
          else{
            CfgActionFree(cfgAct);
            CfgParseDoPrint("CfgActioNew> (%s) Invalide attribute type ->"
			    " `CfgFlag', use `CfgString'\n", pt->keyword);
            return((CfgAction_t *)NULL);
          }
          break;
        case CfgFunction:
	case CfgGetFunction:
          /*--------------------------------------------------- Function type */
          if(dataType->args[j].attribute == (void *)NULL){
            CfgActionFree(cfgAct);
            CfgParseDoPrint("CfgActionNew> Pointer of function is `nil'\n");
            return((CfgAction_t *)NULL);
          }
          else{
            dataType->args[j].scan    = CfgDefaultScan;
            dataType->args[j].variant = CFG_NONE_KEY;
            dataType->exec            = CfgTrue;
          }
          break;
        default:
          CfgParseDoPrint("CfgActionNew> Unknown type -> %d (%s)\n",
			  pt->dargs[j].type, pt->keyword);
          CfgActionFree(cfgAct);
          return((CfgAction_t *)NULL);
      }
    }

    pt = pt->next;
  }

  /*--------------------------------- Add info for define and include keyword */
  {
    CfgDataType_t *dataType;

    dataType                    = &cfgAct->array[i++];
    dataType->keyword           = strdup("define");
    dataType->nb                = 1;
    dataType->args[0].attribute = (void *)NULL;
    dataType->args[0].type      = CfgProcessing;
    dataType->args[0].format    = strdup(CfgFormat[CfgProcessing]);
    dataType->args[0].variant   = CFG_KEY_ATTRIBUTE;
    dataType->args[0].scan      = CfgDefineScan;

    dataType                    = &cfgAct->array[i];
    dataType->keyword           = strdup("include");
    dataType->nb                = 1;
    dataType->args[0].attribute = (void *)NULL;
    dataType->args[0].type      = CfgProcessing;
    dataType->args[0].format    = strdup(CfgFormat[CfgProcessing]);
    dataType->args[0].variant   = CFG_KEY_ATTRIBUTE;
    dataType->args[0].scan      = CfgIncludeScan;
  }

  /*-------------------------------------------------- Check doublon keywords */
  for(i=0; i<cfgAct->nb; i++){
    for(j=(i+1); j<cfgAct->nb; j++){
      /*------------------------------------ Count the number of same keyword */
      if(strcmp(cfgAct->array[j].keyword, cfgAct->array[i].keyword) == 0){
        cfgAct->array[i].index++;
      }
    }
  }

  /*------------------------------------------------ Add processing Attributs */
  cfgAct->nb += 2;

  cfgParse->cfgAction = cfgAct;

  return(cfgAct);
}

/*----------------------------------------------------------------------------*/
/*- CfgCheckCBlock -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void CfgCheckCBlock(char *token, int lineno){

  if((strstr(token, "/*") != (char *)NULL) &&
     (strstr(token, "*/") == (char *)NULL)){
    CfgParseDoPrint("CfgCheckCBlock> warning: `/*' within line comment <%d>\n",
		    lineno);
  }

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgCheckLastDataType -----------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgCheckLastDataType(CfgAction_t *cfgAct){

  if(cfgAct->previous != (CfgDataType_t *)NULL){
    CfgDataType_t *dtypep = cfgAct->previous;

    cfgAct->previous = (CfgDataType_t *)NULL;
    cfgAct->rest     = 0;

    /*------------------------------------------- Execute user's function */
    if(CfgFunctionExec(cfgAct->cfgParse, dtypep) == CFG_FAIL){
      return(CFG_FAIL);
    }
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgConfigExtSprintf ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgConfigExtSprintf(CfgParse_t *cfgParse, char *format, ...){
va_list args;
int     len    = cfgParse->config? strlen(cfgParse->config): 0;
int     nbytes = len;
char    buffer[CFG_BUFSIZE];

  if(cfgParse->extract == CfgFalse){ return(0); }

  va_start(args, format);
  nbytes += vsprintf(buffer, format, args);
  va_end(args);

  cfgParse->config = realloc(cfgParse->config, nbytes + 1);
  if(cfgParse->config == (char *)NULL){
    CfgParseDoPrint("CfgConfigExtSprintf> realloc failed");
    return(CFG_FAIL);
  }

  sprintf(&cfgParse->config[len], "%s", buffer);

  return(nbytes);
}

/*----------------------------------------------------------------------------*/
/*- CfgDataTypeFree ----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgDataType_t *CfgDataTypeFree(CfgDataType_t *cfgDataType, int nb){
register int i, j;

  if(cfgDataType == (CfgDataType_t *)NULL){
    CfgParseDoPrint("CfgDataTypeFree> Invalid argument \"nil\" pointer passed.\n");
    return(cfgDataType);
  }

  for(i=0; i<nb; i++){
    for(j=0; j<cfgDataType[i].nb; j++){
      if(cfgDataType[i].args[j].format != (char *)NULL){
        free((void *)cfgDataType[i].args[j].format);
        cfgDataType[i].args[j].format = (char *)NULL;
      }
    }
  }

  free((void *)cfgDataType);

  return((CfgDataType_t *)NULL);
}

/*----------------------------------------------------------------------------*/
/*- CfgDataTypeNew -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgDataType_t *CfgDataTypeNew(int nb){
CfgDataType_t *cfgDataType;

  cfgDataType = calloc(nb, sizeof(CfgDataType_t));
  if(cfgDataType == (CfgDataType_t *)NULL){
    CfgParseDoPrint("CfgDataTypeNew> calloc failed");
    return((CfgDataType_t *)NULL);
  }

  return(cfgDataType);
}

/*----------------------------------------------------------------------------*/
/*- CfgDefaultScan -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgDefaultScan(CfgAction_t *cfgAct, char *token, CfgArgs_t *args){
  return(0);
}

/*----------------------------------------------------------------------------*/
/*- CfgDefineScan ------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgDefineScan(CfgAction_t *cfgAct, char *token, CfgArgs_t *args){
int nbytes = 0;

  if(token[0] != '\0'){
    char key[CFG_MAX_NAME];
    char string[CFG_BUFSIZE];
    int  nc = 0;
    
    /* CfgParseDoPrint("CfgDefineScan> token:\"%s\"\n", token); */
    CfgScanToken(args->type, token, args->format, key, &nc, CFG_MAX_NAME);
    if( nc < 0 ) return(nbytes);
    nbytes = nc;
    CfgScanToken(args->type, &token[nc], args->format, string, &nc, CFG_BUFSIZE);
    if( nc < 0 ) return(nbytes);
    
    nbytes += nc;

    if( CfgDefineStackNew(cfgAct, key, string) == CFG_FAIL ) nbytes = -1;
  }

  return(nbytes);
}

/*----------------------------------------------------------------------------*/
/*- CfgDefineStackNew --------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgDefineStackNew(CfgAction_t *cfgAct, char *key, char *string){
CfgDefine_t *definep;
 int len;

  cfgAct->define.stack = realloc(cfgAct->define.stack,
                                 sizeof(CfgDefine_t) * (cfgAct->define.nb + 1));

  if(cfgAct->define.stack == (CfgDefine_t *)NULL){
    CfgParseDoPrint("CfgDefineStackNew> realloc: %s\n", strerror(errno));
    return(CFG_FAIL);
  }

  definep = &cfgAct->define.stack[cfgAct->define.nb];
  cfgAct->define.nb++;

  memset((char *)definep, 0, sizeof(CfgDefine_t));

  if( (len = strlen(key)) >= CFG_MAX_NAME ) {
    CfgParseDoPrint("CfgDefineStackNew> define name too long: %d/%d\n",
		    len, CFG_MAX_NAME);
    return(CFG_FAIL);
  } 
  strcpy(definep->key, key);

  /* if( (len = strlen(string)) >= CFG_MAX_NAME ) { */
  /*   CfgParseDoPrint("CfgDefineStackNew> define string too long: %d/%d\n", */
  /* 	    len, CFG_MAX_NAME); */
  /*   return(CFG_FAIL); */
  /* }  */
  /* strcpy(definep->string, string); */
  
  if( !(definep->string = strdup( string )) ){
   CfgParseDoPrint("CfgDefineStackNew> define string malloc failed\n" );
    return(CFG_FAIL);
  }
  
  definep->ncalls++;
  return(CFG_OK);
}


/*----------------------------------------------------------------------------*/
/*- _CfgFileToBuf ------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
char *_CfgFileToBuf(char *filename, CfgParse_t *cfgp){
FILE         *fp;
 char         *buffer, *head = NULL, *c;
CfgInfo_t    cfgInfo;
CfgParse_t   *cfgParse;
 register int size, nbytes, i, err, line, j;

 if( !filename || !filename[0] ||
     (fp = fopen(filename, "r")) == (FILE *)NULL ){
    CfgParseDoPrint("_CfgFilteToBuf> fopen(%s): %s\n", filename, strerror(errno));
    return((char *)NULL);
  }

  fseek(fp, 0, SEEK_END);

  /*---------------------------------------------- Get byte size of open file */
  nbytes = ftell(fp);
  size   = nbytes + 1;

  rewind(fp);

  if((buffer = malloc(size + sizeof(CfgInfo_t))) == (char *)NULL){
    CfgParseDoPrint("_CfgFilteToBuf> malloc failed");
    fclose(fp);
    return((char *)NULL);
  }

  memset((char *)&cfgInfo, 0, sizeof(CfgInfo_t));
  memset(buffer, 0, size);

  /*------------------------------------ Fill buffer stream from file on disk */
  fread(buffer, 1, nbytes, fp);
  
  for( c = buffer, line =  err = j = i = 0; i<size; i++, c++, j++ ) {
    if( c[0] == 10 /*LF*/ || c[0] == 13 /*CR*/ ) { 
      line++; 
      j = 0;
    }
    if( isprint(c[0]) || 
	c[0] == 10 /*LF*/ || c[0] == 13 /*CR*/ ||
	c[0] ==  9 /*HT*/ || c[0] == 0 /*NUL*/ ) continue;
    /* if( isascii(c[0]) ) continue; */
    err++;
    CfgParseDoPrint("_CfgFileToBuf>ERROR - file: %s - line_nb: %d - character_nb: %d >"
		    " value \'%c\'(%d) - %d\n", 
		    filename, line+1, j, c[0], c[0], err  );
  } 

 if(err){
   CfgParseDoPrint("_CfgFilteToBuf> %d not PRINTABLE characters present\n", err);
   free(buffer);
   fclose(fp);
   return((char *)NULL);
  }

  

  if(cfgp == (CfgParse_t *)NULL){
    char *tmpfile;

    if((cfgParse = CfgParseNew()) == (CfgParse_t *)NULL){
      free(buffer);
      return((char *)NULL);
    }

    tmpfile = strdup(filename);
    strcpy(cfgParse->dirname, CfgDirname(tmpfile));
    free(tmpfile);
  }
  else{
    cfgParse        = cfgp;
    cfgInfo.include = CfgTrue;
  }

  /*------------- Store filename into CfgInfo and append at the end of buffer */
  strcpy(cfgInfo.filename, filename);
  memcpy( &buffer[size], (char *)&cfgInfo, sizeof(CfgInfo_t));

  fclose(fp);

  if(CfgParseBuffer_r(cfgParse, buffer, &head) == CFG_FAIL){
    free(buffer);
    CfgParseFree(cfgParse);
    return((char *)NULL);
  }

  free(buffer);

  if(cfgp == (CfgParse_t *)NULL){
    size = strlen(cfgParse->buffer) + 1;

    cfgParse->buffer = realloc(cfgParse->buffer, size + sizeof(CfgInfo_t));
    if(cfgParse->buffer == (char *)NULL){
      CfgParseFree(cfgParse );
      return((char *)NULL);
    }

    cfgInfo.processed = CfgTrue;
    memcpy(&cfgParse->buffer[size], (char *)&cfgInfo, sizeof(CfgInfo_t));
    buffer = cfgParse->buffer;
    cfgParse->buffer = NULL;

    CfgParseFree(cfgParse);
  }
  else{ buffer = cfgParse->buffer; }

  return(buffer);
}

/*----------------------------------------------------------------------------*/
/*- CfgFlagScan --------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgFlagScan(CfgAction_t *cfgAct, char *token, CfgArgs_t *args){
CfgParse_t *cfgParse = cfgAct->cfgParse;

  if(cfgParse->extract == CfgFalse){
    *((CfgBool_t *)args->attribute) = CfgTrue;
  }

  return(0);
}

/*----------------------------------------------------------------------------*/
/*- _CfgFree -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void _CfgFree(CfgParse_t *cfgParse){

  if(cfgParse->next != (CfgParse_t *)NULL) {
    _CfgFree(cfgParse->next);
    cfgParse->next = (CfgParse_t *)NULL;
  }

  if(cfgParse->keyword != (char *)NULL) {
    free(cfgParse->keyword);
    cfgParse->keyword = (char *)NULL;
  }

  if(CfgError.msg != (char *)NULL){
    free(CfgError.msg);
    CfgError.msg = (char *)NULL;
    CfgError.len = 1;
  }

  free(cfgParse);

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgFunctionExec ----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgFunctionExec(CfgParse_t *cfgParse, CfgDataType_t *dataType){
register int i, rtn, idx = 0, iarg = 0, loop;
int (*cfgPtrFunc)();

  /*------------------------------------------------------------ Nothing done */
  if(cfgParse->extract == CfgTrue){ return(CFG_OK); }
  if(dataType->exec == CfgFalse){ return(CFG_OK); }

  /*------------------------------------------ Initialise pointer of function */
  cfgPtrFunc = (int (*)())dataType->args[0].attribute;

  /*--------------------------------------------- Clear user's function stack */
  memset((char *)dataType->stack.array, 0, sizeof dataType->stack.array);

  if( dataType->countArg) { /*-------------------------- Variable declaration */
    loop = dataType->nargs+1;
    if( loop > dataType->nb ) loop = dataType->nb;
  } else loop = dataType->nb;

  /*---------------------------------------------- Fill user's function stack */
  dataType->stack.array[idx++] = (Ulong)dataType->farg;
  iarg++;

  for(i=1; i<loop; i++){
    if(iarg == (dataType->iarg + 1)){
      dataType->stack.array[idx++] = (Ulong)(dataType->nargs - dataType->iarg);
      iarg++;
    }

    if(dataType->args[i].type == CfgReal){
      volatile CfgDouble_t darg;

      darg.d                       = *(double *)dataType->args[i].attribute;
      dataType->stack.array[idx++] = darg.i[0];
      dataType->stack.array[idx++] = darg.i[1];
    } else {
      dataType->stack.array[idx++] = *(Ulong *)dataType->args[i].attribute;
    }

    iarg++;
  }
/*   printf("%s nb %d, iarg %d nargs %d idx %d\n", dataType->keyword, dataType->nb, */
/*   dataType->iarg, dataType->nargs, idx ); */

  /*---------------------------------------------------- Call user's function */
  dataType->user.result = CFG_OK;
  dataType->user.index = 1;
  dataType->user.iarg_done = CfgFalse;

  if (dataType->args[0].type == CfgGetFunction) { 
    rtn  = (*cfgPtrFunc) (dataType->farg, dataType);
    rtn |= dataType->user.result;
  } else
    rtn = (*cfgPtrFunc) (dataType->stack);

  /*----------------------------------------- Free allocated argument pointer */
  for(i=1; i<loop; i++){
    switch(dataType->args[i].type) {
      case CfgString:{
        char *string = *(char **)dataType->args[i].attribute;
        if(string != (char *)NULL){ string[0] = '\0'; free(string); }
  *(char **)dataType->args[i].attribute = NULL;
        break;
      }
      case CfgReal:
        *(double *)dataType->args[i].attribute = 0.;
        break;
      default:
        *(Ulong *)dataType->args[i].attribute = 0UL;
    }
  }

  return(rtn);
}

typedef union {
  int 		 int_value;
  long int	 long_int_value;
  double 	 double_value;
  char 		*string;
} _CfgParseValue_t;

_CfgParseValue_t _CfgParseGetNextValue (CfgDataType_t *dataType,
				    CfgType_t type)
{
  _CfgParseValue_t value;
  int index;
  void *ptr;

  value.long_int_value = 0;

  if (dataType == NULL) {
    CfgParseDoPrint("CfgParseGetNextValue> NULL dataType pointer\n");
    dataType->user.result = CFG_FAIL;
    return value;
  }
  
  index = dataType->user.index;
  if (dataType->iarg + 1 != index || dataType->user.iarg_done)
    dataType->user.index++;
  else {
    /* Va_list starts here. Return its item number as CfgDec */
    dataType->user.iarg_done = CfgTrue;
    if (CfgDec != type) {
      CfgParseDoPrint("CfgParseGetNextValue> Type mismatch for VA_LIST item nbr\n");
      dataType->user.result = CFG_FAIL;
      return value;
    }
    value.int_value = dataType->nargs - dataType->iarg;
    return value; 
  }

  if (index >= dataType->nb) {
   CfgParseDoPrint("CfgParseGetNextValue> Data index out of bounds");
   dataType->user.result = CFG_FAIL;
   return value;
  } 

  if (dataType->args[index].attribute == NULL) {
    CfgParseDoPrint("CfgParseGetNextValue> Attribute #%i is a NULL pointer\n",
		    index);
    dataType->user.result = CFG_FAIL;
    return value;
  }
  
  if (dataType->args[index].type != type) {
    CfgParseDoPrint("CfgParseGetNextValue> Type mismatch for attribute #"
		    "%i (%s instead of %s [%s])\n", 
		    index, CfgTypeName[type],
		    CfgTypeName[dataType->args[index].type],
		    dataType->keyword);
    dataType->user.result = CFG_FAIL;
    return value;
  }

  ptr = dataType->args[index].attribute;
  switch (type) {
    case CfgReal: 
      value.double_value = *((double *) ptr);
      break;
    case CfgHex:
      value.long_int_value = *((long int *) ptr);
      break;
    case CfgDec:
    case CfgEnum:
      value.int_value = *((int *) ptr);
      break;
    case CfgString:
      case CfgConstString:
      value.string = *((char **) ptr);
      break;
      default:
      CfgParseDoPrint("CfgParseGetNextValue> Invalid type for attribute #%i\n",
		      index);
      dataType->user.result = CFG_FAIL;
  }

  return value;  
}

double CfgParseGetNextReal (CfgDataType_t *dataType)
{
  return _CfgParseGetNextValue (dataType, CfgReal).double_value;
}

int CfgParseGetNextEnum (CfgDataType_t *dataType)
{
  return _CfgParseGetNextValue (dataType, CfgEnum).int_value;
}

int CfgParseGetNextDec (CfgDataType_t *dataType)
{
  return _CfgParseGetNextValue (dataType, CfgDec).int_value;
}

long int CfgParseGetNextHex (CfgDataType_t *dataType)
{
  return _CfgParseGetNextValue (dataType, CfgHex).long_int_value;
}

char *CfgParseGetNextString (CfgDataType_t *dataType)
{
  return _CfgParseGetNextValue (dataType, CfgString).string;
}

char *CfgParseGetNextConstString (CfgDataType_t *dataType)
{
  return _CfgParseGetNextValue (dataType, CfgConstString).string;
}

int CfgParseGetResult (CfgDataType_t *dataType)
{
  return dataType->user.result;
}

/*----------------------------------------------------------------------------*/
/*- CfgGetLine_r -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static char *CfgGetLine_r(char *string, char **head){
char         *This, *tail = NULL;
static char  line[4096];
register int len;

  if(string != (char *)NULL){ This = string; }
  else{ This = *head; }

  if ( !This || This[0] == '\0')
  	return NULL;

  tail = strchr(This, '\n');

  line[0] = '\0';

  if (tail == NULL)
    len = strlen (This);
  else
    len = (int)(tail - This);

  strncpy(line, (char *)This, len);
  line[len] = '\0';

  This     += (len + (tail != NULL ? 1 : 0));
  *head     = This;

  return (line);
}

/*----------------------------------------------------------------------------*/
/*- CfgGetLine ---------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static char *CfgGetLine(char *string){
static char *head = (char *)NULL;

  return(CfgGetLine_r(string, &head));
}

/*----------------------------------------------------------------------------*/
/*- CfgGetEnumAttr -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgGetEnumAttr(CbfH_t *hEnum, char *key, long *datap){
CbfHEntry_t *itemp;

  if((itemp = CbfHFind(hEnum, key)) != (CbfHEntry_t *)NULL){
    *datap = (long)itemp->data;
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgGetNext ---------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static CfgParse_t *CfgGetNext(CfgParse_t *cfgParse){
CfgParse_t *This = (CfgParse_t *)NULL;

  if(cfgParse->keyword == (char *)NULL){
    This = cfgParse;
  }
  else{
    CfgParse_t *pt;

    for(pt = cfgParse; pt->next != (CfgParse_t *)NULL; pt = pt->next);

    This = malloc(sizeof(CfgParse_t));

    if(This == (CfgParse_t *)NULL){
      CfgParseDoPrint("CfgGetNext> malloc: %s\n", strerror(errno));
      return((CfgParse_t *)NULL);
    }

    memset((char *)This, 0, sizeof(CfgParse_t));

    pt->next = This;
  }

  return(This);
}

/*----------------------------------------------------------------------------*/
/*- CfgIncludeScan -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgIncludeScan(CfgAction_t *cfgAct, char *token, CfgArgs_t *args){
int nbytes = strlen(token);

  if(token[0] != '\0'){
    char tmpfile[CFG_MAXPATHLEN];
    char filename[CFG_MAXPATHLEN];
    char basename[CFG_MAXPATHLEN];
    CfgParse_t *cfgParse = cfgAct->cfgParse;

    CfgScanToken(args->type, token, args->format, tmpfile, &nbytes,
		 CFG_MAXPATHLEN);
    if( nbytes < 0 ) return(nbytes);
    strcpy(filename, strcpy(basename, tmpfile));

    if(strcmp(CfgDirname(tmpfile), ".") == 0){
      sprintf(filename, "%s/%s", cfgParse->dirname, basename);
    }

    if(_CfgFileToBuf(filename, cfgParse) != (char *)NULL){
      CfgIncludeStackNew(cfgAct, basename);
    } else { /*-------------------- To stop if the include file is not found */
      nbytes = -1;
    }
  }

  return(nbytes);
}

/*----------------------------------------------------------------------------*/
/*- CfgIncludeStackNew -------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgIncludeStackNew(CfgAction_t *cfgAct, char *filename){
CfgInclude_t *includep;

  cfgAct->include.stack = realloc(cfgAct->include.stack,
                                  sizeof(CfgInclude_t)*(cfgAct->include.nb+1));

  if(cfgAct->include.stack == (CfgInclude_t *)NULL){
    CfgParseDoPrint("CfgIncludeStackNew> realloc: %s\n", strerror(errno));
    return(CFG_FAIL);
  }

  includep = &cfgAct->include.stack[cfgAct->include.nb];
  cfgAct->include.nb++;

  memset((char *)includep, 0, sizeof(CfgInclude_t));

  strcpy(includep->filename, filename);
  includep->ncalls++;

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgItemScan --------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgItemScan(CfgAction_t *cfgAct, char *token, CfgArgs_t *args){
char buffer[CFG_BUFSIZE];
int  nbytes = 0;
CfgParse_t *cfgParse = cfgAct->cfgParse;

 CfgScanToken(args->type, token, args->format, buffer, &nbytes, CFG_BUFSIZE);
 if( nbytes < 0 ) return(nbytes);

 if(cfgParse->extract == CfgFalse){
   if(args->type == CfgBoolean){
     CfgBool_t *boolp = (CfgBool_t *)args->attribute;
     
     CfgSetBoolean(buffer, boolp);
   }

   if(args->type == CfgEnum){
     long *datap = (long *)args->attribute;
     
     CfgGetEnumAttr(cfgAct->hEnum, buffer, datap);
   }
 }
 
 return(nbytes);
}

/*----------------------------------------------------------------------------*/
/*- CfgNumericalScan ---------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgNumericalScan(CfgAction_t *cfgAct, char *token, CfgArgs_t *args){
int nbytes = 0;
CfgParse_t *cfgParse = cfgAct->cfgParse;

  if(cfgParse->extract == CfgTrue){
    char buffer[sizeof(double)];

    CfgScanToken(args->type, token, args->format, buffer, &nbytes,
		 sizeof(double));
  }
  else{
    CfgScanToken(args->type, token, args->format, (char *)args->attribute,
                 &nbytes, -1);
  }

  return(nbytes);
}

/*----------------------------------------------------------------------------*/
/*- CfgParseBuffer_r ---------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgParseBuffer_r(CfgParse_t *cfgParse, char *buffer, char **head){
char         *line, *This, *token[2];
char         *sbuf  = buffer;
register int size, lineno = 0;
CfgAction_t  *cfgAct;
CfgInfo_t    cfgInfo;

  if((cfgParse == (CfgParse_t *)NULL) || (buffer == (char *)NULL)){
    CfgParseDoPrint("CfgParseBuffer_r> Invalid argument \"nil\" pointer passed.\n");
    CfgSetErrorMsg("parsing error");
    return(CFG_FAIL);
  }

  size = strlen(buffer) + 1;

  memcpy((char *)&cfgInfo, &buffer[size], sizeof(CfgInfo_t));
  cfgParse->processed = cfgInfo.processed;

  CfgExtSprintf(cfgParse, "# <%s> =====>\n", cfgInfo.filename);

  if((cfgAct = CfgActionNew(cfgParse)) == (CfgAction_t *)NULL){
    CfgSetErrorMsg("parsing error");
    return(CFG_FAIL);
  }

  This = CfgProcessBuffer(buffer);

  if((line = CfgGetLine_r(This, head)) != (char *)NULL){
    token[0] = CfgSkipSpace(line);
    token[1] = CfgReverseSkipSpace(line);

    if(((token[0][0] == '/') && (token[0][1] == '*')) &&
       ((token[1][-1] == '*') && (token[1][0] == '/'))){

      CfgExtSprintf(cfgParse, "%s\n", line);
      CFG_DEBUG((__FUNCTION__": (1) %c%c ... %c%c (%s)\n",
                 token[0][0], token[0][1], token[1][-1], token[1][0], line));
    }

    lineno++;
    if(line[0] != '\0'){
      if(CfgParseLine(cfgAct, line, lineno) == CFG_FAIL){
        CfgSetErrorMsg("parsing error");
        free(This);
        return(CFG_FAIL);
      }
    }
    else{
      CfgExtSprintf(cfgParse, "\n");
      CfgConfigExtSprintf(cfgParse, "\n");
    }
  }

  while((line = CfgGetLine_r((char *)NULL, head)) != (char *)NULL){
    token[0] = CfgSkipSpace(line);
    token[1] = CfgReverseSkipSpace(line);

    if(((token[0][0] == '/') && (token[0][1] == '*')) &&
       ((token[1][-1] == '*') && (token[1][0] == '/'))){

      CfgExtSprintf(cfgParse, "%s\n", line);
      CFG_DEBUG((__FUNCTION__": (2) %c%c ... %c%c (%s)\n",
                 token[0][0], token[0][1], token[1][-1], token[1][0], line));
    }

    lineno++;
    if(line[0] != '\0'){
      if(CfgParseLine(cfgAct, line, lineno) == CFG_FAIL){
        CfgSetErrorMsg("parsing error");
        free(This);
        return(CFG_FAIL);
      }
    }
    else{ CfgExtSprintf(cfgParse, "\n"); }
  }

  if(CfgCheckLastDataType(cfgAct) == CFG_FAIL){
    CfgSetErrorMsg("parsing error");
    free(This);
    return(CFG_FAIL);
  }

  if(cfgParse->comment.end == CfgFalse){
    if(cfgInfo.filename[0] == '\0'){
      CfgParseDoPrint( "CfgParseBuffer_r>" 
		       "ERROR into the buffer {%p}\n"
		       "\tend of C block comment `*/' not founded <%d>\n",
		       sbuf, cfgParse->comment.lineno);
    }
    else{
      CfgParseDoPrint( "CfgParseBuffer_r>"
		       "ERROR in file \"%s\"\n"
		       "\tend of C block comment `*/' not founded <%d>\n",
		       cfgInfo.filename, cfgParse->comment.lineno);
    }

    CfgSetErrorMsg("parsing error");
    free(This);
    return(CFG_FAIL);
  }

  free(This);

  CfgExtSprintf(cfgParse, "# <===== <%s>\n", cfgInfo.filename);

  *head = (char *)NULL;

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgParseLine -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgParseLine(CfgAction_t *cfgAct, char *line, int lineno){
register int length, i, idx;
CbfHEntry_t  *itemp;
char         *token, *last;
char         keyword[CFG_BUFSIZE], extendedLine[CFG_BUFSIZE * 4];
int          nbytes;
CfgParse_t   *cfgParse = cfgAct->cfgParse;
CfgBool_t    end = CfgFalse;

  /*------------------------ Extende line for define key that can be replaced */
  strcpy(extendedLine, line);

  token = CfgSkipComment(cfgParse, extendedLine, lineno);
  if(token == (char *)NULL){ return(CFG_OK); }

  length = CfgReplaceDefine(cfgAct, token, lineno);
  idx = nbytes = 0;

  last = CfgReverseSkipSpace(token);

  if(last[0] == ';'){
    last[0] = ' ';
    length  = (int)(last - token);
    end     = CfgTrue;
  }

  do{
    CfgDataType_t *dtypep;
    int           rest;

    /*--------------- Assume that 1st word into the token line is the keyword */
    strcpy(keyword, "");
    sscanf(token, "%s%n", keyword, &nbytes);
    idx += nbytes;
    rest = 0;

    CFG_DEBUG((__FUNCTION__": token: %s\n\t\tkeyword: [%s]\n", token, keyword));

    /*---------------------------------------------- Search recorded keywords */
    dtypep = CfgSearchDataType(cfgAct, keyword);
    if(dtypep == (CfgDataType_t *)NULL){
      if(cfgAct->previous != (CfgDataType_t *)NULL){
        dtypep = cfgAct->previous;
        rest   = cfgAct->rest;
        idx    = 0;
        nbytes = 0;
        strcpy(keyword, dtypep->keyword);
      }
      else{
        if(cfgParse->match == CfgFalse){
          CfgConfigExtSprintf(cfgParse, "%s\n", token);
        }
        return(CFG_OK);
      }
    }
    else{
      if(cfgParse->match == CfgTrue){
        CfgConfigExtSprintf(cfgParse, "%s\n", token);
      }

      /*---------------------- Found a new keyword try to execute
                               user's function with previous attributs -------*/
      if(CfgCheckLastDataType(cfgAct) == CFG_FAIL){ 
	CfgParseDoPrint("CfgParseLine> ERROR - previous line  to \'%s\' - "
			"failed executing the user's function\n", line );
	return(CFG_FAIL);
      }

      dtypep->nargs = 0;
    }

    CFG_DEBUG((__FUNCTION__": process keyword: %s\n", dtypep->keyword));

    token += nbytes;

    /*-------------------------------------------------- Parse all attributes */
    for(i=rest; i < dtypep->nb; i++) {
      /*-------------------------------------------------------- Skip comment */
      token = CfgSkipComment(cfgParse, token, lineno);

      idx += cfgParse->comment.nspaces;

      CFG_DEBUG((__FUNCTION__": [i: %d/%d] token: (%s) [idx: %d/%d]\n",
                 i, dtypep->nb, token? token: "<null>", idx, length));

      if(((idx == length) &&
         (i < dtypep->nb) &&
         (dtypep->args[i].variant != CFG_KEY)) ||
         (token == (char *)NULL)){

        if(end == CfgFalse){
          CFG_DEBUG((__FUNCTION__": keep info of keyword: %s [%d]\n",
                     dtypep->keyword, i));
          cfgAct->previous = dtypep;
          cfgAct->rest     = i;
          return(CFG_OK);
        }
        else{ break; }
      }

      /*----------------------------------------------- Get value of attribut */
      nbytes = dtypep->args[i].scan(cfgAct, token, &dtypep->args[i]);
      if(nbytes == -1){
	CfgParseDoPrint("CfgParseLine> line \'%s\'- "
			"failed extracting the input parameters\n", line );
	return(CFG_FAIL); 
      }
      if((nbytes > 0) && (dtypep->countArg == CfgTrue)){ dtypep->nargs++; }

      token += nbytes;
      idx   += nbytes;

      CFG_DEBUG((__FUNCTION__": (1) [%d/%d] idx: %d/%d, nbytes: %d\n",
                 i, dtypep->nb, idx, length, nbytes));
    }

    /*----------------------------------------------- Execute user's function */
    if(CfgFunctionExec(cfgParse, dtypep) == CFG_FAIL){ 
      CfgParseDoPrint("CfgParseLine> ERROR - line \'%s\'- "
		      "failed executing the user's function\n", line );
      return(CFG_FAIL);
    }

    dtypep->nargs    = 0;
    cfgAct->previous = (CfgDataType_t *)NULL;
    cfgAct->rest     = 0;

    /*------------------------------------------ Update doublon keyword index */
    if((itemp = CbfHFind(cfgAct->hTable, keyword)) != (CbfHEntry_t *)NULL){
      if((long)itemp->data > 1){ itemp->data--; }
    }

    /*---------------------------- Try to remove comment between two keywords */
    token = CfgSkipComment(cfgParse, token, lineno);
    if(token == (char *)NULL){ return(CFG_OK); }

    idx += cfgParse->comment.nspaces;

    CFG_DEBUG((__FUNCTION__": (2) [%d/%d] idx: %d/%d, nbytes: %d\n",
               i, dtypep->nb, idx, length, nbytes));

    CfgReplaceDefine(cfgAct, token, lineno);

  }while(idx < length);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgProcessBuffer ---------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static char *CfgProcessBuffer(char *buffer){
register char *sptr, *nbuf = (char *)NULL, *obuf = buffer;
register int bsize = strlen(buffer) + 1;
register int size = bsize + sizeof(CfgInfo_t);
CfgInfo_t cfgInfo;

  if((nbuf = (char *)calloc(1, size)) == (char *)NULL){
    return(buffer);
  }

  memcpy((char *)&cfgInfo, &buffer[bsize], sizeof(CfgInfo_t));

  do{
    if(((sptr = strchr(obuf, '\\')) != (char *)NULL) && (sptr[1] != '"')){
      long nbytes = (long)(sptr - obuf);

      /*----- Previous character is not a white space, replace '\' by a space */
      if(sptr[-1] != ' '){
        sptr[0] = ' ';
        nbytes++;
      }

      strncat(nbuf, obuf, nbytes);
      obuf = CfgSkipSpace(++sptr);
    }
    else{
      if(sptr != (char *)NULL){
        long nbytes = (long)(sptr - obuf) + 1;

        strncat(nbuf, obuf, nbytes);
        obuf = ++sptr;
      }
    }
  }while(sptr != (char *)NULL);

  if(obuf != (char *)NULL){ strcat(nbuf, obuf); }

  bsize = strlen(nbuf) + 1;

  memcpy(&nbuf[bsize], (char *)&cfgInfo, sizeof(CfgInfo_t));

  return(nbuf);
}

/*----------------------------------------------------------------------------*/
/*- CfgReplaceDefine ---------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgReplaceDefine(CfgAction_t *cfgAct, char *token, int lineno){
register int i;
CfgParse_t *cfgParse = cfgAct->cfgParse;

  if( token == strstr(token, "define") || token == strstr(token, "include") 
      ) {
    CfgExtSprintf(cfgParse,
                  "# line <%d> commented out by `CfgParse' preprocessor\n"
                  "# %s\n\n",
                  lineno, token);
  } else if(token[0] != '\0') {
    for(i=cfgAct->define.nb-1; i>=0; i--){
      char *This;
      char *sptr = token;

      do{
        if((This = strstr(sptr, cfgAct->define.stack[i].key)) != (char *)NULL){
          char string[CFG_BUFSIZE * 4];
          int  size = (int)(This - token);
          int  len  = strlen(cfgAct->define.stack[i].key);
	  /*---------------------------------- Replace define key by a string */
          strncpy(string, token, size);
          string[size] = '\0';
          strcat(string, cfgAct->define.stack[i].string);
          strcat(string, &token[size + len]);
          strcpy(token, string);
          sptr = token + size;
        }
      } while(This != (char *)NULL);
    }

    CfgExtSprintf(cfgParse, "%s\n", token);
  }

  return(strlen(token));
}

/*----------------------------------------------------------------------------*/
/*- CfgReverseSkipSpace ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static char *CfgReverseSkipSpace(char *string){
register int i, len = strlen(string) - 1;

  /*------------------------------ Skip space, tab, vertical formfeed, etc... */
  for(i=len; (isspace((int)string[i]) != 0) && (i >= 0); i--);

  return(&string[i]);
}

/*----------------------------------------------------------------------------*/
/*- CfgScanToken -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgScanToken(CfgType_t type, char *token, char *format,
                        char *string, int *nbytes, int maxsize ){

  if((type == CfgString)||(type == CfgConstString)||(type == CfgProcessing)){
    int  len, idx;
    char *dummy, *pstr = (char *)NULL;
    CfgBool_t dquote = CfgFalse;

    /*-------------------------------------------------- Skip spaces and tabs */
    dummy = CfgSkipSpace(token);
    idx   = 0;

    if((dummy[idx] == '\\') && (dummy[idx+1] == '"')){
      idx   += 2;
      dquote = CfgTrue;
      pstr   = strstr(&dummy[idx], "\\\"");
    }
    else{
      if(dummy[idx] == '"'){
        idx++;
        pstr = index(&dummy[idx], '"');
      }
    }

    if(pstr != (char *)NULL){
      /*-------------------------------- Get size for extracting the string */
      len = (int)(pstr - dummy) - !dquote;
      if( maxsize > 0 && len >= maxsize ) {
 	*nbytes = -1;
	CfgParseDoPrint("CfgScanToken> ERROR -type %s - failed: "
			"size too long for \"%s\"\n",
			CfgTypeName[type], token );
	return(CFG_OK);
     }
      if(dquote == CfgTrue){
        string[0] = '"';
        strncpy(&string[1], &dummy[idx], len - idx);
        string[len-1] = '"';
      }
      else{ strncpy(string, &dummy[idx], len); }
      string[len] = '\0';

      /*-------------------------------------- Compute number of bytes read */
      if( maxsize > 0 && (len+2) >= maxsize ) {
  	*nbytes = -1;
	CfgParseDoPrint("CfgScanToken> ERROR -type %s - failed: "
			"size too long for \"%s\"\n",
			CfgTypeName[type], token );
      } else { *nbytes = (int)(dummy - token) + len + 2; }
      return(CFG_OK);
    }
    else{ 
      sscanf(token, format, string, nbytes);
      if(  maxsize > 0 && strlen(string) >= maxsize ) { 
	*nbytes = -1;
	CfgParseDoPrint("CfgScanToken> ERROR -type %s - failed: "
			"size too long for \"%s\"\n",
			CfgTypeName[type], token );
	return(CFG_OK);
      }
    }
  }
  else { 
    sscanf(token, format, string, nbytes);
    if(  maxsize > 0 &&  strlen(string) >= maxsize ) { 
      *nbytes = -1;
      CfgParseDoPrint("CfgScanToken> ERROR - type %s - failed: "
		      "size too long for \"%s\"\n",
		      CfgTypeName[type], token );
      return(CFG_OK);
    }
    /* Ignore trailing characters until next space or tab or end of string */
    while (token[*nbytes] != '\0' &&
	   !isspace (token[*nbytes]))
	   (*nbytes)++;
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgSearchDataType --------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgDataType_t *CfgSearchDataType(CfgAction_t *cfgAct, char *keyword){
register int i;
CbfHEntry_t  *itemp;

  for(i=0; i<cfgAct->nb; i++){
    CfgDataType_t *dtypep = &cfgAct->array[i];

    if(strcmp(dtypep->keyword, keyword) == 0){
      if((strcmp(keyword, "define") != 0) && (strcmp(keyword, "include") != 0)){
        /*-------------------------------------------- Record doublon keyword */
        if((itemp = CbfHFind(cfgAct->hTable, keyword)) == (CbfHEntry_t *)NULL){
          itemp = CbfHEnter(cfgAct->hTable, keyword, (void *)dtypep->index);
          if(itemp == (CbfHEntry_t *)NULL){
            CfgParseDoPrint("CfgSearchDataType> CbfHEnter: %s\n",
			    strerror(errno));
          }
        }
        else{
          if(((long)itemp->data > 0) && ((long)itemp->data != dtypep->index)){
            continue;
          }
        }
      }
      return(&cfgAct->array[i]);
    }
  }

  return((CfgDataType_t *)NULL);
}

/*----------------------------------------------------------------------------*/
/*- CfgSearchKey -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgSearchKey(CfgParse_t *cfgParse, char *keyword){
CfgParse_t *This;

  for(This = cfgParse; This != (CfgParse_t *)NULL; This =This->next){
    if(strcmp(This->keyword, keyword) == 0){ return(CFG_OK); }
  }

  return(CFG_FAIL);
}

/*----------------------------------------------------------------------------*/
/*- CfgSetBoolean  -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgSetBoolean(char *word, CfgBool_t *boolp){
register int i, size;
/* char         buffer[CFG_BUFSIZE]; */
char        *buffer = NULL;

  size = strlen(word);
  if( !(buffer = (char*)malloc(size)) ) return(CFG_FAIL);
  /*-------------------------------------------- Convert string to lower case */
  for(i=0; i<strlen(word); i++){ buffer[i] = (char)tolower((int)word[i]); }
  buffer[i] = '\0';

  /*--------------------------------------------------- Set the correct value */
  if((strcmp(buffer, "true") == 0) ||
     (strcmp(buffer, "yes") == 0)  ||
     (strcmp(buffer, "on") == 0)){
    *boolp = CfgTrue;
  }
  if((strcmp(buffer, "false") == 0) ||
     (strcmp(buffer, "no") == 0)    ||
     (strcmp(buffer, "off") == 0)){
    *boolp = CfgFalse;
  }
  /*--------------------------------------------------------------------------*/
  if( buffer )free(buffer);
  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgSetLineComment --------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void CfgSetLineComment(char *token, CfgComment_t *cmt, int lineno){

  if(cmt->C_block == CfgFalse){
    cmt->start = CfgTrue;
    cmt->end   = CfgTrue;
    CfgCheckCBlock(token, lineno);
  }

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgSkipComment -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static char *CfgSkipComment(CfgParse_t *cfgParse, char *line, int lineno){
CfgComment_t *cmt = &cfgParse->comment;
char *token;

  if(cmt->end == CfgTrue){
    cmt->start   = CfgFalse;
    cmt->C_block = CfgFalse;
  }

  cmt->nspaces = 0;
  token        = CfgSkipSpace(line);
  cmt->nspaces = (int)(token - line);

  switch(token[0]){
    case '/':
      if(token[1] == '/'){ CfgSetLineComment(token, cmt, lineno); }
      if(token[1] == '*'){
        if(cmt->start == CfgTrue){
          CfgParseDoPrint( "CfgSkipComment>"
			   " warning: `/*' within C block comment <%d>\n", lineno);
        }
        cmt->start   = CfgTrue;
        cmt->end     = CfgFalse;
        cmt->C_block = CfgTrue;
        cmt->lineno  = lineno;
        cmt->count++;
      }
      break;
    case '-':
      if(token[1] == '-'){ CfgSetLineComment(token, cmt, lineno); }
      break;
    case '#':
      CfgSetLineComment(token, cmt, lineno);
      break;
  }

  if(cmt->start == CfgTrue){
    if(cmt->C_block == CfgTrue){
      char *endCmt, *nextWord;

      if(((strstr(token, "/*") == (char *)NULL) && (cmt->lineno != lineno)) ||
         (cmt->lineno == lineno)){
        if((endCmt = strstr(token, "*/")) != (char *)NULL){
          cmt->count--;
          if(cmt->count == 0){ cmt->end = CfgTrue; }
          if(cmt->lineno == lineno){
            nextWord      = CfgSkipSpace(&endCmt[2]);
            cmt->nspaces += nextWord? (int)(nextWord - token): 0;
            return(nextWord);
          }
        }
      }
    }

    CfgExtSprintf(cfgParse, "%s\n", line);

    return((char *)NULL);
  }

  return(token);
}

/*----------------------------------------------------------------------------*/
/*- CfgSkipSpace -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static char *CfgSkipSpace(char *string){
register int i, len = strlen(string);

  /*------------------------------ Skip space, tab, vertical formfeed, etc... */
  for(i=0; (isspace((int)string[i]) != 0) && (i < len); i++);

  return(&string[i]);
}

/*----------------------------------------------------------------------------*/
/*- CfgStringScan ------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgStringScan(CfgAction_t *cfgAct, char *token, CfgArgs_t *args){
char buffer[CFG_BUFSIZE];
int  nbytes = 0;
char **string = (char **)args->attribute;
CfgParse_t *cfgParse = cfgAct->cfgParse;

  *string = (char *)NULL;

  if(token[0] != '\0'){
    CfgScanToken(args->type, token, args->format, buffer, &nbytes, CFG_BUFSIZE);
    if(nbytes >= 0 && cfgParse->extract == CfgFalse){
      if(args->type == CfgString){ *string = strdup(buffer); }
      if(args->type == CfgConstString){ strcpy((char *)args->attribute, buffer); }
    }
  }

  return(nbytes);
}

/*----------------------------------------------------------------------------*/
/*- CfgExtSprintf ------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgExtSprintf(CfgParse_t *cfgParse, char *format, ...){
va_list args;
int     len    = cfgParse->buffer? strlen(cfgParse->buffer): 0;
int     nbytes = len;
char    buffer[CFG_BUFSIZE];

  if(cfgParse->processed == CfgTrue){ return(0); }

  va_start(args, format);
  nbytes += vsprintf(buffer, format, args);
  va_end(args);

  cfgParse->buffer = realloc(cfgParse->buffer, nbytes + 1);
  if(cfgParse->buffer == (char *)NULL){
    CfgParseDoPrint("CfgExtSprintf> realloc failed");
    return(CFG_FAIL);
  }

  sprintf(&cfgParse->buffer[len], "%s", buffer);

  return(nbytes);
}

/******************************************************************************/
/* PUBLIC FUNCTIONS                                                           */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*- CfgBufToFile -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgBufToFile(char *buffer, char *filename){
FILE *fp;
char *body, *line, *head;
char file[CFG_MAXPATHLEN] = "";
CfgBool_t newFile = CfgFalse;
register int idx;

  if((fp = fopen(filename, "w")) == (FILE *)NULL){
    CfgParseDoPrint("CfgBufToFile> %s fopen error", filename);
    return(CFG_FAIL);
  }

  body = head = (char *)NULL;

  line = CfgGetLine_r(buffer, &head);
  idx  = strlen(line) - 9;

  if(!strstr(line, "# <") || !strstr(&line[idx], "> =====>")){
    fprintf(fp, "%s\n", line);
  }

  while((line = CfgGetLine_r((char *)NULL, &head)) != (char *)NULL){
    idx = strlen(line) - 9;

    if(strstr(line, "# <") && strstr(&line[idx], "> =====>")){
      sscanf(line, "# <%[a-zA-Z0-9./_-]> =====>", file);
      newFile = CfgTrue;
      continue;
    }

    if(newFile == CfgTrue){
      int rtn;

      idx = strlen(line) - 1;

      if(strstr(line, "# <===== <") && (line[idx] == '>')){
        char *tmpfile[2];
        char pathname[CFG_MAXPATHLEN];

        sscanf(line, "# <===== <%[a-zA-Z0-9./_-]>", file);

        tmpfile[0] = strdup(filename);
        tmpfile[1] = strdup(file);

        if(strcmp(CfgDirname(tmpfile[1]), ".") == 0){
          sprintf(pathname, "%s/%s", CfgDirname(tmpfile[0]), file);
        }
        else{
         strcpy(pathname, file);
        }

        rtn = CfgBufToFile(body, pathname);

        free(tmpfile[0]);
        free(tmpfile[1]);
        free(body);
        body    = (char *)NULL;
        newFile = CfgFalse;

        if(rtn == CFG_FAIL){ return(CFG_FAIL); }
        continue;
      }

      if(strncmp(line, "# include", 9) || strncmp(line, "# line <", 8)){
        int len    = body? strlen(body): 0;
        int nbytes = len + strlen(line) + 1;

        if((body = realloc(body, nbytes + 1)) == (char *)NULL){
          CfgParseDoPrint("CfgBufToFile> realloc failed ");
          return(CFG_FAIL);
        }

        sprintf(&body[len], "%s\n", line);
      }
    }
    else{
      idx = strlen(line) - 1;

      if(strncmp(line, "# line <", 8) &&
        (!strstr(line, "# <===== <") || (line[idx] != '>'))){
        if(!strncmp(line, "# include", 9)){ fprintf(fp, "%s", &line[2]); }
        else{ fprintf(fp, "%s\n", line); }
      }
    }
  }

  fclose(fp);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgDbGet -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
char *CfgDbGet(int nb, void *object, void *(*extractType)(),
               char *(*extractItem)()){
register int i, j, size, rc;
char         *item, *buffer = (char *)NULL;
void         *type;
CfgInfo_t    cfgInfo;

  if(object == (char **)NULL){
    CfgParseDoPrint("CfgDbGet> Invalid argument \"nil\" pointer passed.\n");
    return((char *)NULL);
  }

  for(i=0, rc=0, size=1; i<nb; i++){
    type  = extractType(object, i);
    item  = extractItem(type);
    size += strlen(item) + 1;

    if((buffer = realloc(buffer, size)) == (char *)NULL){
      CfgParseDoPrint("CfgDbGet> realloc failed");
      return((char *)NULL);
    }

    for(j=0; j<strlen(item); j++){
      if(item[j] == '?'){ item[j] = '"'; }
    }

    rc += sprintf(&buffer[rc], "%s\n", item);
  }

  memset((char *)&cfgInfo, 0, sizeof(CfgInfo_t));
  strcpy(cfgInfo.filename, "from Db");
  cfgInfo.processed = CfgTrue;
  size              = strlen(buffer) + 1;

  if((buffer = realloc(buffer, size + sizeof(CfgInfo_t))) == (char *)NULL){
    return((char *)NULL);
  }

  memcpy( &buffer[size], (char *)&cfgInfo, sizeof(CfgInfo_t));

  return(buffer);
}

/*----------------------------------------------------------------------------*/
/*- CfgDbPut -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgDbPut(char *buffer, char *name, void (*record)()){
register int i;
char *object = (char *)NULL, *line;


  if(buffer == (char *)NULL){
    CfgParseDoPrint("CfgDbPut> Invalid argument \"nil\" pointer passed.\n");
    return(CFG_FAIL);
  }

  if((line = CfgGetLine(buffer)) != (char *)NULL){
    int  rc = 0, size = strlen(name) + strlen(line) + CFG_TEXT_LEN + 1;

    if((object = realloc(object, size)) == (char *)NULL){
      CfgParseDoPrint("CfgDbPut> realloc failed");
      free(object);
      return(CFG_FAIL);
    }

    for(i=0; i<strlen(line); i++){
      if(line[i] == '"'){ line[i] = '?'; }
    }

    rc += sprintf(&object[rc], "string[] %s =\n{\n\"%s\"\n", name, line);

    while((line = CfgGetLine((char *)NULL)) != (char *)NULL){

      size += strlen(line) + 3;

      if((object = realloc(object, size)) == (char *)NULL){
        CfgParseDoPrint("CfgDbPut> realloc failed");
        free(object);
        return(CFG_FAIL);
      }

      for(i=0; i<strlen(line); i++){
        if(line[i] == '"'){ line[i] = '?'; }
      }

      rc += sprintf(&object[rc], "\"%s\"\n", line);
    }

    strcat(object, "}\n");
  }

  record(object);

  free(object);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgFileToBuf -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
char *CfgFileToBuf(char *filename){
  return(_CfgFileToBuf(filename, (CfgParse_t *)NULL));
}

/*----------------------------------------------------------------------------*/
/*- CfgMapFile ---------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
char *CfgMapFile(char *filename){
register int nbytes;
FILE         *fp;
char         *buffer;

  if((fp = fopen(filename, "r")) == (FILE *)NULL){
    CfgParseDoPrint("CfgMapFile> fopen(%s): %s\n", filename, strerror(errno));
    return((char *)NULL);
  }

  /*---------------------------------------------- Get byte size of open file */
  fseek(fp, 0, SEEK_END);
  nbytes = ftell(fp);
  rewind(fp);

  /*--------------------------------------------------- Allocate memory space */
  if((buffer = malloc(nbytes + 1)) == (char *)NULL){
    CfgParseDoPrint("CfgMapFile> malloc failed");
    fclose(fp);
    return((char *)NULL);
  }

  memset(buffer, 0, (nbytes + 1));

  /*------------------------------------ Fill buffer stream from file on disk */
  fread(buffer, 1, nbytes, fp);

  fclose(fp);

  return(buffer);
}

/*----------------------------------------------------------------------------*/
/*- CfgParseAdd --------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgParseAdd(CfgParse_t *cfgParse, char *key, int nb, ...){
va_list      ap;
CfgParse_t        *This;
register int i;

  if(cfgParse == (CfgParse_t *)NULL){
    CfgParseDoPrint("CfgParseAdd> invalid pointer argument\n");
    return(CFG_FAIL);
  }

  if(nb > CFG_MAX_ATTR){
    CfgParseDoPrint( "CfgParseAdd> "
		     "number of parameter `%d' exeeded the maximum %d\n",
		     nb, CFG_MAX_ATTR);
    return(CFG_FAIL);
  }

  if((This = CfgGetNext(cfgParse)) == (CfgParse_t *)NULL){ return(CFG_FAIL); }

  This->keyword = strdup(key);
  This->nb      = nb;

  va_start(ap, nb);

  for(i=0; i<nb;i++){
    This->dargs[i].type      = va_arg(ap, CfgType_t);
    This->dargs[i].attribute = va_arg(ap, void *);
  }

  va_end(ap);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgParseArg --------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgParseArg(CfgParse_t *cfgParse, int argc, char *argv[]){
register int i, length = 0;
char         *buffer = (char *)NULL;

  if(argc == 1){
    return(CFG_NO_ARG);
  }
  else{
    for(i=1; i<argc; i++){
      length += (strlen(argv[i]) + 1);
      buffer  = realloc(buffer, length);

      if(CfgSearchKey(cfgParse, argv[i]) == CFG_OK){
        if(i > 1){ strcat(buffer, "\n"); }
        else{ buffer[0] = '\0'; }
      }
      else{ strcat(buffer, " "); }

      strcat(buffer, argv[i]);
    }

    buffer[length-1] = '\0';

    if(CfgParseBuffer(cfgParse, buffer) == CFG_FAIL){
      free(buffer);
      return(CFG_FAIL);
    }

    free(buffer);
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgParseBuffer -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgParseBuffer(CfgParse_t *cfgParse, char *buffer){
  return(CfgParseBuffer_r(cfgParse, buffer, &cfgParse->head));
}

/*----------------------------------------------------------------------------*/
/*- CfgParseExtract ----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
char *CfgParseExtract(CfgParse_t *cfgParse, char *buffer, CfgBool_t match){
register int rtn;

  cfgParse->match   = match;
  cfgParse->extract = CfgTrue;
  if( cfgParse->config ) { 
    free( cfgParse->config );
    cfgParse->config  = NULL;
  }
  rtn               = CfgParseBuffer(cfgParse, buffer);
  cfgParse->extract = CfgFalse;

  if(rtn == CFG_FAIL){ return((char *)NULL); }

  return(cfgParse->config);
}

/*----------------------------------------------------------------------------*/
/*- CfgParseFile -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgParseFile(CfgParse_t *cfgParse, char *filename){
char *buffer;

  if((buffer = CfgFileToBuf(filename)) == (char *)NULL){ return(CFG_FAIL); }

  if(CfgParseBuffer(cfgParse, buffer) == CFG_FAIL){
    free(buffer);
    return(CFG_FAIL);
  }

  free(buffer);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgParseFree -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgParse_t *CfgParseFree(CfgParse_t *cfgParse){

  if(cfgParse == (CfgParse_t *)NULL){
    CfgParseDoPrint("CfgParseFree> invalid pointer argument\n");
    return((CfgParse_t *)NULL);
  }

  if(!CbfCleanFind("CfgParseFree", (void *)cfgParse)){
    return((CfgParse_t *)NULL);
  }

  if(CbfCleanDone("CfgParseFree", (void *)cfgParse) == CFG_EXIT){
    return((CfgParse_t *)NULL);
  }

  CfgActionFree(cfgParse->cfgAction);

  if( cfgParse->config ) {
    free(cfgParse->config ); cfgParse->config = NULL; 
  }

  if( cfgParse->buffer ) {
    free(cfgParse->buffer); cfgParse->buffer = NULL; 
  }

  cfgParse->hEnum = CbfHFree(cfgParse->hEnum);

  _CfgFree(cfgParse);

  return((CfgParse_t *)NULL);
}

/*----------------------------------------------------------------------------*/
/*- CfgParseFunctionAdd ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int _VaCfgParseFunctionAdd(CfgParse_t *cfgParse, char *key, 
			  int (*function)(), CfgBool_t UseStack,
			  void *arg, int nb, va_list ap){
CfgParse_t   *This;
register int i, idata, index;
register int iarg, narg;

  if(cfgParse == (CfgParse_t *)NULL){
    CfgParseDoPrint("_VaCfgParseFunctionAdd> invalid pointer argument\n");
    return(CFG_FAIL);
  }

  if((nb + 1) > CFG_MAX_ATTR){
    CfgParseDoPrint( "_VaCfgParseFunctionAdd>"
		     " number of parameter `%d' exeeded the maximum %d\n",
		     (nb + 1), CFG_MAX_ATTR);
    return(CFG_FAIL);
  }

  if((This = CfgGetNext(cfgParse)) == (CfgParse_t *)NULL){ return(CFG_FAIL); }

  iarg                     = (-nb & 0xFFFF0000) >> 16;
  narg                     = -nb & 0x0000FFFF;
  This->keyword            = strdup(key);
  This->countArg           = (nb < 0)? CfgTrue: CfgFalse;
  This->iarg               = iarg - 1;
  This->farg               = arg;

  This->nb = 1;
  This->dargs[0].type      = UseStack ? CfgFunction : CfgGetFunction;
  This->dargs[0].attribute = (void *)function;


  if( nb > 0 ) { /*---------------------------------------- Standard Function */
    for(i=1, idata=0; (i<nb+1 ) && (idata<CFG_MAX_ATTR); i++){
      This->dargs[i].type = va_arg(ap, CfgType_t);
      This->dargs[i].attribute = (void *)&This->dummy[idata];
      if(This->dargs[i].type == CfgReal){ idata += 2; }
      else{ idata += 1; }
      This->nb++;
    }
  } else {       /*----------------------------------------- VA_LIST Function */
    for(i=1, idata=0, index=0; (i<CFG_MAX_ATTR) && (idata<CFG_MAX_ATTR); i++){
      if(i <= ((iarg - 1) + narg)) {
  This->dargs[i].type = va_arg(ap, CfgType_t);
  This->nb++;
      } else {
  This->dargs[i].type = This->dargs[iarg + index].type;
  index               = (index + 1) % narg;
  if( index == 0 ) This->nb += narg;
      }
      This->dargs[i].attribute = (void *)&This->dummy[idata];
      if(This->dargs[i].type == CfgReal){ idata += 2; }
      else{ idata += 1; }
    }
  }

  return(CFG_OK);
}

int CfgParseFunctionAdd(CfgParse_t *cfgParse, char *key, int (*function)(),
                        void *arg, int nb, ...)
{
  va_list va_args;
  int result;

  va_start (va_args, nb);
  result = _VaCfgParseFunctionAdd (cfgParse, key, function, CfgTrue, arg, nb,
				   va_args);
  va_end (va_args);

  return result;
}

/*----------------------------------------------------------------------------*/
/*- CfgParseGetFunctionAdd -------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgParseGetFunctionAdd(CfgParse_t *cfgParse, char *key, int (*function)(),
			   void *arg, int nb, ...)
{
  va_list va_args;
  int result;

  va_start (va_args, nb);
 result = _VaCfgParseFunctionAdd (cfgParse, key, function, CfgFalse, arg, nb,
				  va_args);
  va_end (va_args);

  return result;
}

/*----------------------------------------------------------------------------*/
/*- CfgParseNew --------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgParse_t *CfgParseNew(void){
CfgParse_t *cfgParse;
const CfgComment_t comment_initializer = {CfgFalse, CfgTrue, CfgFalse, 0, 0, 0};

#ifdef USE_MW
  /*------------------------------------ Collect stats on a line number basis */
  mwStatistics(MW_STAT_LINE);
#endif /* USE_MW */

  cfgParse = malloc(sizeof(CfgParse_t));

  if(cfgParse == (CfgParse_t *)NULL){
    CfgParseDoPrint("CfgParseNew>: malloc: %s\n", strerror(errno));
    return((CfgParse_t *)NULL);
  }

  memset((char *)cfgParse, 0, sizeof(CfgParse_t));

  cfgParse->comment = comment_initializer;
  cfgParse->match   = CfgUnknown;
  cfgParse->hEnum   = CbfHNew(CFG_MAX_HENUM);

  if(cfgParse->hEnum == (CbfH_t *)NULL){
    free((void *)cfgParse);
    return((CfgParse_t *)NULL);
  }

  CfgError.len = 1;

  CbfCleanAddRoot( "CfgParseFree", (CbfCleanFnt_t)CfgParseFree, cfgParse);

  return(cfgParse);
}

/*----------------------------------------------------------------------------*/
/*- CfgParseSetEnumAttr ------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgParseSetEnumAttr(CfgParse_t *cfgParse, char *key, long data){
CbfHEntry_t *itemp;

  if(cfgParse == (CfgParse_t *)NULL){
    CfgParseDoPrint("CfgParseSetEnumAttr> invalid pointer argument\n");
    return(CFG_OK);
  }

  if((itemp = CbfHFind(cfgParse->hEnum, key)) == (CbfHEntry_t *)NULL){
    if(CbfHEnter(cfgParse->hEnum, key, (char *)data) == (CbfHEntry_t *)NULL){
       CfgParseDoPrint("CfgParseSetEnumAttr> CbfHEnter: %s\n", strerror(errno));
      return(CFG_FAIL);
    }
  }
  else{ itemp->data = (char *)data; }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgPrintError ------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
char *CfgPrintError(void){

  if(CfgError.len == 1){
    if((CfgError.msg = strdup("No error occured")) == (char *)NULL){
      CfgParseDoPrint( "CfgPrintError> strdup: %s\n", strerror(errno));
      return((char *)NULL);
    }
  }

  CfgParseDoPrint( "%s\n", CfgError.msg);

  return(CfgError.msg);
}

/*----------------------------------------------------------------------------*/
/*- CfgSetErrorMsg -----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgSetErrorMsg(char *format, ...){
va_list args;
char    message[2048];
int     size;

  va_start(args, format);

  size = vsprintf(message, format, args);

  CfgError.msg = realloc(CfgError.msg, CfgError.len + size);

  if(CfgError.msg == (char *)NULL){
    CfgParseDoPrint( "CfgSetErrorMsg> realloc: %s\n", strerror(errno));
    return(CFG_FAIL);
  }

  strcpy((char *)&CfgError.msg[CfgError.len - 1], message);

  CfgError.len                  += size;
  CfgError.msg[CfgError.len - 1] = '\0';

  va_end(args);

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*---------------------------------------------------- CfgParseInstallPrinter */
/*----------------------------------------------------------------------------*/
CfgParsePrinter_t CfgParseInstallPrinter( CfgParsePrinter_t printer ) {
  CfgParsePrinter_t old;

  /*--------------------------------------------------------------------------*/
  old = CfgParsePrinter;
  CfgParsePrinter = printer;
  /*--------------------------------------------------------------------------*/
  return(old);
}

