/* SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Cfg - A library for Virgo process implementation
 *
 * Copyright © 2001-2021 Laboratoire de physique des particules d'Annecy - CNRS
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Fatih Bellachia <fatih.bellachia@lapp.in2p3.fr>
 *   Laurent Fournier <laurent.fournier@lapp.in2p3.fr>
 *   Alain Masserot <alain.masserot@lapp.in2p3.fr>
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sched.h>
#include <pthread.h>
#include <errno.h>
#include <sys/time.h>
#include <signal.h>
#include <CfgQueue.h>

#ifndef pthread_mutexattr_default
#define pthread_mutexattr_default (const pthread_mutexattr_t *)NULL
#endif /* pthread_mutexattr_default */

static CfgQueuePrinter_t CfgQueuePrinter = (CfgQueuePrinter_t)vprintf;

/******************************************************************************/
/*                       PRIVATE FUNCTIONS PROTOTYPES                         */
/******************************************************************************/
static void CfgQueueAlarm(int);
static int  CfgQueuePrint(char *format, ...);
static int  CfgQueueSemWait(sem_t *, double);

/******************************************************************************/
/*                           PRIVATE FUNCTIONS                                */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*- CfgQueueAlarm ------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static void CfgQueueAlarm(int signo){

  CFG_DEBUG(("CfgQueueTimeout: Timeout expired...\n"));

  return;
}

/*----------------------------------------------------------------------------*/
/*- CfgQueuePrint ------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgQueuePrint(char *format, ...){
va_list args;

  if(CfgQueuePrinter){
    va_start(args, format);
    CfgQueuePrinter(format, args);
    va_end(args);
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgQueueSemWait ----------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
static int CfgQueueSemWait(sem_t *semp, double timeout){
struct sigaction act, oact;
struct itimerval itimer;
volatile int rtn = CFG_OK;

  errno = 0;

  if(timeout > 0.){
    /*---------------------------------------------- Setup the signal handler */
    sigemptyset(&act.sa_mask);
    act.sa_handler = CfgQueueAlarm;
    act.sa_flags   = 0;
    if(sigaction(SIGALRM, &act, &oact) == -1) { return(CFG_FAIL); }

    /*----------------------------------------------------------- Start Timer */
    memset((char *)&itimer, 0, sizeof(struct itimerval));
    itimer.it_value.tv_sec  = (time_t)timeout;
    itimer.it_value.tv_usec = (time_t)((timeout - (time_t)timeout) * 1e6);

    if(setitimer(ITIMER_REAL, &itimer, (struct itimerval *)NULL) == -1){
      return(CFG_FAIL);
    }
  }

  if(timeout >= 0.){
    /*---------------------------------------------- Perform a semaphore lock */
    rtn = sem_wait(semp);
  }

  if(timeout > 0.){
    sigset_t a_mask;

    /*----------------------------------------------------- Mask signal ALARM */
    sigemptyset(&a_mask);
    sigaddset(&a_mask, SIGALRM);
    sigprocmask(SIG_SETMASK, &a_mask, (sigset_t *)NULL);

    /*------------------------------------------------------------ Stop Timer */
    memset((char *)&itimer, 0, sizeof(struct itimerval));

    if(setitimer(ITIMER_REAL, &itimer, (struct itimerval *)NULL) == -1){
      return(CFG_FAIL);
    }

    /*--------------------------------------------- Restore old signal action */
    if(sigaction(SIGALRM, &oact, (struct sigaction *)NULL) == -1) {
      return(CFG_FAIL);
    }

    /*--------------------------------------------------- Unmask signal ALARM */
    sigdelset(&a_mask, SIGALRM);
    sigprocmask(SIG_SETMASK, &a_mask, (sigset_t *)NULL);
  }

  return(rtn);
}

/******************************************************************************/
/*                            PUBLIC FUNCTIONS                                */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*- CfgDequeue ---------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
void *CfgDequeue(CfgQueue_t *queue, double timeout){
CfgRequest_t  *itemp;
register void *request;

  if(queue == (CfgQueue_t *)NULL){
    fprintf(stderr, __FUNCTION__": invalide pointer argument\n");
    return((void *)NULL);
  }

  if(CfgQueueSemWait(&queue->sync, timeout) == -1){
    if((errno != EINTR) && (errno != 0)){
      perror(__FUNCTION__": CfgQueueSemWait");
    }

    return((void *)NULL);
  }

  /*------------------------------------------------ Start of critical region */
  if(pthread_mutex_lock(&queue->lock) == -1){
    perror(__FUNCTION__": pthread_mutex_lock");
    return((void *)NULL);
  }

  if(queue->nbRequest > 0){
    itemp        = queue->head;
    queue->head = itemp->next;

    /*------------- If head is nil pointer, this was last request on the list */
    if(queue->head == (CfgRequest_t *)NULL){
      queue->tail = (CfgRequest_t *)NULL;
    }

    /*-------------------------------------------- Get user's request element */
    request = itemp->request;

    /*------------------------- Decrease the total number of pending requests */
    queue->nbRequest--;

    free(itemp);
  }
  else{
    request = (void *)NULL;
  }

  /*-------------------------------------------------- End of critical region */
  if(pthread_mutex_unlock(&queue->lock) == -1){
    perror(__FUNCTION__": pthread_mutex_unlock");
    return((void *)NULL);
  }

  return(request);
}

/*----------------------------------------------------------------------------*/
/*- CfgEnqueue ---------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgEnqueue(CfgQueue_t *queue, void *request){
CfgRequest_t *itemp;

  if(queue == (CfgQueue_t *)NULL){
    fprintf(stderr, __FUNCTION__": invalide pointer argument\n");
    return(CFG_FAIL);
  }

  if(queue->nbRequest >= queue->threshold){
    CfgQueuePrint("CfgQueue> Cannot put item @ %08x "
		  "into the queue %s [%d/%d]\n",
		  request, queue->name, queue->nbRequest, queue->threshold);
    return(CFG_FAIL);
  }

  itemp = (CfgRequest_t *)malloc(sizeof(CfgRequest_t));

  if(itemp == (CfgRequest_t *)NULL){
    perror(__FUNCTION__": malloc");
    return(CFG_FAIL);
  }

  /*-------------------------------- Put user's element into the request list */
  itemp->request = request;
  itemp->next    = (CfgRequest_t *)NULL;

  /*------------------------------------------------ Start of critical region */
  if(pthread_mutex_lock(&queue->lock) == -1){
    perror(__FUNCTION__": pthread_mutex_lock");
    return(CFG_FAIL);
  }

  /*------------------- Add new request to the end of the list, updating list */
  if(queue->nbRequest == 0){
    queue->head = itemp;
    queue->tail = itemp;
  }
  else{
    queue->tail->next = itemp;
    queue->tail       = itemp;
  }

  /*------------------------ Increase total number of pending requests by one */
  queue->nbRequest++;

  /*-------------------------------------------------- End of critical region */
  if(pthread_mutex_unlock(&queue->lock) == -1){
    perror(__FUNCTION__": pthread_mutex_lock");
    return(CFG_FAIL);
  }

  if(sem_post(&queue->sync) == -1){
    perror(__FUNCTION__": sem_post");
    return(CFG_FAIL);
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgQueueFlush ------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgQueueFlush(CfgQueue_t *queue, int nbRequest){
register void *request;
register int  i;

  if(nbRequest == 0){ return(CFG_OK); }

  for(i=0; i<nbRequest; i++){
    if((request = CfgDequeue(queue, 20e-3)) != (void *)NULL){ free(request); }
  }

  return(CFG_OK);
}

/*----------------------------------------------------------------------------*/
/*- CfgQueueFree -------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgQueue_t *CfgQueueFree(CfgQueue_t *queue){

  if(queue == (CfgQueue_t *)NULL){ return(queue); }

  CfgQueueFlush(queue, queue->nbRequest);

  sem_destroy(&queue->sync);
  
  pthread_mutex_destroy(&queue->lock);

  free(queue->name);
  free(queue);

  return((CfgQueue_t *)NULL);
}

/*----------------------------------------------------------------------------*/
/*- CfgQueueGetValue ---------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
int CfgQueueGetValue(CfgQueue_t *queue){
int sval;

  sem_getvalue(&queue->sync, &sval);

  return(sval);
}

/*----------------------------------------------------------------------------*/
/*- CfgQueueInstallPrinter ---------------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgQueuePrinter_t CfgQueueInstallPrinter(CfgQueuePrinter_t printer){
CfgQueuePrinter_t oprinter = CfgQueuePrinter;

  CfgQueuePrinter = printer;

  return(oprinter);
}

/*----------------------------------------------------------------------------*/
/*- CfgQueueNamedNew ---------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgQueue_t *CfgQueueNamedNew(int nelem, char *name){
CfgQueue_t *queue;
char buffer[16];


  if((queue = (CfgQueue_t *)malloc(sizeof(CfgQueue_t))) == (CfgQueue_t *)NULL){
    perror(__FUNCTION__": malloc(queue)");
    return((CfgQueue_t *)NULL);
  }

  memset((char *)queue, 0, sizeof(CfgQueue_t));

  queue->threshold = nelem;
 
  if(name == (char *)NULL){
    sprintf(buffer, "@ %p", queue);
    name = buffer;
  }

  if((queue->name = strdup(name)) == NULL){
    perror(__FUNCTION__": strdup(name)");
    return((CfgQueue_t *)NULL);
  }

  if(pthread_mutex_init(&queue->lock, pthread_mutexattr_default) == -1){
    perror(__FUNCTION__": pthread_mutex_init");
    free(queue);
    return((CfgQueue_t *)NULL);
  }

  if(sem_init(&queue->sync, 0, 0) == -1){
    perror(__FUNCTION__": sem_init");
    free(queue);
    return((CfgQueue_t *)NULL);
  }

  CfgQueuePrint("CfgQueue> Create Queue %s of %d element(s)\n",
                queue->name, queue->threshold);
  
  return(queue);
}

/*----------------------------------------------------------------------------*/
/*- CfgQueueNew --------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
CfgQueue_t *CfgQueueNew(int nelem){
  return(CfgQueueNamedNew(nelem, (char *)NULL));
}
