/* SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Cfg - A library for Virgo process implementation
 *
 * Copyright © 2001-2021 Laboratoire de physique des particules d'Annecy - CNRS
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *   Fatih Bellachia <fatih.bellachia@lapp.in2p3.fr>
 *   Laurent Fournier <laurent.fournier@lapp.in2p3.fr>
 *   Alain Masserot <alain.masserot@lapp.in2p3.fr>
 */

#include <sys/param.h> 
#include <errno.h> 
#include <math.h> 
#include <signal.h> 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <CmMessage.h>
/* #include <Db.h> */
#include <CfgVirgo.h>
#include <sys/resource.h>
#define __USE_XOPEN
#include <sys/stat.h>

#ifdef USE_MW
#include <memwatch.h>
#endif /* USE_MW */


/*--------------------------------------------------------- External variables */
extern char *CfgOutHostState[];
#define CFG_STTEND -1
#define CFG_DBCHAR '\''
#define CFG_DBMSG "Beware: String quoted with '"
extern CfgData_t *CfgData; 

/*----------------------------------------------- Internal function definition */
static CmMessageStatus CfgNSCleanupHandler (CmMessage message, char *sender, 
					    char *serverName );
static CmMessageStatus CfgCmDefault( CmMessage message, char *sender,
				     char *serverName);

static CmMessageStatus CfgCmFdGetConfig( CmMessage message, char *sender,
					 char *serverName);
static CmMessageStatus CfgCmKill( CmMessage message, char *sender,
				  char *serverName);
static CmMessageStatus CfgCmRFlag( CmMessage message, char *sender,
				   char *serverName);
static CmMessageStatus CfgCmSaveConfig( CmMessage message, char *sender,
					char *serverName );
static CmMessageStatus CfgCmUIRqActivation( CmMessage message,  char *sender,
					    char *serverName);
static CmMessageStatus CfgCmUIRqInfo( CmMessage message,  char *sender,
				      char *serverName);
static CmMessageStatus CfgCmUIRqInfoLog( CmMessage message,  char *sender,
					 char *serverName);
/* static int CfgBufToDb( char *config, char *confTag ); */
/* static char *CfgDbToBuf( char *confTag ); */

static int CfgDataMsgSendClAdd(  CfgData_t *cfgData, const CfgMsg_t *msg );
static int CfgDataMsgSendClClose( CfgData_t *cfgData );
static int CfgDataMsgSendClOpen( CfgData_t *cfgData );
static void CfgDataMsgRClose( CfgData_t * p);
static int CfgDataParseCmCleanupTmo( CfgData_t *p, CfgDataType_t *data );
static int CfgDataParseCmDomain( CfgData_t *p, CfgDataType_t *data );
static int CfgDataParseCmSocket( CfgData_t *p, CfgDataType_t *data );
static int CfgDataParseDebug( CfgData_t *p, CfgDataType_t *data );
static int CfgDataParseNoAction( CfgData_t *p, CfgDataType_t *data );
static int CfgDataParseNoDbSave( CfgData_t *p, CfgDataType_t *data );
static int CfgDataParseNoFileSave( CfgData_t *p, CfgDataType_t *data );
static int CfgDataParsePrio( CfgData_t *p, CfgDataType_t *data );
static int CfgDataParsePWD( CfgData_t *p, CfgDataType_t *data );
static int CfgDataParseRFlag( CfgData_t *p, CfgDataType_t *data );
static int CfgDataParseSchedAffinity( CfgData_t *p, CfgDataType_t *data );

static int CfgCmCleanup( char *cmName, double tmo );
static int CfgEnd( CfgData_t *p );
static int CfgIdleBasic( CfgData_t *p );
static int CfgIdleCm( CfgData_t *p );
static int CfgIdleParse( char *config );
static int CfgLogFPrinterv( char *format, va_list args );
static int CfgMsgPrinterv( char *format, va_list args );

/*---------------------------------------------------- CfgOutBuildMsg function */
static int CfgOutFdConfigData( CmMessage message, void *arg,
                               char *receiver, char *group );
static int CfgOutUIActivation( CmMessage message, void *arg,
                               char *receiver,  char *group );
static int CfgOutUIInfoAdd( CmMessage message, void *arg, char *receiver,
			    char *group );
static int CfgOutUIInfoAddLog( CmMessage message, void *arg, char *receiver,
			       char *group );
static int CfgOutUIInfoClose( CmMessage message, void *arg, char *receiver,
			      char *group );
static int CfgOutUIInfoOpen( CmMessage message, void *arg, char *receiver,
			     char *group );
static int CfgOutUIInfoOpenLog( CmMessage message, void *arg, char *receiver,
				char *group );

static int UI_flag = -1;  /*----------------------------- 0: UIInfo - 1: UILog */ 


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CmNSCleanupHandler */
/*-----------------------------------------------------------------------------*/
static CmMessageStatus CfgNSCleanupHandler (CmMessage message, char *sender, 
					    char *serverName ) {
  char* txt;

  txt = CmMessageGetText (message);
  CfgMsgPrintInfo("CfgNSCleanupHandler> %s", txt);
  return (CmMessageBreak);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgCmDefault */
/*-----------------------------------------------------------------------------*/
static CmMessageStatus CfgCmDefault( CmMessage message, char *sender,
				     char *serverName) {

  /*---------------------------------------------------------------------------*/
  CfgMsgAddInfo( "CfgCmDefault> cmType %s from %s", CmMessageGetType(message),
                 sender );
  /*---------------------------------------------------------------------------*/
  return(CmMessageOk);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgCmFdGetConfig */
/*-----------------------------------------------------------------------------*/
static CmMessageStatus CfgCmFdGetConfig( CmMessage message, char *sender,
                                  char *serverName) {
  int runN;

  if( !CfgDataGetConfig(CfgData) ) return(CmMessageOk);
  /*----------------------------------------------------------- Get run number */
  if( CmMessageGetItemType( message ) == CmMessageItemInt ) {
    runN = CmMessageGetInt( message );
  } else runN = -1;
  /*----------------------------- Add the sender to CFG_CM_FB_GET_CONFIG group */
  CfgOutGroupConsumerAdd( CFG_CM_FB_GET_CONFIG, sender, 1, 1, CfgOutDynamic,
                          NULL, NULL);
  /*------------------------------------------------------ FdGetConfig Message */
  CfgOutGroupPost( CFG_CM_FB_GET_CONFIG, CfgOutFdConfigData, &runN );
  /*---------------------------------------------------------------------------*/
  return(CmMessageBreak);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------------ CfgCmExit */
/*-----------------------------------------------------------------------------*/
CmMessageStatus CfgCmExit( CmMessage message, char *sender,
                           char *serverName) {

  /*-------------------------------------------------------- Put Debug message */
  CfgMsgAddInfo( "CfgCmExit> Receive from %s",  sender );
  /*-------------------------------------------------- Set CfgFinished to True */
/*   CfgReachState( SuServerEnd ); */
  CfgTerminate();
  /*---------------------------------------------------------------------------*/
  return(CmMessageBreak);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------------ CfgCmKill */
/*-----------------------------------------------------------------------------*/
CmMessageStatus CfgCmKill( CmMessage message, char *sender,
                           char *serverName) {
/*   CfgSetState( CFG_STTEND ); */
  /*-------------------------------------------------------- Put Debug message */
  CfgMsgAddInfo("CfgCmKill>Receive from %s\n",  sender );
  kill( 0, SIGINT );
  /*---------------------------------------------------------------------------*/
  return(CmMessageBreak);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgCmReStart */
/*-----------------------------------------------------------------------------*/
CmMessageStatus CfgCmReStart( CmMessage message, char *sender,
                              char *serverName) {
  int state;
  char *tag;

  /*-------------------------------------------------------- Put Debug message */
  CfgMsgAddFmt( CFG_INFO, CFGDBGCM, "Receive %s from %s\n",
                CFG_CM_RESTART, sender );
  /*--------------------------------------------- Update the configuration tag */
  if( CmMessageGetItemType(message) == CmMessageItemText ){
    tag = CmMessageGetText(message);
  } else tag = NULL;
  /*----------------------------------------------- ReStart up to Active state */
  CfgReachState(CfgServerEnd);
  /*-------------------------------------------------- End of CfgIdle function */
  CfgIdleParse( CfgConfigLoad(tag) );
  for( state = CfgServerIdle; state < CfgServerGolden; state++ ) {
     CfgReachState(state);
  }
  return(CmMessageBreak);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CfgCmRFlag */
/*-----------------------------------------------------------------------------*/
static CmMessageStatus CfgCmRFlag( CmMessage message, char *sender,
                            char *serverName) {
  int dbg, log, stdOut, cmTrace;
  double periodTime;
  char *tag;

  CfgMsgAddFmt( CFG_INFO, CFGDBGCM, "Receive %s from %s\n", CFG_CM_RFLAG,
                sender );
  dbg = log = stdOut = cmTrace = -1;
  periodTime = -1;
  while( CmMessageGetItemType(message) == CmMessageItemText ) {
    tag = CmMessageGetText(message);
    tag = CfgTolower(tag);
    /*------------------------------------ Treat the tag and the related value */
    if( !strcmp( tag, "debug" ) ) {       /*------------------------ debug Tag */
      if( CmMessageGetItemType(message) == CmMessageItemInt ) {
        dbg = CmMessageGetInt(message);
      } else dbg = 0;                       /*------- Set to the default value */
      continue;
    }
    if( !strcmp( tag, "logoff" ) ) {       /*--------------------- Logfile Off */
      log = 0; continue;
    }
    if( !strcmp( tag, "logon" ) ) {        /*---------------------- Logfile On */
      log = 1; continue;
    }
    if( !strcmp( tag, "stdouton" ) ) {     /*------------------- Stdout Tag On */
      stdOut = 1; continue;
    }
    if( !strcmp( tag, "stdoutoff" ) ) {    /*------------------ Stdout Tag Off */
      stdOut = 0; continue;
    }
    if( !strcmp( tag, "cmtraceon" ) ) {    /*---------- Cm trace on with level */
      cmTrace = CmMessageGetInt(message);
      CmMessageTraceOn( cmTrace ); continue;
    }
    if( !strcmp( tag, "cmtraceoff" ) ) {   /*-------------------- Cm trace off */
      cmTrace = 0; CmMessageTraceOff(); continue;
    }
    if( !strcmp( tag, "cmconnecton" ) ) {  /*------------- Cm connect debug On */
      CmConnectDebugOn();  continue;
    }
    if( !strcmp( tag, "cmconnectoff" ) ) { /*------------ Cm connect debug Off */
      CmConnectDebugOff();  continue;
    }
    if( !strcmp( tag, "periodtime" ) ) { /*------------------------ periodTime */
      if( CmMessageGetItemType(message) == CmMessageItemDouble ) {
        periodTime = CmMessageGetDouble(message);
      } else periodTime = CFG_PERIOD;      /*-------- Set to the default value */
    }
  }
  /*---------------------------------------------------------------------------*/
  CfgDataSetDebug( CfgData, dbg );
  CfgDataSetRFlag( CfgData, log, stdOut, cmTrace );
  CfgDataSetPeriodTime( CfgData, periodTime );
  /*---------------------------------------------------------------------------*/
  return(CmMessageOk);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgCmSaveConfig */
/*-----------------------------------------------------------------------------*/
static CmMessageStatus CfgCmSaveConfig( CmMessage message, char *sender,
					char *serverName ) {
  char *tag, *config;
  int noFileSave;  

  /*--------------------------------------------------------------------------*/
  if( CmMessageGetItemType(message) == CmMessageItemText ) {
    tag = CmMessageGetText(message);
  } else tag = CfgDataGetTag(CfgData);
  if( !(config = CfgDataGetConfig(CfgData)) ) {
    CfgMsgAddInfo( "CfgCmSaveConfig> Config unknown");
    return(CmMessageOk);
  }
  /*--------------------------------------------------- Save the configuration */
  CfgMsgAddInfo( "CfgCmSaveConfig> as %s - sender is %s", tag, sender );
  CfgDataGetCommitFlag( CfgData , &noFileSave, NULL );
  /*------------------- If noDbName and tag is a file: force to commit on file */
  if( !CfgDataGetSysName(CfgData) && CfgFromFile(tag) )
    CfgDataSetNoFileSave( CfgData, 0 );
  CfgConfigCommit( tag );
  /*------------------------------------------------- Restore the commit flags */
  CfgDataSetNoFileSave( CfgData, noFileSave );
  /*---------------------------------------------------------------------------*/
  return(CmMessageOk);
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CfgCmUIRqActivation */
/*-----------------------------------------------------------------------------*/
static CmMessageStatus CfgCmUIRqActivation( CmMessage message,  char *sender,
                                     char *serverName) {
  CmMessage reply = CmMessageNew();
  /*--------------------------------------------Build the UIActivation Message */
  CfgOutUIActivation( reply, NULL, sender, NULL );
  CmMessageSend( reply, sender );
  /*---------------------------------------------------------------------------*/
  CmMessageDelete( reply );
  return(CmMessageOk);
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgCmUIRqInfo */
/*-----------------------------------------------------------------------------*/
static CmMessageStatus CfgCmUIRqInfo( CmMessage message,  char *sender,
                               char *serverName) {

  if( UI_flag == 1 ) {
    CfgMsgAddInfo("CfgCmUIRqInfo> UIRqLog already received - sender \"%s\"",
		     (sender ? sender : "Unknown" ) );
    return( CmMessageOk );
  }
  UI_flag = 0;
  /*-------------------------------- Add the sender to CFG_CM_UI_RQ_INFO group */
  if( CfgOutGroupConsumerAdd( CFG_CM_UI_RQ_INFO, sender, 5, 1, CfgOutDynamic,
			      NULL, NULL) == CFG_FAIL  )
    return( CmMessageBreak );

  /*----------------------------------------- Ask to send UIActivation Message */
  CfgCmUIRqActivation( message, sender, serverName );
  /*----------------------------------------------- Only for the first request */
  if( !CbfCleanFind( "CfgDataMsgRClose", CfgData ) ) { 
    CbfCleanAddRoot( "CfgDataMsgRClose", (CbfCleanFnt_t)CfgDataMsgRClose,
		     CfgData );
    /*---------------------------------------------------------- Stdout report */
    CfgDataSetRFlag( CfgData, -1, CfgFalse, -1 );
  }
  /*---------------------------------------------------------------------------*/
  return(CmMessageBreak);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgCmUIRqInfoLog */
/*-----------------------------------------------------------------------------*/
static CmMessageStatus CfgCmUIRqInfoLog( CmMessage message,  char *sender,
					 char *serverName) {

  if( UI_flag == 0 ) {
    CfgMsgAddInfo("CfgCmUIRqInfoLog> UIRqInfo already received - sender \"%s\"",
		     (sender ? sender : "Unknown" ) );
    return( CmMessageOk );
  }
  UI_flag = 1;
  /*-------------------------------- Add the sender to CFG_CM_UI_RQ_INFO group */
  if( CfgOutGroupConsumerAdd( CFG_CM_UI_RQ_LOG, sender, 5, 1,
			      CfgOutDynamic,
			      NULL, NULL) == CFG_FAIL  )
    return( CmMessageBreak );
    
  /*----------------------------------------- Ask to send UIActivation Message */
  CfgCmUIRqActivation( message, sender, serverName );
  /*----------------------------------------------- Only for the first request */
  if( !CbfCleanFind( "CfgDataMsgRClose", CfgData ) ) { 
    CbfCleanAddRoot( "CfgDataMsgRClose", (CbfCleanFnt_t)CfgDataMsgRClose,
		     CfgData );
    /*---------------------------------------------------------- Stdout report */
    CfgDataSetRFlag( CfgData, -1, CfgFalse, -1 );
  }
  /*---------------------------------------------------------------------------*/
  return(CmMessageBreak);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CfgBufToDb */
/*-----------------------------------------------------------------------------*/
/* static int CfgBufToDb( char *config, char *confTag ) { */
/*   DbVector *dbHistory; */
/*   char *tmp, *str, *chr, *line; */
/*   int rtn = CFG_OK; */

/*   if( !confTag || !confTag[0] ) return(CFG_FAIL); */
/*   /\*--------------------------------------------- Replace all \" by CFG_DBCHAR *\/ */
/*   for( tmp = config; (chr = strchr(tmp, '\"')) != NULL; tmp = chr+1 ) { */
/*     chr[0] = CFG_DBCHAR; */
/*   } */
/*   /\*------------------------------------------------------ Create the DbObject *\/ */
/*   str = CfgStrFmtCat( NULL, CfgFalse, "string[] %s = {", confTag ); */
/*   str = CfgStrFmtCat( str, CfgFalse, "\"%s\"\n", CFG_DBMSG ); */
/*   /\*---------------------------------------- Translate line to DbString object *\/ */
/*   for( line = config; (chr = strchr(line, '\n')) != NULL; line = chr+1 ) { */
/*     chr[0] = '\0'; */
/*     str = CfgStrFmtCat( str, CfgFalse, "\"%s\"\n", line ); */
/*     chr[0] = '\n'; */
/*   } */
/*   /\*--------------------------------------------- Replace all CFG_DBCHAR by \" *\/ */
/*   for( tmp = config; (chr = strchr(tmp, CFG_DBCHAR)) != NULL; tmp = chr+1 ) { */
/*     chr[0] = '"'; */
/*   } */
/*   /\*------------------------------------------------------- Close the DbObject *\/ */
/*   str = CfgStrFmtCat( str, CfgFalse, NULL, "}" ); */
/*   DbCreateObject( str ); */
/*   if( (DbGetError() != DbErrorOk) || !CmConnectGetReference(DbServerName()) ) { */
/*     CfgMsgAddError("CfgBufToDb> error %s", DbGetErrorInfo()); */
/*     rtn = CFG_FAIL; */
/*   } else if( (dbHistory = DbGetVersionHistory( confTag )) ) { */
/*     CfgMsgAddInfo("CfgBufToDb> %s - %s", confTag, */
/*       DbStringGetContents(DbStringVectorGetElement( dbHistory, 0))); */
/*   } */
/*   free( str ); */
/*   /\*---------------------------------------------------------------------------*\/ */
/*   return(rtn); */
/* } */


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CfgDbToBuf */
/*-----------------------------------------------------------------------------*/
/* static char *CfgDbToBuf( char *confTag ) { */
/*   DbVector *dbVector; */
/*   DbString *dbStr; */
/*   DbVector *dbHistory; */
/*   char *str, *chr, *buf; */
/*   int i, nbEl; */

/*   if( !confTag || !confTag[0] ) return(NULL); */
/*   /\*---------------------------------------------------------------------------*\/ */
/*   dbVector = DbVectorObjectLoad( confTag ); */
/*   if( !dbVector ) { */
/*     CfgMsgAddError("CfgDbToBuf> %s not available", confTag ); */
/*     return(NULL); */
/*   } else if( (dbHistory = DbGetVersionHistory( confTag )) ) { */
/*     CfgMsgAddInfo("CfgDbToBuf> %s - %s", confTag, */
/*       DbStringGetContents(DbStringVectorGetElement( dbHistory, 0))); */
/*   } */
/*   /\*---------------------------------------------------------------------------*\/ */
/*   nbEl = DbVectorGetSize( dbVector ); */
/*   buf = NULL; */
/*   for( i = 0; i < nbEl; i++ ) { */
/*     dbStr = DbStringVectorGetElement( dbVector, i ); */
/*     if( strstr( DbStringGetContents(dbStr), CFG_DBMSG) ) continue; */
/*     buf = CfgStrFmtCat( buf, CfgFalse, "%s\n", DbStringGetContents(dbStr)); */
/*   } */
/*   /\*--------------------------------------------- Replace all CFG_DBCHAR by \" *\/ */
/*   for( str = buf;  (chr = strchr(str, CFG_DBCHAR)) != NULL; str = chr+1 ) { */
/*     chr[0] = '"'; */
/*   } */
/*   /\*---------------------------------------------------------------------------*\/ */
/*   return(buf); */
/* } */


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CfgDataMsgSendClAdd */
/*-----------------------------------------------------------------------------*/
static int CfgDataMsgSendClAdd( CfgData_t *cfgData, const CfgMsg_t *msg ) {
  
  /*---------------------------------------------------------------------------*/
  if( CfgMsgGetLevel(msg) == CFG_CLINFO ) return(CFG_OK);
  /*---------------------------------------------------- CfgOut default domain */
  CfgOutGroupPost( CFG_CM_UI_RQ_INFO, CfgOutUIInfoAdd, (void*)msg );
  CfgOutGroupPost( CFG_CM_UI_RQ_LOG, CfgOutUIInfoAddLog, (void*)msg );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------ CfgDataMsgSendClClose */
/*-----------------------------------------------------------------------------*/
static int CfgDataMsgSendClClose( CfgData_t *cfgData ) {
  
  /*---------------------------------------------------------------------------*/
  CfgOutGroupPost( CFG_CM_UI_RQ_INFO, CfgOutUIInfoClose, NULL );
  CfgOutGroupPost( CFG_CM_UI_RQ_LOG, CfgOutUIInfoClose, NULL );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataMsgSendClOpen */
/*-----------------------------------------------------------------------------*/
static int CfgDataMsgSendClOpen( CfgData_t *cfgData ) {
  
  /*---------------------------------------------------------------------------*/
  CfgOutGroupPost( CFG_CM_UI_RQ_INFO, CfgOutUIInfoOpen, NULL );
  CfgOutGroupPost( CFG_CM_UI_RQ_LOG, CfgOutUIInfoOpenLog, NULL );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgDataMsgRClose */
/*-----------------------------------------------------------------------------*/
static void CfgDataMsgRClose( CfgData_t * p) {

  if( !CbfCleanFind("CfgDataMsgRClose", p) ) return;
  /*---------------------------------------------------------------------------*/
  CfgMsgAddInfo("CfgDataMsgRClose> started");
  CfgMsgSendWithTimeout( -2 );
  CbfCleanDone("CfgDataMsgRClose", p);
  CfgMsgAddInfo("CfgDataMsgRClose> done");
  /*---------------------------------------------------------------------------*/
  return;
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------- CfgDataParseCmCleanupTmo */
/*-----------------------------------------------------------------------------*/
static int CfgDataParseCmCleanupTmo( CfgData_t *p, CfgDataType_t *data ) {
  double tmo;

  tmo = CfgParseGetNextReal( data );
  /*---------------------------------------------------------------------------*/
  if( CfgParseGetResult( data ) != CFG_OK ) {
    CfgMsgAddError( "CfgDataParseCmCleanupTmo> troubles extracting the parameters\n"
		    "\tsyntaxe: CFG_CMCLEANUP_TMO <tmo>" );
    return(CFG_FAIL);
  }
  /*---------------------------------------------------------------------------*/
  CfgDataSetCmCleanupTmo( p, tmo );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataParseCmDomain */
/*-----------------------------------------------------------------------------*/
static int CfgDataParseCmDomain( CfgData_t *p, CfgDataType_t *data ) {
  char *cmDomain;

  if( !(cmDomain = CfgParseGetNextString( data )) || !cmDomain[0] )
    return(CFG_OK);
  /*---------------------------------------------------------------------------*/
  if( CfgParseGetResult( data ) != CFG_OK ) {
    CfgMsgAddError( "CfgDataParseCmDomain> troubles extracting the parameters\n"
		    "\tsyntaxe: CFG_CMDOMAIN string:<CmDomainName|None|NONE>" );
    return(CFG_FAIL);
  }
  /*---------------------------------------------------------------------------*/
  CfgDataSetCmDomainName( p, cmDomain );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataParseCmSocket */
/*-----------------------------------------------------------------------------*/
static int CfgDataParseCmSocket( CfgData_t *p, CfgDataType_t *data ) {
  char *tag;
  int snd, rcv;

  snd = CfgParseGetNextDec( data );
  rcv = CfgParseGetNextDec( data );
  tag = CfgParseGetNextString( data );
  if( tag && tag[0] == '\0' ) tag = NULL;
  /*---------------------------------------------------------------------------*/
  if( CfgParseGetResult( data ) != CFG_OK ) {
    CfgMsgAddError("CfgDataParseCmSocket> troubles extracting the parameters\n"
		   "\tsyntaxe: CFG_CMSOCKET int:<transmitter buffer size> "
		   "int:<receiver buffer size> char:<tag>");
    return(CFG_FAIL);
  }
  /*---------------------------------------------------------------------------*/
  CmConnectSetSocketSize( snd, rcv, tag );
  CfgMsgAddInfo("CfgDataParseCmSocket> snd %d, rcv %d, tag %s", snd, rcv, 
		(tag ? tag : "*"));
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgDataParseDebug */
/*-----------------------------------------------------------------------------*/
static int CfgDataParseDebug( CfgData_t *p, CfgDataType_t *data ) {
  int dbg;

  dbg = CfgParseGetNextDec( data );
  /*---------------------------------------------------------------------------*/
  return( CfgDataSetDebug( p, dbg ));
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataParseNoAction */
/*-----------------------------------------------------------------------------*/
static int CfgDataParseNoAction( CfgData_t *p, CfgDataType_t *data ) {
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgDataParseNoDbSave */
/*-----------------------------------------------------------------------------*/
static int CfgDataParseNoDbSave( CfgData_t *p, CfgDataType_t *data ) {

  CfgParseGetNextDec( data );
  /*---------------------------------------------------------------------------*/
  return( CfgDataSetNoDbSave( p, 1) );
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------- CfgDataParseNoFileSave */
/*-----------------------------------------------------------------------------*/
static int CfgDataParseNoFileSave( CfgData_t *p, CfgDataType_t *data ) {

  CfgParseGetNextDec( data );
  /*---------------------------------------------------------------------------*/
  return( CfgDataSetNoFileSave( p, 1) );
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgDataParsePrio */
/*-----------------------------------------------------------------------------*/
static int CfgDataParsePrio( CfgData_t *p, CfgDataType_t *data ) {
  int prio;

  prio = CfgParseGetNextDec( data );
  /*---------------------------------------------------------------------------*/
  return( CfgDataSetPrio( p, prio ));
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgDataParsePWD */
/*-----------------------------------------------------------------------------*/
static int CfgDataParsePWD( CfgData_t *cfgData, CfgDataType_t *data ) {
  char fName[MAXPATHLEN], *tmp;
  struct stat fStat;
  char *wd;
  mode_t mode;

  if( !(wd = CfgParseGetNextString( data )) || !wd[0] ) return(CFG_OK);
  /*---------------------------------------------------------------------------*/
  CfgFindPath( fName, sizeof fName, wd, F_OK );   
  mode = (S_IRWXU | S_IRWXG | S_IROTH | S_IWOTH);
  /*--------------------------------- Build/Check the parent working directory */
  if( access(wd, F_OK) == 0 ) { 
    if( !stat( fName, &fStat ) && (fStat.st_mode & S_IFMT) != S_IFDIR ) {
      CfgMsgAddError("CfgDataParseWD> %s exist but not as directory", fName);
      return(CFG_FAIL);
    } 
  } else if( mkdir( wd, mode ) == -1 ) {
    CfgMsgAddError("CfgIDataParseWD> mkdir %s failed -  %s", wd,
		   strerror(errno) );
     return(CFG_FAIL);
  }
  /*----------------------------------------- Build/Check the working directory */
  tmp = CfgStrFmtCat( NULL,  CfgFalse, "%s/%s", wd, CfgDataGetCmName(cfgData));
  /*----------------------------------------------------------------------------*/
  if( access(tmp, F_OK) == 0 ) {
    if( !stat( tmp, &fStat ) && (fStat.st_mode & S_IFMT) != S_IFDIR ) {
      CfgMsgAddError("CfgIDataParseWD> %s exist but not as directory",tmp );
      return(CFG_FAIL);
    } 
  } else if( mkdir( tmp, mode) == -1 ) {
    CfgMsgAddError("CfgDataParseWD> mkdir %s failed - %s", tmp,
		   strerror(errno));
    return(CFG_FAIL);
  }
  /*---------------------------------------------------------------------------*/
  chdir( tmp );
  CfgMsgAddInfo("CfgDataParseWD> log path %s", tmp );
  free(tmp );
  /*---------------------------------------------------------------------------*/
  return(CfgDataLogFilenameInit(cfgData));
}




/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgDataParseRFlag */
/*-----------------------------------------------------------------------------*/
static int CfgDataParseRFlag( CfgData_t *cfgData, CfgDataType_t *data ) {
  int log, stdOut, cmTrace, i, nb, err;
  double periodTime;
  char *tag;

  /*---------------------------------------------------------------------------*/
  periodTime = CfgParseGetNextReal(data);
  /*---------------------------------------------------------------------------*/
  nb = CfgParseGetNextDec(data);
  if( nb ) err = 0; else err = 1;
  log = stdOut = cmTrace = -1;
  /*---------------------------------------------------------------------------*/
  for( i = 0; i < nb; i++ ) {
    tag = CfgParseGetNextString( data );
    tag = CfgTolower(tag);
    if( !strcmp( tag, "logoff" ) ) {       /*--------------------- Logfile Off */
      log = 0; continue;
    }
    if( !strcmp( tag, "logon" ) ) {        /*---------------------- Logfile On */
      log = 1; continue;
    }
    if( !strcmp( tag, "stdouton" ) ) {     /*------------------- Stdout Tag On */
      stdOut = 1; continue;
    }
    if( !strcmp( tag, "stdoutoff" ) ) {    /*------------------ Stdout Tag Off */
      stdOut = 0; continue;
    }
    if( !strcmp( tag, "cmtraceon" ) ) {    /*---------- Cm trace on with level */
      cmTrace = CfgParseGetNextDec(data); i++;
      CmMessageTraceOn( cmTrace ); continue;
    }
    if( !strcmp( tag, "cmtraceoff" ) ) {   /*-------------------- Cm trace off */
      cmTrace = 0; CmMessageTraceOff(); continue;
    }
    err++;
  }
  /*---------------------------------------------------------------------------*/  
  if( CfgParseGetResult( data ) != CFG_OK  || err ) {
    CfgMsgAddError("CfgDataParseRFlag> troubles extracting the parameters\n"
		   "\tsyntaxe: CFG_RFLAG double:<message output period> "
		   "text:<[stdout,log,cmtrace][on,off]>");
    return(CFG_FAIL);
  }
  CfgDataSetPeriodTime( CfgData, periodTime );
  /*---------------------------------------------------------------------------*/
  return(CfgDataSetRFlag( cfgData, log, stdOut, cmTrace ));
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------- CfgDataParseAchedAffinity */
/*-----------------------------------------------------------------------------*/
static int CfgDataParseSchedAffinity( CfgData_t *p, CfgDataType_t *data ) {
  char *schedAffinity;

  schedAffinity = CfgParseGetNextString( data );
  /*---------------------------------------------------------------------------*/
  if( CfgParseGetResult( data ) != CFG_OK  || !schedAffinity ) {
    CfgMsgAddError( "CfgDataParseSchedAffinity> troubles extracting"
		    " the parameters\n"
		    "\tsyntaxe: CFG_SCHEDAFFINITY string:<0,5,7,9-11>" );
    return(CFG_FAIL);
  }
  /*---------------------------------------------------------------------------*/
  return( CfgDataSetSchedAffinity( p, schedAffinity ) );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgCmCleaup */
/*-----------------------------------------------------------------------------*/
static int CfgCmCleanup( char *cmName, double tmo ) {
  CmMessage cleanup;
  char *NS_name;
  char *cmName_cleanup;
  int rtn;

  if( !cmName || !cmName[0] ) return(CFG_FAIL);
  /*----------------------------------------------------------------------- Cm */
  CmMessageInstallPrinter((CmConnectPrinter)CfgMsgPrinterv);
  /*--------------------------------------- Install Cm Message Default Handler */
  CmMessageInstallDefaultHandler(CfgCmDefault);
  /*---------------------------------------------------------------------------*/
  cmName_cleanup = CfgStrFmtCat( NULL, CfgFalse, "%s_cleanup", cmName );
  /*---------------------------------------------------------- Cm Name cleanup */
  if(!CmMessageOpenServer(cmName_cleanup)){
    CfgMsgAddFatal( "CfgCmCleanup> CmMessageOpenServer failed for \"%s\")",
		    cmName_cleanup );
    return(CFG_FAIL);
  }
  NS_name = CmConnectGetName( CmConnectGetNameServer() );
  CmMessageInstallHandler( (CmMessageHandler)CfgNSCleanupHandler, "NSCleanup");
  rtn = CFG_OK;
  if( (cleanup = CmMessageNew()) ) {
    CmMessageSetType( cleanup, "NSDoCleanup" );
    CmMessagePutText( cleanup, cmName );
    if( CmMessageSend( cleanup, NS_name ) ) {
      if( CmMessageWaitWithTimeout( tmo ) == CmConnectTimeoutDetection ) {
	CfgMsgAddError("CfgCmCleanup> NSCleanup not received from %s",
		       NS_name );
	rtn = CFG_FAIL;
      }
    } else rtn = CFG_FAIL;
    CmMessageDelete( cleanup );
  } else rtn = CFG_FAIL;
  CmMessageUninstallHandler("NSCleanup");
  CmMessageCloseServer(cmName_cleanup);
  if( cmName_cleanup ) free(cmName_cleanup);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------------- CfgEnd */
/*-----------------------------------------------------------------------------*/
static int CfgEnd( CfgData_t *p ) {
  
  /*---------------------------------------------------------------------------*/
  CfgDataSetConfigFnt( p, (CfgConfigFnt_t)NULL);
  CfgDataSetConfig( p, NULL );
  CbfCleanFreeTag("CfgClose", p );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgIdleBasic */
/*-----------------------------------------------------------------------------*/
static int CfgIdleBasic( CfgData_t *p ) {
  CfgMsgR_t *msgR;

  if( !p ) return(CFG_FAIL);
  /*---------------------------------------------------------- Install Printer */
  CbfHInstallPrinter((CbfHPrinter_t)CfgMsgPrinterv);
  CbfCleanInstallPrinter((CbfCleanPrinter_t)CfgMsgPrinterv);
  CfgMsgInstallPrinter((CfgMsgPrinter_t)CfgMsgPrinterv);
  CfgOutInstallPrinter((CfgOutPrinter_t)CfgMsgPrinterv);
  CfgParseInstallPrinter((CfgParsePrinter_t)CfgMsgPrinterv);
  /*-------------------------------------------- Message report initialization */
  msgR = CfgDataGetMsgR( p ); 
  CfgMsgRAddFnt( msgR, (CfgMsgSendFnt_t*)CfgDataMsgSendInit, p,
		 (CfgMsgFnt_t*)NULL, (CfgMsgFnt_t*)NULL );
  CfgMsgRAddFnt( msgR, (CfgMsgSendFnt_t*)CfgDataMsgSendStdout, p,
		 (CfgMsgFnt_t*)NULL, (CfgMsgFnt_t*)NULL );
  CfgMsgRAddFnt( msgR, (CfgMsgSendFnt_t*)CfgDataMsgSendLog, p,
		 (CfgMsgFnt_t*)CfgDataMsgLogReportOpen,
		 (CfgMsgFnt_t*)CfgDataMsgLogReportClose );
  CfgDataSetDebug( p, 0 );
  CfgDataSetRFlag( p, CfgTrue, CfgTrue, -1); /*------------ LogFile and Stdout */
  /*-------------------------------------- Cleanup (Should be after CfgOutNew) */
  CbfCleanAddRoot( "CfgClose", (CbfCleanFnt_t)CfgClose, p );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);  
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------------ CfgIdleCm */
/*-----------------------------------------------------------------------------*/
static int CfgIdleCm( CfgData_t *p ) {
  CfgMsgR_t *msgR;
  char *cmName, *cmDomain, *mCmDomain;
  double cmCleanTmo;

  if( !p ) return(CFG_FAIL);
  /*------------------------------- If cmDomain is unknown or cmDomain != NONE */
  cmName     = CfgDataGetCmName(p); 
  cmCleanTmo = CfgDataGetCmCleanupTmo(p);
  cmDomain   = CmConnectGetDomain();
  mCmDomain  = NULL;
  /*---------------------------------------------------------------------------*/
  if( cmDomain && !(mCmDomain = strdup(cmDomain)) ) {
      CfgMsgAddFatal( "CfgIdleCm> strdup fails" );
      return(CFG_FAIL);
  }
  /*---------------------------------------------------------------------------*/
  if( !mCmDomain || strcmp(CfgTolower(mCmDomain), CFG_NOCM ) ) {
    /*------------------------------------------------------------- Cm cleanup */
    if( CfgCmCleanup( cmName, cmCleanTmo ) != CFG_OK ) {
      CfgMsgAddFatal( "CfgIdleCm> CfgCmCleanup fails" );
      return(CFG_FAIL);
    }

    /*--------------------------------------------------------------------- Cm */
    CmMessageInstallPrinter((CmConnectPrinter)CfgMsgPrinterv);
    /*------------------------------------- Install Cm Message Default Handler */
    CmMessageInstallDefaultHandler(CfgCmDefault);
    /*---------------------------------------------------------------- Cm Name */
    if(!CmMessageOpenServer(cmName)){
      CfgMsgAddFatal( "CfgIdlem> CmMessageOpenServer failed (%s)", cmName);
      return(CFG_FAIL);
    }
/*     if( CfgDataGetSysName( p ) ) DbInstallPrinter((DbPrinter)CfgMsgPrinterv); */

    CfgMsgAddInfo( "CfgIdleCm> Cm Server %s launched  - CmDomain %s - CmPort %d",
		   cmName, CmConnectGetDomain(),
		   CmConnectGetPort(CmConnectWhoAmI()) );
    /*--------------------------------------------- Install Cm Message Handler */
    CmMessageInstallHandler(CfgCmExit, CFG_CM_EXIT);
    CmMessageInstallHandler(CfgCmKill, CFG_CM_KILL);
    CmMessageInstallHandler(CfgCmReStart, CFG_CM_RESTART);
    CmMessageInstallHandler(CfgCmRFlag, CFG_CM_RFLAG);
    CmMessageInstallHandler(CfgCmSaveConfig, CFG_CM_SVCFG);
    /*------------------- CFG_CM_FB_GET_CONFIG Cm Hanlder - Slow Frame Builder */
    CmMessageInstallHandler(CfgCmFdGetConfig, CFG_CM_FB_GET_CONFIG );
    /*-------------------------- CFG_CM_UI_RG_ACTIVATION Cm Hanlder - UIClient */
    CmMessageInstallHandler(CfgCmUIRqActivation, CFG_CM_UI_RQ_ACTIVATION );
    CmMessageInstallHandler(CfgCmUIRqInfo, CFG_CM_UI_RQ_INFO );
    CmMessageInstallHandler(CfgCmUIRqInfoLog, CFG_CM_UI_RQ_LOG );
    msgR = CfgDataGetMsgR( p ); 
    CfgMsgRAddFnt( msgR, (CfgMsgSendFnt_t*)CfgDataMsgSendClAdd, CfgData,
		   (CfgMsgFnt_t*)CfgDataMsgSendClOpen,
		   (CfgMsgFnt_t*)CfgDataMsgSendClClose );
    /*-------------------------------------------------------------------------*/
    if( mCmDomain ) free(mCmDomain);
  } else {
    CmConnectSetDomain( NULL );
    CfgMsgAddInfo("CfgIdleCm> Server %s launched - No CM", cmName );
  }
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);  
}



/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgIdleParse */
/*-----------------------------------------------------------------------------*/
static int CfgIdleParse( char *config ) {
  int rtn = CFG_OK;
  CfgParse_t *cfgParse;

  if( !config || !config[0] ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  if( !(cfgParse = CfgParseNew()) ) {
    CfgMsgAddError( "CfgIdleParse> CfgParseNew failed");
    return(CFG_FAIL);
  }
  /*-------------------------------------------------------- Initialize Parser */
  rtn |= CfgParseIdleAdd( cfgParse );
  /*---------------------------------------------------------------------------*/
  rtn |= CfgParseBuffer( cfgParse, config );
  cfgParse = CfgParseFree( cfgParse );
  if( rtn != CFG_OK ) {
    CfgMsgAddFatal( "CfgIdleParse> Error while parsing: %s", CfgPrintError() );
  }
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgLogFPrinterv */
/*-----------------------------------------------------------------------------*/
static int CfgLogFPrinterv( char *format, va_list args ) {
  
  /*---------------------------------------------------------------------------*/
  return( CfgDataLogFPrintv( CfgData, format, args ) );
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgMsgPrintv */
/*-----------------------------------------------------------------------------*/
static int CfgMsgPrinterv( char *format, va_list args ) {
  
  /*---------------------------------------------------------------------------*/
  return( CfgDataMsgPrinterv( CfgData, format, args ) );
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CfgOutFdConfigData */
/*-----------------------------------------------------------------------------*/
static int CfgOutFdConfigData( CmMessage message, void *arg,
                               char *receiver, char *group ) {
  int *runN = (int*)arg;
  char *config, strTime[50], *s, *c;
  time_t startTime; 

  if( strcmp( group, CFG_CM_FB_GET_CONFIG ) ) return(CFG_FAIL);
  /*-------------------------------- Performs a strcat with the current config */
  startTime = CfgDataGetStartTime(CfgData);
  strftime( strTime, sizeof strTime, CFG_TIME_FMT, localtime(&startTime) );
  config = CfgStrFmtCat( NULL, CfgFalse, "# Server %s\n"
			 "# configTag %s - sysName %s\n"
			 "# cmName %s - startTime %s\n",
			 CfgDataGetActivation(CfgData), CfgDataGetTag(CfgData),
			 (s = CfgDataGetSysName(CfgData)) ? s : "Undefined",
			 (c = CfgDataGetCmName(CfgData)) ? c : "Undefined",
			 strTime );
  config = CfgStrFmtCat( config, CfgFalse, NULL, CfgGetConfig() );
  /*---------------------------------------------------------------------------*/
  CmMessageSetType( message, CFG_CM_FB_CONFIG_DATA );
  if( runN && runN[0] != -1 ) CmMessagePutInt( message, runN[0] );
  CmMessagePutText( message, config );
  /*--------------------------------------------------- Free the config string */
  free( config );
  /*--------------------- Ask to remove this consumer when the message is sent */
  CfgOutGroupConsumerRemove( CFG_CM_FB_GET_CONFIG, receiver );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CfgOutUIActivation */
/*-----------------------------------------------------------------------------*/
static int CfgOutUIActivation( CmMessage message, void *arg,
                               char *receiver,  char *group ) {
  char *tmp;

  if( CmMessageGetState( message ) != CmMessageIsNew ) return(CFG_OK);
  /*---------------------------------------------------------------------------*/
  CmMessageSetType( message, CFG_CM_UI_ACTIVATION );
  CmMessagePutText( message, CfgDataGetHostname(CfgData) );
  CmMessagePutText( message, CfgDataGetActivation(CfgData) );
  CmMessagePutInt ( message, CfgDataGetPid(CfgData) );
  CmMessagePutInt ( message, CfgDataGetStartTime(CfgData) );
  CmMessagePutText( message, CfgDataGetCmName(CfgData) );
  tmp = CfgDataGetSysName(CfgData);
  CmMessagePutText( message, (tmp ? tmp : "No subSysName"));
  tmp = CfgDataGetLogFilename(CfgData);
  CmMessagePutText( message, (tmp ? tmp : "No logfile"));
  tmp = CfgDataGetTag(CfgData);
  CmMessagePutText( message, (tmp ? tmp : "No Config Tag"));
  tmp = CfgGetConfig();
  CmMessagePutText( message, ( tmp ? tmp : "No config"));
  tmp = CfgDataGetUserActivation(CfgData);
  CmMessagePutText( message, ( tmp ? tmp : ""));
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgOutUIInfoAdd */
/*-----------------------------------------------------------------------------*/
static int CfgOutUIInfoAdd( CmMessage message, void *arg, char *receiver,
                         char *group ) {
  CfgMsg_t *msg = (CfgMsg_t *)arg;
  
  if( !msg ) return(CFG_OK);
  /*---------------------------------------------------- Add to the Cm Message */
  CmMessagePutInt ( message, CfgMsgGetLevel(msg) );
  CmMessagePutText( message, CfgMsgGetTxt(msg) );
  CmMessagePutInt ( message, CfgMsgGetCount(msg) );
  /*---------------------------------------------------------------------------*/
  return(CFG_CONTINUE);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CfgOutUIInfoAddLog */
/*-----------------------------------------------------------------------------*/
static int CfgOutUIInfoAddLog( CmMessage message, void *arg, char *receiver,
			       char *group ) {
  CfgMsg_t *msg = (CfgMsg_t *)arg;
  
  if( !msg ) return(CFG_OK);
  /*---------------------------------------------------- Add to the Cm Message */
  CmMessagePutInt ( message, CfgMsgGetLevel(msg) );
  CmMessagePutText( message, CfgMsgGetTxt(msg) );
  CmMessagePutInt ( message, CfgMsgGetCount(msg) );
  CmMessagePutInt ( message, CfgMsgGetTimeStamp(msg) );
  /*--------------------------------------------------------------------------*/
  return(CFG_CONTINUE);
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgOutUIInfoClose */
/*-----------------------------------------------------------------------------*/
static int CfgOutUIInfoClose( CmMessage message, void *arg, char *receiver,
			      char *group ) {

  /*--------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgOutUIInfoOpen */
/*-----------------------------------------------------------------------------*/
static int CfgOutUIInfoOpen( CmMessage message, void *arg, char *receiver,
			     char *group ) {
  CfgServerState_t state;

  /*--------------------------------------------------------------- Put Header */
  if( CmMessageGetState( message ) == CmMessageIsNew ) {
    state = CfgDataGetState(CfgData);
    CmMessageSetType( message, CFG_CM_UI_INFO );
    CmMessagePutInt ( message, CfgDataGetCurrentTime(CfgData) );
    CmMessagePutInt ( message, state );
    CmMessagePutText( message, CfgDataGetStateStr(CfgData, state) );
    CmMessagePutInt ( message, CfgDataGetMemSize(CfgData));
 }
  /*--------------------------------------------------------------------------*/
  return(CFG_CONTINUE);
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------- CfgOutUIInfoOpenLog */
/*-----------------------------------------------------------------------------*/
static int CfgOutUIInfoOpenLog( CmMessage message, void *arg, char *receiver,
				char *group ) {
  CfgServerState_t state;

  /*--------------------------------------------------------------- Put Header */
  if( CmMessageGetState( message ) == CmMessageIsNew ) {
    state = CfgDataGetState(CfgData);
    CmMessageSetType( message, CFG_CM_UI_LOG );
    CmMessagePutInt ( message, CfgDataGetCurrentTime(CfgData) );
    CmMessagePutInt ( message, state );
    CmMessagePutText( message, CfgDataGetStateStr(CfgData, state) );
    CmMessagePutInt ( message, CfgDataGetMemSize(CfgData));
    CmMessagePutInt ( message, CfgDataGetConfigTime(CfgData) );
  }
  /*--------------------------------------------------------------------------*/
  return(CFG_CONTINUE);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------------- CfgConfAdd */
/*-----------------------------------------------------------------------------*/
int CfgConfAdd( char *fmt, ... ) {
  va_list args;
  int length;

  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  length = CfgDataConfAddv( CfgData, fmt, args );
  va_end( args );
  /*---------------------------------------------------------------------------*/
  return(length);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgConfigCommit */
/*-----------------------------------------------------------------------------*/
int CfgConfigCommit( char *confTag ) {
  char *config,  *file/* , *dbName */;
  int rtn = CFG_OK, noFileSave, noDbSave;

  if( !confTag ) confTag = CfgDataGetTag( CfgData );
  if( !confTag || !(config = CfgGetConfig()) ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
/*   dbName = CfgDataGetSysName( CfgData ); */
  CfgDataGetCommitFlag( CfgData, &noFileSave, &noDbSave );
  /*---------------------------------------------------------------------------*/
  if( CfgFromFile( confTag ) == CfgTrue )  { /*--------------- Load from file */
    /*----------------------------------------------------------- Save to File */
    if( !noFileSave )  { 
      /*--------------------------------- Save the previous configuration file */
      CfgMsgAddInfo("CfgConfigCommit> confTag %s", confTag);
      if( !access(confTag, F_OK) ) { /*---------------------- Rename the file  */
        file = CfgStrFmtCat( NULL, CfgFalse, "%s_%s", confTag,
                             CfgDataGetStrTime( CfgData ) );
        rename( confTag, file );
        CfgMsgAddInfo( "CfgConfigCommit> To file %s - prev %s", confTag, file );
        free( file );
      } else CfgMsgAddInfo( "CfgConfigCommit> To file %s", confTag );
      /*------------------------------------------- Save in the standard file  */
      rtn |= CfgBufToFile( config, confTag);
    }
    /*------------------------------------------------------------- Save on Db */
/*     if( !noDbSave  && dbName ) {  */
/*       CfgMsgAddInfo( "CfgConfigCommit> To Db as %s", dbName ); */
/*       rtn |= CfgBufToDb( config, dbName ); */
/*     } */
  } else { /*------------------------------------------------------ Save to Db */
    CfgMsgAddError( "CfgConfigCommit> To Db as %s - not supported", confTag );
    rtn = CFG_FAIL;
/*     rtn |= CfgBufToDb( config, confTag ); */
/*     /\*-------------------- Save in the CmName DbObject if confTag is different *\/ */
/*     if( dbName && strcmp( confTag, dbName) ) { */
/*       CfgMsgAddInfo( "CfgConfigCommit> To Db as %s", dbName ); */
/*       rtn |= CfgBufToDb( config, dbName ); */
/*     } */
  }
  /*---------------------------------------------------------------------------*/
  CfgMsgAddInfo( "CfgConfigCommit> %s", (rtn==CFG_OK?"Ok":"Fail") );
  return( rtn );
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgConfigLoad */
/*-----------------------------------------------------------------------------*/
char *CfgConfigLoad( char *confTag ) {
  char *buffer;
  char tag[MAXPATHLEN];

  if( !confTag ) confTag = CfgDataGetTag( CfgData ) ;
  if( !confTag ) return(NULL);
  /*------------------------ Search for .cfg to manage configuration from file */
  tag[0] = '\0';
  if( CfgFromFile( confTag ) == CfgTrue )  {
    CfgFindPath(tag, MAXPATHLEN-1, confTag, F_OK );
    CfgMsgAddInfo( "CfgConfigLoad> %s loaded from file \'%s\'", confTag, tag );
    buffer = CfgFileToBuf(tag);
  } else {
    CfgMsgAddError( "CfgConfigLoad> %s loaded from Db not supported", confTag );
    buffer = NULL;
/*     buffer = CfgDbToBuf(confTag); */
  }
  if( buffer && (buffer[0] == '\0')  ) { free(buffer); buffer = NULL; }
  if( !buffer )  {
    CfgMsgAddError("CfgConfigLoad> %s fails - tag '%s'", confTag, tag );
    return( buffer );
  }
  CfgMsgAddInfo("CfgConfigLoad> %s Ok - size %d bytes", confTag,
		(int)strlen(buffer));
  /*---------------------------------------------------------------------------*/
  CfgDataSetConfig( CfgData, buffer );
  CfgLogFPrint( "Load current config :\n%s\n", buffer );
  /*---------------------------------------------------------------------------*/
  return( buffer );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------------- CfgClose */
/*-----------------------------------------------------------------------------*/
void CfgClose(void) {

  if( !CbfCleanFind("CfgClose", CfgData) ) return;
  /*---------------------------------------------------------------------------*/
  if( CfgDataGetCmDomainName(CfgData) ) {
    /*----------------------------------------------- Uninstall Cfg Cm Handler */
    CmMessageUninstallHandler(CFG_CM_EXIT);
    CmMessageUninstallHandler(CFG_CM_KILL);
    CmMessageUninstallHandler(CFG_CM_RESTART);
    CmMessageUninstallHandler(CFG_CM_RFLAG);
    CmMessageUninstallHandler(CFG_CM_SVCFG);
    /*------------------- CFG_CM_FB_GET_CONFIG Cm Hanlder - Slow Frame Builder */
    CmMessageUninstallHandler(CFG_CM_FB_GET_CONFIG );
    /*-------------------------- CFG_CM_UI_RG_ACTIVATION Cm Hanlder - UIClient */
    CmMessageUninstallHandler(CFG_CM_UI_RQ_ACTIVATION );
    CmMessageUninstallHandler(CFG_CM_UI_RQ_INFO );
    CmMessageUninstallHandler(CFG_CM_UI_RQ_LOG );
    /*---------------------------------------------------------------- Cleanup */
/*     if( CfgDataGetSysName( CfgData ) ) { */
/*       DbInstallPrinter((DbPrinter)NULL); */
/*       DbCleanup();        /\*--------------------------------------- Db cleanup *\/ */
/*     } */
    /*-------------------------------------------------------- CfgData Cleanup */
    CmMessageInstallPrinter((CmConnectPrinter)NULL);
    CmMessageCloseServer(CfgDataGetCmName(CfgData));
    CfgDataSetCmDomainName( CfgData, NULL );
  }
  
  CfgMsgPrintInfo( "CfgClose> Done" );
  /*---------------------------------------------------------- Install Printer */
  CfgOutInstallPrinter((CfgOutPrinter_t)CfgLogFPrinterv);
  CfgMsgInstallPrinter((CfgMsgPrinter_t)CfgLogFPrinterv);
  CbfCleanInstallPrinter((CbfCleanPrinter_t)CfgLogFPrinterv);
  CbfHInstallPrinter((CbfHPrinter_t)CfgLogFPrinterv);
  CbfCleanDone("CfgClose", CfgData);
  /*---------------------------------------------------------------------------*/
  return;
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgFinished */
/*-----------------------------------------------------------------------------*/
int CfgFinished( void ) {

  /*---------------------------------------------------------------------------*/
  return( CbfCleanFinished() );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgFromFile */
/*-----------------------------------------------------------------------------*/
CfgBool_t CfgFromFile( char *confTag ) {
  char *tag;

  /*-------------------------- Search for . to manage configuration from file */
  return( (tag = strrchr(confTag, '.')) && tag );
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgGetCmName */
/*-----------------------------------------------------------------------------*/
char *CfgGetCmName( void ) {
  /*---------------------------------------------------------------------------*/
  return( CfgDataGetCmName(CfgData));
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgGetConfig */
/*-----------------------------------------------------------------------------*/
char *CfgGetConfig( void ) {
  char *config;

  /*----------------------------------------- Read the requested configuration */
  if( !(config = CfgDataGetConfig( CfgData )) ) CfgConfigLoad(NULL);
  /*---------------------------------------------------------------------------*/
  return( CfgDataGetConfig( CfgData ) );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgGetDebug */
/*-----------------------------------------------------------------------------*/
int CfgGetDebug( void ) {
  /*---------------------------------------------------------------------------*/
  return( CfgDataGetDebug(CfgData));
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgGetLogStream */
/*-----------------------------------------------------------------------------*/
FILE *CfgGetLogStream( void ) {

  /*---------------------------------------------------------------------------*/
  return( CfgDataGetLogStream(CfgData) );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgGetLogStreamFlush */
/*-----------------------------------------------------------------------------*/
FILE *CfgGetLogStreamFlush( double tmo ) {

  /*---------------------------------------------------------------------------*/
  if( !tmo ) tmo = -0.005;
  if( tmo > 0 ) tmo *= -1;
  CfgDataMsgRSend( CfgData, tmo );
  /*---------------------------------------------------------------------------*/
  return( CfgDataGetLogStream(CfgData) );
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------- CfgGetRequestedState */
/*-----------------------------------------------------------------------------*/
CfgServerState_t CfgGetRequestedState( void ) {
  /*---------------------------------------------------------------------------*/
  return( CfgDataGetRequestedState(CfgData));
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgGetState */
/*-----------------------------------------------------------------------------*/
CfgServerState_t CfgGetState( void ) {
  /*---------------------------------------------------------------------------*/
  return( CfgDataGetState(CfgData));
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------------- CfgIdle */
/*-----------------------------------------------------------------------------*/
int CfgIdle( int argc, char *argv[], int msgNb ) {

  if( argc < 2 ){
    fprintf(stderr, "%s <Config<CmName>[.cfg]> ... <CmName>\n", argv[0] );
    return(CFG_FAIL);
  }
  /*---------------------------------------------------------------------------*/

  /*----------------------------------------------------- Basic initialisation */
  if( !(CfgData = CfgDataNew(argc, argv, msgNb )) ) {
    CfgMsgAddFatal( "CfgIdle> CfgDataNew fails" );
    return(CFG_FAIL);
  }
  signal(SIGPIPE, SIG_IGN); /*----------- Managed by each libraies(Cm, Cl .. ) */

  CfgIdleBasic( CfgData );
  /*---------------------------------------------------------------------------*/
  if( CfgIdleParse( CfgConfigLoad(NULL) ) != CFG_OK ) {
    CfgMsgAddFatal( "CfgIdle> CfgIdleParse fails" );
    return(CFG_FAIL);
  }
  /*---------------------------------------------------------------------------*/ 
  CfgIdleCm( CfgData );
  /*---------------------------------------------------------------------------*/
  CfgDataSetStateFnt( CfgData, CfgServerIdle, CfgServerEnd,
		      (CfgStateReachFnt_t)CfgEnd, CfgData );
  /*---------------------------------------------------------------------------*/
  CfgDataSetRFlag( CfgData, CfgTrue, CfgTrue, -1 );
  CfgMsgSendWithTimeout( -1 );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------- CfgIdleWithUserActivationFnt */
/*-----------------------------------------------------------------------------*/
int CfgIdleWithUserActivationFnt( int argc, char *argv[], int msgNb,
				  CfgUserActivationGetFnt_t f, void *param ) {

  if( argc < 2 ){
    fprintf(stderr, "%s <Config<CmName>[.cfg]> ... <CmName>\n", argv[0] );
    return(CFG_FAIL);
  }
  /*---------------------------------------------------------------------------*/

  /*----------------------------------------------------- Basic initialisation */
  if( !(CfgData = CfgDataNew(argc, argv, msgNb )) ) {
    CfgMsgAddFatal( "CfgIdleWithUserActivationFnt> CfgDataNew fails" );
    return(CFG_FAIL);
  }
  signal(SIGPIPE, SIG_IGN); /*----------- Managed by each libraies(Cm, Cl .. ) */

  CfgIdleBasic( CfgData );
  /*---------------------------------------------------------------------------*/
  if( CfgIdleParse( CfgConfigLoad(NULL) ) != CFG_OK ) {
    CfgMsgAddFatal( "CfgIdle> CfgIdleParse fails" );
    return(CFG_FAIL);
  }
  CfgDataSetUserActivationFnt( CfgData, f, param );
  /*---------------------------------------------------------------------------*/ 
  CfgIdleCm( CfgData );
  /*---------------------------------------------------------------------------*/
  CfgDataSetStateFnt( CfgData, CfgServerIdle, CfgServerEnd,
		      (CfgStateReachFnt_t)CfgEnd, CfgData );
  /*---------------------------------------------------------------------------*/
  CfgDataSetRFlag( CfgData, CfgTrue, CfgTrue, -1 );
  CfgMsgSendWithTimeout( -1 );
  /*---------------------------------------------------------------------------*/
  return(CFG_OK);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgLogFPrint */
/*-----------------------------------------------------------------------------*/
int CfgLogFPrint( char *fmt, ... ) {
  va_list args;
  int rtn; 

  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgDataLogFPrintv( CfgData, fmt, args );
  va_end( args );
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgMsgAddFmtv */
/*-----------------------------------------------------------------------------*/
int CfgMsgAddFmtv(int errLvl, int debug, char *tag, char *fmt, va_list args) {
 
  /*---------------------------------------------------------------------------*/
  return(CfgDataMsgAddFmtv( CfgData, errLvl, debug, tag, fmt, args ));
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------------- CfgMsgAddFmt */
/*-----------------------------------------------------------------------------*/
int CfgMsgAddFmt(int errLvl, int debug, char *fmt, ... ) {
  va_list args;
  int rtn;

  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgMsgAddFmtv( errLvl, debug, NULL, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgMsgAddFmtTag */
/*-----------------------------------------------------------------------------*/
int CfgMsgAddFmtTag(int errLvl, int debug, char *tag, char *fmt, ... ) {
  va_list args;
  int rtn;

  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgMsgAddFmtv( errLvl, debug, tag, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgMsgAddInfo */
/*-----------------------------------------------------------------------------*/
int CfgMsgAddInfo(char *fmt, ... ) {
  va_list args;
  int rtn;

  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgMsgAddFmtv( CFG_INFO, 0, NULL, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CfgMsgAddError */
/*-----------------------------------------------------------------------------*/
int CfgMsgAddError(char *fmt, ... ) {
  va_list args;
  int rtn;

  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgMsgAddFmtv( CFG_ERROR, 0, NULL, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------- CfgMsgAddFatal */
/*-----------------------------------------------------------------------------*/
int CfgMsgAddFatal(char *fmt, ... ) {
  va_list args;
  int rtn;
 
  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgMsgAddFmtv( CFG_FATAL, 0, NULL, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------- CfgMsgAddUserInfo */
/*-----------------------------------------------------------------------------*/
int CfgMsgAddUserInfo(char *fmt, ... ) {
  va_list args;
  int rtn; 
 
  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgMsgAddFmtv( CFG_USER_INFO, 0, NULL, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgMsgAddWarning */
/*-----------------------------------------------------------------------------*/
int CfgMsgAddWarning(char *fmt, ... ) {
  va_list args;
  int rtn; 

  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgMsgAddFmtv( CFG_WARNING, 0, NULL, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}



/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgMsgPrintInfo */
/*-----------------------------------------------------------------------------*/
int CfgMsgPrintInfo(char *fmt, ... ) {
  va_list args;
  int rtn; 

  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgMsgAddFmtv( CFG_INFO, CFGSEND, NULL, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*----------------------------------------------------------- CfgMsgPrintError */
/*-----------------------------------------------------------------------------*/
int CfgMsgPrintError(char *fmt, ... ) {
  va_list args;
  int rtn; 

  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgMsgAddFmtv( CFG_ERROR, CFGSEND, NULL, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*--------------------------------------------------------- CfgMsgPrintWarning */
/*-----------------------------------------------------------------------------*/
int CfgMsgPrintWarning(char *fmt, ... ) {
  va_list args;
  int rtn;

  /*---------------------------------------------------------------------------*/
  va_start( args, fmt );
  rtn = CfgMsgAddFmtv( CFG_WARNING, CFGSEND, NULL, fmt, args );
  va_end(args);
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------ CfgMsgSendWithTimeout */
/*-----------------------------------------------------------------------------*/
int CfgMsgSendWithTimeout( double tmo ) {
  int nbInfo, nbLog;

  nbInfo = CfgOutGroupNbConsumerGet( CFG_CM_UI_RQ_INFO );
  nbLog  = CfgOutGroupNbConsumerGet( CFG_CM_UI_RQ_LOG );
  if( UI_flag > -1 && nbInfo < 1 &&  nbLog < 1 ) {
    UI_flag = -1; 
    CfgMsgAddInfo("CfgMsgSendWithTimoeut> UI_flag reseted -"
		  " nbInfo %d, nbLog %d", nbInfo, nbLog );
  }
  /*---------------------------------------------------------------------------*/
  return(CfgDataMsgRSend( CfgData, tmo ));
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgParseIdleAdd */
/*-----------------------------------------------------------------------------*/
int CfgParseIdleAdd( CfgParse_t *p ) {
  int rtn = CFG_OK;

  if( !p ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  rtn |= CfgParseGetFunctionAdd( p, "CFG_CMCLEANUP_TMO",
				 (int(*)())CfgDataParseCmCleanupTmo,
				 CfgData, 1, CfgReal );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_CMDOMAIN",
				 (int(*)())CfgDataParseCmDomain,
				 CfgData, 1, CfgString );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_CMSOCKET",
				 (int(*)())CfgDataParseCmSocket,
				 CfgData, 3, CfgDec, CfgDec, CfgString );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_DEBUG", 
				 (int(*)())CfgDataParseDebug,
				 CfgData, 1, CfgDec );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_NODBSAVE",
				 (int(*)())CfgDataParseNoDbSave,
				 CfgData, 1, CfgDec );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_NOFILESAVE",
				 (int(*)())CfgDataParseNoFileSave,
				 CfgData, 1, CfgDec );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_PRIO",
				 (int(*)())CfgDataParsePrio, 
				 CfgData, 1, CfgDec );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_PWD", (int(*)())CfgDataParsePWD,
				 CfgData, 1, CfgString );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_RFLAG",
				 (int(*)())CfgDataParseRFlag, CfgData,
				 CFG_VA_LIST_IDX(2,1), CfgReal, CfgString );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_SCHEDAFFINITY",
				 (int(*)())CfgDataParseSchedAffinity, 
				 CfgData, 1, CfgString );
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------- CfgParseIdleAddNoAction */
/*-----------------------------------------------------------------------------*/
int CfgParseIdleAddNoAction( CfgParse_t *p ) {
  int rtn = CFG_OK;

  if( !p ) return(CFG_FAIL);
  /*---------------------------------------------------------------------------*/
  rtn |= CfgParseGetFunctionAdd( p, "CFG_CMCEANUP_TMO",
				 (int(*)())CfgDataParseNoAction,
				 CfgData, 1, CfgReal );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_CMDOMAIN",
				 (int(*)())CfgDataParseNoAction,
				 CfgData, 1, CfgString );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_CMSOCKET",
				 (int(*)())CfgDataParseNoAction,
				 CfgData, 3, CfgDec, CfgDec, CfgString );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_DEBUG", 
				 (int(*)())CfgDataParseNoAction,
				 CfgData, 1, CfgDec );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_NODBSAVE",
				 (int(*)())CfgDataParseNoAction,
				 CfgData, 1, CfgDec );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_NOFILESAVE",
				 (int(*)())CfgDataParseNoAction,
				 CfgData, 1, CfgDec );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_PRIO",
				 (int(*)())CfgDataParseNoAction, 
				 CfgData, 1, CfgDec );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_PWD", (int(*)())CfgDataParseNoAction,
				 CfgData, 1, CfgString );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_RFLAG", 
				 (int(*)())CfgDataParseNoAction, CfgData,
				 CFG_VA_LIST_IDX(2,1), CfgReal, CfgString );
  rtn |= CfgParseGetFunctionAdd( p, "CFG_SCHEDAFFINITY",
				 (int(*)())CfgDataParseNoAction, 
				 CfgData, 1, CfgString );
  /*---------------------------------------------------------------------------*/
  return(rtn);
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgReachState */
/*-----------------------------------------------------------------------------*/
int CfgReachState( CfgServerState_t state ) {
  CfgServerState_t sState, lState;
  int rtn; 

  if( (sState = CfgDataGetState(CfgData)) == state ) return(state);
  /*---------------- Transition Configured to Active: Commit the configuration */
  if( sState == CfgServerConfigured && state == CfgServerActive ) {
    CfgConfigCommit( NULL );
  }
  
  /*---------------------------------------------------------------------------*/
  rtn = CfgDataReachState(CfgData, state );
  lState = CfgDataGetState(CfgData);
  /*---------------------------------------------------------------------------*/
  CfgMsgPrintInfo("CfgReachState> %s(%s) %s",
		  CfgDataGetStateStr(CfgData, state),
		  CfgDataGetStateStr(CfgData, lState),
		  (rtn == CFG_OK ? "Ok" : "Fail") );
  /*---------------------------------------------------------------------------*/
  return( rtn );
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgRuntimeStr */
/*-----------------------------------------------------------------------------*/
char *CfgRunTimeStr( char *liveTime, size_t len ) {
  
  /*---------------------------------------------------------------------------*/
  return( CfgRunTimeStrBuilt( CfgDataGetStartTime(CfgData), liveTime, len) ); 
}


/*-----------------------------------------------------------------------------*/
/*------------------------------------------------------------ CfgSetConfigFnt */
/*-----------------------------------------------------------------------------*/
CfgConfigFnt_t CfgSetConfigFnt( CfgConfigFnt_t configFnt ) {
  
  /*---------------------------------------------------------------------------*/
  return( CfgDataSetConfigFnt(CfgData, configFnt) );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgSetDebug */
/*-----------------------------------------------------------------------------*/
int CfgSetDebug( int debug ) {
  
  /*---------------------------------------------------------------------------*/
  return( CfgDataSetDebug(CfgData, debug) );
}


/*-----------------------------------------------------------------------------*/
/*-------------------------------------------------------------- CfgReachState */
/*-----------------------------------------------------------------------------*/
int CfgSetStateFnt( int cState, int rState,
			CfgStateReachFnt_t f, void *q ) {
  
  /*---------------------------------------------------------------------------*/
  return( CfgDataSetStateFnt( CfgData, cState, rState, f, q ) );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgSetRFlag */
/*-----------------------------------------------------------------------------*/
int CfgSetRFlag( int log, int stdOut, int cmTrace ) {
  
  /*---------------------------------------------------------------------------*/
  return( CfgDataSetRFlag(CfgData, log, stdOut, cmTrace) );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------- CfgSetUserActivationFnt */
/*-----------------------------------------------------------------------------*/
int CfgSetUserActivationFnt( CfgUserActivationGetFnt_t f, void *param ) {
  
  /*---------------------------------------------------------------------------*/
  return( CfgDataSetUserActivationFnt( CfgData, f, param ) );
}


/*-----------------------------------------------------------------------------*/
/*---------------------------------------------------------------- CfgTerminate */
/*------------------------------------------------------------------------------*/
void CfgTerminate( void ) {

  CfgMsgAddInfo("CfgTerminate> set");
  CbfTerminate();
  /*---------------------------------------------------------------------------*/
  return;
}
