/*---------------------------------------------------------------------------*/
/*        Copyright (c) 1996 LAL Orsay, UPS-IN2P3-CNRS (France).             */
/*                                                                           */
/* Redistribution and use in source and binary forms, with or without        */
/* modification, are permitted provided that the following conditions        */
/* are met:                                                                  */
/* 1. Redistributions of source code must retain the above copyright         */
/*    notice, this list of conditions and the following disclaimer.          */
/* 2. Redistributions in binary form must reproduce the above copyright      */
/*    notice, this list of conditions and the following disclaimer in the    */
/*    documentation and/or other materials provided with the distribution.   */
/* 3. All advertising materials mentioning features or use of this software  */
/*    must display the following acknowledgement:                            */
/*      This product includes software developed by the Computer Application */
/*      Development Group at LAL Orsay (Laboratoire de l'Accelerateur        */
/*      Linaire - UPS-IN2P3-CNRS).                                           */
/* 4. Neither the name of the Institute nor of the Laboratory may be used    */
/*    to endorse or promote products derived from this software without      */
/*    specific prior written permission.                                     */
/*                                                                           */
/* THIS SOFTWARE IS PROVIDED BY THE LAL AND CONTRIBUTORS ``AS IS'' AND       */
/* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE     */
/* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR        */
/* PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE LAL OR CONTRIBUTORS BE      */
/* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR       */
/* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF      */
/* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS  */
/* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN   */
/* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)   */
/* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF    */
/* THE POSSIBILITY OF SUCH DAMAGE.                                           */
/*---------------------------------------------------------------------------*/
#include <CmMessage.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

int vprintf (const char *format, va_list ap);

/*
  Regular expressions. This is currently used only
  for CmMessageServerWaitXxx
  */
#include <Reg.h>

/*
  Conversion mechanisms (a-la-XDR)
  */
#include <Cvt.h>
#include <CmHandle.h>

/*----------------------------------------------------------------*/
#define FRAME_SIZE 2048        /* Increment for message buffer    */
#define MAGIC_WORD 0xDEADDEAD  /* Marker in the message frame     */
#define ALIGNMENT 8
#define CmMessageTypeSize 16   /* Max visible length of message 
                                 types (including the trailing 0) */

typedef enum
{
  MessageOk = 0,
  MessageError,
  MessageNoBuffer,
  MessageUnsufficientBuffer,
  MessageIncompleteData,
  MessageOverlay,
  MessageBadHead,
  MessageBadTail,
  MessageBadLength,
  MessageBadData,
  MessageBadConnect
} Status;

typedef struct _CmMessageHeadRec* CmMessageHead;
typedef struct _CmMessageHeadRec
{
  CvtSwap swap;                       /* The conversion sequence order       */
  unsigned char magic[sizeof(int)];   /* A magic marker                      */
  unsigned char length[sizeof(int)];  /* Total length of the message         */
  unsigned char tail[sizeof(int)];    /* Offset to tail                      */
  char type[CmMessageTypeSize];       /* Truncated message type              */
} CmMessageHeadRec;

typedef struct _CmMessageTailRec* CmMessageTail;
typedef struct _CmMessageTailRec
{
  unsigned char magic[sizeof(int)];   /* Just the same magic marker as in Head */
} CmMessageTailRec;

#define SIZEOF_HEAD (sizeof (CmMessageHeadRec))
#define SIZEOF_TAIL (sizeof (CmMessageTailRec))
#define SIZEOF_PROTOCOL (SIZEOF_HEAD+1+SIZEOF_TAIL)

typedef struct _CmMessageRec
{
  CmHandle         connect;
  char*            data;      /* The complete message buffer         */
  
  char*            start;     /* The frame being read                */
  char*            current;   /* The current reader position         */
  
  int              allocated;
  int              bytes;
  CmHandle         cvt;
  Status           status;
  CmMessageState   state;
  int              iterators;
  List             arrayRefList;
} CmMessageRec;

/*----------------------------------------------------------------*/
/*      MessageIterators.                                         */
/*----------------------------------------------------------------*/
/*                                                                */
/*  A MessageIterator describes the state of Message being        */
/* sent to a Connect.                                             */
/*                                                                */
/*  While this occur, the Message is in the state Sending.        */
/*                                                                */
/*  There is a sub-iterator running on ArrayRefList and a         */
/* sub-state for sending the part of the message before each      */
/* external array (or between two external arrays)                */
/*                                                                */
/*  Then for each portion of the data being sent, there is        */
/* the number of bytes remaining since not all the packet's bytes */
/* are always possible to send (the system buffers may not be     */
/* big enough for any packet size)                                */
/*                                                                */
/*----------------------------------------------------------------*/

typedef struct _CmMessageIteratorRec
{
  CmHandle         message;
  CmHandle         connect;
  char*            start;
  int              bytes;
  Entry*           entry;
  int              prefix_sent;
  int              finished;
  int              dead;
  CmMessageHandler handler;
  CmHandle         message_copy;
  CmHandle         connect_copy;
} CmMessageIteratorRec;

/*----------------------------------------------------------------*/
/*      Array management.                                         */
/*----------------------------------------------------------------*/

typedef struct _CmArrayRec* CmArray;
typedef struct _CmArrayRec
{
  unsigned char type[sizeof(int)];
  unsigned char elements[sizeof(int)];
} CmArrayRec;

typedef struct _CmArrayRefRec* CmArrayRef;
typedef struct _CmArrayRefRec
{
  int offset;
  void* address;
} CmArrayRefRec;

static CmArrayRef CmArrayRefNew (int offset, void* address);
static void CmArrayRefDelete (CmArrayRef array_ref);

/*----------------------------------------------------------------*/

/*----------------------------------------------------------------*/
/*      Local variables.                                          */
/*----------------------------------------------------------------*/

/*----------------------------------------------------------------*/
static int CmMessageInitialized = 0;

static int CmMessageCount         = 0;
static int CmMessageDataCount     = 0;
static int CmHandlerCount         = 0;
static int CmArrayRefCount        = 0;
static int CmMessageIteratorCount = 0;
static int CmConnectDebug         = 0;

static GCClass CmHandlerGCClass = NULL;

static int Size[] = {
  sizeof(char),
  sizeof(short),
  sizeof(int),
  sizeof(float),
  sizeof(double)
};
/*----------------------------------------------------------------*/

/*----------------------------------------------------------------*/
/*      Local static functions.                                   */
/*----------------------------------------------------------------*/

static void* CmMessageExtend (CmMessage message, int bytes);
static int CmMessageAlign (int offset);
static void CmMessagePutSynchro (CmMessage message);
static int CmMessageGetSynchro (CmMessage message);
static void CmMessagePutItemType (CmMessage message, CmMessageItemType type);
static int CmMessageCheckItemType (CmMessage message, CmMessageItemType type);
static void CmMessagePutHeader (CmMessage message);
static void CmMessagePutTail (CmMessage message);
static void CmMessageSkipHeader (CmMessage message);
static char* CmMessageHeadGetSwap (CmMessageHead message);
static int CmMessageHeadGetMagic (CmMessageHead message, Cvt cvt);
static int CmMessageHeadGetLength (CmMessageHead message, Cvt cvt);
static void CmMessageHeadAddToLength (CmMessageHead message, 
                                      Cvt cvt, 
                                      int bytes);
static void CmMessageHeadSetTail (CmMessageHead message, int offset);
static int CmMessageHeadGetTail (CmMessageHead message, Cvt cvt);
static char* CmMessageHeadGetType (CmMessageHead message);
static void CmMessageHeadSetType (CmMessageHead message, char* type);
static int CmMessageTailGetMagic (CmMessageTail message, Cvt cvt);
static void CmMessageInstallArrayRef (CmMessage message, CmArrayRef ref);

static CmMessageStatus MessageDebugControl (CmMessage message,
                                            char* sender,
                                            char* serverName);

static void CmMessageDump (CmMessage message, char* name, int from);

static CmMessageIterator CmMessageIteratorNew (CmMessage message,
                                               CmConnect connect,
                                               CmMessageHandler handler);
static void CmMessageIteratorFirst (CmMessageIterator iterator);

static CmConnectStatus CmMessageReceiveHandler (CmConnect connect);
static CmConnectStatus CmMessageRearmHandler (CmConnect connect);
static Status CmMessageDoReceive (CmMessage message, CmConnect connect);
static int CmMessageExtendForReception (CmMessage message, int needed);
static int CmMessageReceiveData (CmMessage message, CmConnect connect,
                                 int data);
static void* CmMessageMove (CmMessage message, int bytes);

/*----------------------------------------------------------------*/
/*      Handler management.                                       */
/*----------------------------------------------------------------*/

typedef struct _CmHandlerRec* CmHandler;
typedef struct _CmHandlerRec
{
  char type[CmMessageTypeSize];
  CmMessageHandler address;
} CmHandlerRec;

static char* CmHandlerNamer (CmHandler handler);
static char* CmHandlerBuildName (char* type);
static CmHandler CmHandlerNew (char* type, CmMessageHandler function);
static CmHandler CmHandlerFind (char* type);
static void CmHandlerDelete (CmHandler handler);


/*----------------------------------------------------------------*/
/*      Static objects.                                           */
/*----------------------------------------------------------------*/

#define CmHandlerTableSize 50

static struct
{
  CHasher          handlers;
  CmMessageHandler defaultHandler;
  RegExp           filter;
  int              traceOn;
  CmConnectPrinter printer;
} CmMessageMgr;

static CmMessageItemType ItemTypes[] = 
{
  CmMessageItemChar,
  CmMessageItemShort,
  CmMessageItemInt,
  CmMessageItemLong,
  CmMessageItemFloat,
  CmMessageItemDouble,
  CmMessageItemArray,
  CmMessageItemText,
  CmMessageItemBytes,
  CmMessageItemTail,
  CmMessageItemUnknown
};




/*----------------------------------------------------------------*/
/*                                                                */
/*  Global management.                                            */
/*                                                                */
/*----------------------------------------------------------------*/











/*----------------------------------------------------------------*/
int CmMessageForceStartup (char* name)
/*----------------------------------------------------------------*/
{
  CmConnectForceStartup ();
  CmMessageInitialized = 0;
  return (CmMessageStartup (name));
}

/*----------------------------------------------------------------*/
void CmMessageInitialize ()
/*----------------------------------------------------------------*/
{
  if (!CmMessageInitialized)
    {
      char* trace;

      CmMessageInitialized = 1;

      CmMessageMgr.handlers       = NULL;
      CmMessageMgr.defaultHandler = 0;
      CmMessageMgr.filter         = NULL;
      CmMessageMgr.traceOn        = 0;
      CmMessageMgr.printer        = (CmConnectPrinter) vprintf;
      CmConnectDebug              = 0;

      {
        char* env;

        env = getenv ("CMCONNECTDEBUG");
        if (env != NULL)
          {
            CmConnectDebug = 0;

            sscanf (env, "%d", &CmConnectDebug);
          }
      }

      {
        GCDeleteMethod erase = (GCDeleteMethod) CmHandlerDelete;
        
        CmHandlerGCClass = GCClassNew ("CmHandler", erase);
      }
      
      GCReferenceAssign (CmMessageMgr.handlers,
                         CStrHasherNew (CmHandlerTableSize,
                                        (CValuator) CmHandlerNamer));
      
      trace = getenv ("CMMESSAGEDEBUG");
      if (trace)
        {
          sscanf (trace, "%d", &CmMessageMgr.traceOn);
        }
      
      CmMessageInstallHandler ((CmMessageHandler) MessageDebugControl,
                               "CmMessageDebug");

      CmConnectInitialize ();
    }
}

/*----------------------------------------------------------------*/
int CmMessageStartup (char* name)
/*----------------------------------------------------------------*/
{
  printf ("Cm> This function is now obsolete.\n");
  printf ("Cm> Please use CmMessageOpenServer instead.\n");
  return (0);
}

/*----------------------------------------------------------------*/
int CmMessageStartupMultiple (char* name)
/*----------------------------------------------------------------*/
{
  printf ("Cm> This function is now obsolete.\n");
  printf ("Cm> Please use CmMessageOpenMultipleServer instead.\n");
  return (0);
}

/*----------------------------------------------------------------*/
int CmMessageOpenServer (char* name)
/*----------------------------------------------------------------*/
{
  CmConnect me;

  CmMessageInitialize ();
  
  me = CmConnectOpenServer (name);
  if (me == NULL) return (0);

  return (1);
}

/*----------------------------------------------------------------*/
int CmMessageOpenMultipleServer (char* name)
/*----------------------------------------------------------------*/
{
  CmConnect me;

  CmMessageInitialize ();
  
  me = CmConnectOpenMultipleServer (name);
  if (me == NULL) return (0);

  return (1);
}

/*----------------------------------------------------------------*/
int CmMessageCloseServer (char* name)
/*----------------------------------------------------------------*/
{
  CmMessageCleanup ();
  return (0);
}

/*----------------------------------------------------------------*/
void CmMessageCleanup ()
/*----------------------------------------------------------------*/
{
  if (CmMessageInitialized)
    {
      CHasherErase (CmMessageMgr.handlers);
      GCReferenceClear (CmMessageMgr.handlers);

      if (CmMessageMgr.filter != NULL)
        {
          RegExpDelete (CmMessageMgr.filter);
          CmMessageMgr.filter = NULL;
        }
      
      CmMessageInitialized = 0;

      CmConnectCleanup ();
      
      CmMessageCheckBalance ();
    }
}

/*--------------------------------------------------------------*/
void CmMessageCheckBalance ()
/*--------------------------------------------------------------*/
{
  GCCleanup ();

  if (CmMessageCount != 0)
    {
      fprintf (stderr, "Cm>>> Bad balance on CmMessage : %d\n",
               CmMessageCount);
    }
      
  if (CmMessageDataCount != 0)
    {
      fprintf (stderr, "Cm>>> Bad balance on CmMessageData : %d\n",
               CmMessageDataCount);
    }
      
  if (CmHandlerCount != 0)
    {
      fprintf (stderr, "Cm>>> Bad balance on CmHandler : %d\n",
               CmHandlerCount);
    }
      
  if (CmArrayRefCount != 0)
    {
      fprintf (stderr, "Cm>>> Bad balance on CmArrayRef : %d\n",
               CmArrayRefCount);
    }
      
  if (CmMessageIteratorCount != 0)
    {
      fprintf (stderr, "Cm>>> Bad balance on CmMessageIterator : %d\n",
               CmMessageIteratorCount);
    }
  
  CmConnectCheckBalance ();
}

/*----------------------------------------------------------------*/
CmConnectPrinter CmMessageInstallPrinter (CmConnectPrinter printer)
/*----------------------------------------------------------------*/
{
  CmConnectPrinter old;

  CmMessageInitialize ();

  old = CmMessageMgr.printer;

  CmMessageMgr.printer = printer;

  CmConnectInstallPrinter (printer);

  return (old);
}





/* private functions ---------------------------------------------*/




/*----------------------------------------------------------------*/
static int DoPrint (char* format, ...)
/*----------------------------------------------------------------*/
{
  va_list args;

  if (CmMessageMgr.printer)
    {
      va_start (args, format);
  
      CmMessageMgr.printer (format, args);
      
      va_end (args);
    }
  return (0);
}









/*----------------------------------------------------------------*/
/*                                                                */
/*  Message object management.                                    */
/*                                                                */
/*----------------------------------------------------------------*/









/*----------------------------------------------------------------*/
CmMessage CmMessageNew ()
/*----------------------------------------------------------------*/
{
  CmMessage message = (CmMessage) malloc (sizeof (CmMessageRec));

  if (message == NULL)
    {
      DoPrint ("Cm> Malloc error on Message.\n");
    }
  else
    {
      CmMessageCount++;

      memset (message, 0, sizeof (CmMessageRec));
      message->state = CmMessageIsNew;

      CmConnectHandleInstall (&message->connect);
      CmCvtHandleInstall (&message->cvt);
      ListInit (&message->arrayRefList);
    }

  return (message);
}

/*----------------------------------------------------------------*/
void CmMessageDelete (CmMessage message)
/*----------------------------------------------------------------*/
{
  if (message == NULL) return;

  CmConnectHandleUninstall (&message->connect);
  CmCvtHandleUninstall (&message->cvt);
  
  if (message->data != NULL)
    {
      free (message->data);
      CmMessageDataCount--;
      
      message->data    = NULL;
      message->start   = NULL;
      message->current = NULL;
    }
  
  if (ListCount (&message->arrayRefList))
    {
      Entry* entry;

      entry = message->arrayRefList.first;
      while (entry != NULL)
        {
          CmArrayRef ref = (CmArrayRef) entry->reference;
          if (ref != NULL)
            {
              CmArrayRefDelete (ref);
            }

          entry = entry->next;
        }

      ListClear (&message->arrayRefList);
    }

  memset (message, 0, sizeof (CmMessageRec));
  message->state = CmMessageIsDead;
  free (message);
  CmMessageCount--;
}

/*----------------------------------------------------------------*/
void CmMessageReset (CmMessage message)
/*----------------------------------------------------------------*/
/*                                                                */
/* o Reset the indices to the beginning of the allocated buffer.  */
/* o Only used for messages to be sent.                           */
/*                                                                */
/*----------------------------------------------------------------*/
{
  if (!message) return;

  if (message->state == CmMessageIsSending) return;
  
  message->start   = message->data;
  message->current = message->data;
  message->bytes   = 0;

  if (ListCount (&message->arrayRefList))
    {
      Entry* entry;

      entry = message->arrayRefList.first;
      while (entry != NULL)
        {
          CmArrayRef ref = (CmArrayRef) entry->reference;
          if (ref != NULL)
            {
              CmArrayRefDelete (ref);
            }

          entry = entry->next;
        }

      ListClear (&message->arrayRefList);
    }

  message->state = CmMessageIsNew;
}

/*----------------------------------------------------------------*/
void CmMessageSetType (CmMessage message, char* type)
/*----------------------------------------------------------------*/
{
  if (!message) return;

  if (message->state == CmMessageIsSending) return;
  
  /*
     Extend est utilise ici pour garantir que le header est
     installe.
     Si il l'est deja, il ne se passe rien.
  */
  CmMessageExtend (message, 0);

  CmMessageHeadSetType ((CmMessageHead) message->start, type);
}

/*----------------------------------------------------------------*/
char* CmMessageGetType (CmMessage message)
/*----------------------------------------------------------------*/
{
  if (!message) return (0);
  if (!message->start) return (0);
  if (message->state == CmMessageIsDead) return (0);

  return (CmMessageHeadGetType ((CmMessageHead) message->start));
}

/*----------------------------------------------------------------*/
CmMessageState CmMessageGetState (CmMessage message)
/*----------------------------------------------------------------*/
{
  if (!message) return (CmMessageIsDead);
  return (message->state);
}

/*----------------------------------------------------------------*/
CmConnect CmMessageGetConnect (CmMessage message)
/*----------------------------------------------------------------*/
{
  if (message == NULL) return (NULL);

  return (CmConnectHandleGet (&message->connect));
}

/*----------------------------------------- Added by F. BELLACHIA */
/*----------------------------------------------------------------*/
int CmMessageGetMessageLength (CmMessage message)
/*----------------------------------------------------------------*/
{
  Cvt cvt;
 
  if (message == NULL) return (0);

  cvt = CmCvtHandleGet (&message->cvt);

  return (CmMessageHeadGetLength ((CmMessageHead) message->start, cvt));
}




/*----------------------------------------------------------------*/
/*                                                                */
/*  Message protocol management.                                  */
/*                                                                */
/*----------------------------------------------------------------*/









/*----------------------------------------------------------------*/
int CmMessageSend (CmMessage message, char* name)
/*----------------------------------------------------------------*/
{
  CmMessageIterator iterator = NULL;
  CmConnectCondition rtn;
  int status;

  iterator = CmMessagePost (message, name, 0);

  if (iterator == NULL) return (0);
  /*--------------------------------------------------------------*/
  do {
    rtn = CmConnectWaitOnIterator( iterator );
/*     DoPrint( "CmMessageSend> to %s - status %d/%d\n", name, rtn, CmConnectNormalCompletion ); */
  } while ( rtn != CmConnectNormalCompletion &&
	    rtn != CmConnectTimeoutDetection &&
	    rtn != CmConnectErrorCondition );

  /*--------------------------------------------------------------*/
  switch( rtn ) {
  case CmConnectTimeoutDetection:
    DoPrint("CmMessageSend> to %s - status CmConnectTimeoutDetection(%d)\n",
	    name, rtn );
    status = 0;
    break;
    
  case CmConnectErrorCondition:
    DoPrint("CmMessageSend> to %s - status CmConnectError(%d)\n",
	    name, rtn );
    status = 0;
    break;

  case CmConnectNormalCompletion:
  default:
    status = 1;
    break;
  }
  
  if( !status ) CmMessageIteratorStop( iterator );
  /*--------------------------------------------------------------*/
  CmMessageIteratorDelete (iterator);

  return (status);
}

/*----------------------------------------------------------------*/
CmMessageIterator CmMessagePost (CmMessage message, char* name,
                                 CmMessageHandler handler)
/*----------------------------------------------------------------*/
{
  CmConnect  connect;
  CmMessageIterator iterator = NULL;

  if (!message) return (0);
  if ((message->state == CmMessageIsDead) ||
      (message->state == CmMessageIsNew)) return (NULL);

  if (!(connect = CmConnectNew (name)))
  {
    DoPrint ("Cm> Unknown destination %s.\n", name);
    message->status = MessageBadConnect;
    return (NULL);
  }

  if (message->data == NULL) return (NULL);

  CmMessagePutTail (message);

  if (CmMessageMgr.traceOn > 0) CmMessageDump (message, name, 0);

  iterator = CmMessageIteratorNew (message, connect, handler);
  CmConnectInstallMessageIterator (connect, iterator);

  if (CmMessageIteratorFinished (iterator))
    {
      DoPrint ("CmMessagePost> Cannot start sending message to %s\n",
               CmConnectGetName(connect));
      CmMessageIteratorStop( iterator );
      CmMessageIteratorDelete( iterator );
      iterator = NULL;
    }

  return (iterator);
}



/* private functions ---------------------------------------------*/



/*----------------------------------------------------------------*/
/*                                                                */
/*  Message buffer management.                                    */
/*                                                                */
/*----------------------------------------------------------------*/








/* private functions ---------------------------------------------*/



/*----------------------------------------------------------------*/
static void* CmMessageExtend (CmMessage message, int bytes)
/*----------------------------------------------------------------*/
/*                                                                */
/* o Extension of the message buffer                              */
/* o Only used for building a message to be sent.                 */
/*                                                                */
/*----------------------------------------------------------------*/
{
  int  currentPosition;
  void* pointer;

  if (!message) return (NULL);
  
  if (message->state == CmMessageIsDead) return (0);
  if (message->state == CmMessageIsSending) return (0);
  
  if (message->state == CmMessageIsClosed) CmMessageReset (message);
  if (message->state == CmMessageIsNew)
  {
    if (message->data == NULL)
      {
        message->allocated = FRAME_SIZE;
        message->data = (char*) malloc (message->allocated);
        if (message->data == NULL)
          {
            DoPrint ("Cm> Memory error.\n");
            
            message->allocated = 0;
          }
        else CmMessageDataCount++;
      }
    CmMessageReset (message);
    CmMessagePutHeader (message);
  }
  
  currentPosition = message->current - message->data;
  if (currentPosition + bytes > message->allocated)
  {
    int frames = ((currentPosition + bytes)/FRAME_SIZE) + 1;
    int startPosition = message->start - message->data;

    message->allocated = frames * FRAME_SIZE;
    message->data = (char*) realloc (message->data, message->allocated);
    if (message->data == NULL)
    {
      DoPrint ("Cm> Memory error.\n");
      CmConnectExit (CmErrorBadMemory);
    }
    message->current = message->data + currentPosition;
    message->start = message->data + startPosition;
  }
  pointer = message->current;
  if (bytes)
    {
      Cvt cvt = CmCvtHandleGet (&message->cvt);

      message->current += bytes;
      message->bytes += bytes;
      CmMessageHeadAddToLength ((CmMessageHead) message->start,
                                cvt, bytes);
    }
  return (pointer);
}

/*----------------------------------------------------------------*/
/*                                                                */
/*  Message handlers.                                             */
/*                                                                */
/*----------------------------------------------------------------*/


/*----------------------------------------------------------------*/
void CmMessagePrepareHandler (CmConnect connect)
/*----------------------------------------------------------------*/
{
  CmConnectInitHandlers (connect,
                         (CmConnectReceiveHandler) CmMessageReceiveHandler,
                         (CmConnectReceiveHandler) CmMessageRearmHandler);
}


/*----------------------------------------------------------------*/
CmMessageHandler CmMessageInstallDefaultHandler (CmMessageHandler function)
/*----------------------------------------------------------------*/
{
  CmMessageHandler old;

  old = CmMessageMgr.defaultHandler;
  CmMessageMgr.defaultHandler = function;
  return (old);
}

/*----------------------------------------------------------------*/
CmMessageHandler CmMessageInstallHandler (CmMessageHandler function,
                                          char* type)
/*----------------------------------------------------------------*/
{
  CmMessageHandler old;
  CmHandler handler;

  if( !CmMessageInitialized ) return(function);

  handler = CmHandlerFind (type);
  if (handler)
  {
    old = handler->address;
    handler->address = function;
  }
  else
  {
    old = NULL;
    handler = CmHandlerNew (type, function);
    if (handler != NULL)
      {
        CHasherAdd (CmMessageMgr.handlers, handler);
      }
  }
  return (old);
}

/*----------------------------------------------------------------*/
void CmMessageUninstallHandler (char* type)
/*----------------------------------------------------------------*/
{
  CmHandler handler;

  if( !CmMessageInitialized ) return;

  handler = CmHandlerFind (type);
  if (handler)
  {
    CHasherRemove (CmMessageMgr.handlers, handler);
  }
}

/*----------------------------------------------------------------*/
int CmMessageAreYouThere (char* name)
/*----------------------------------------------------------------*/
{
  CmConnect connect;

  connect = CmConnectNew (name);

  if (connect != NULL) return (1);
  return (0);
}

/*--------------------------------------------------------------*/
CmConnectCondition CmMessageCheck ()
/*--------------------------------------------------------------*/
{
  return (CmConnectCheck ());
}

/*----------------------------------------------------------------*/
CmConnectCondition CmMessageWait ()
/*----------------------------------------------------------------*/
{
  return (CmConnectWait ());
}

/*----------------------------------------------------------------*/
CmConnectCondition CmMessageWaitWithTimeout (double seconds)
/*----------------------------------------------------------------*/
{
  return (CmConnectWaitWithTimeout (seconds));
}

/*--------------------------------------------------------------*/
CmConnectCondition CmMessageServerCheck (char* pattern)
/*--------------------------------------------------------------*/
{
  CmConnectCondition condition;
  
  if (CmMessageMgr.filter != NULL)
    {
      RegExpDelete (CmMessageMgr.filter);
      CmMessageMgr.filter = NULL;
    }

  CmMessageMgr.filter = RegExpNew (pattern);
  
  condition = CmConnectCheck ();

  if (CmMessageMgr.filter != NULL)
    {
      RegExpDelete (CmMessageMgr.filter);
      CmMessageMgr.filter = NULL;
    }

  return (condition);
}

/*----------------------------------------------------------------*/
CmConnectCondition CmMessageServerWait (char* pattern)
/*----------------------------------------------------------------*/
{
  CmConnectCondition condition;
  
  if (CmMessageMgr.filter != NULL)
    {
      RegExpDelete (CmMessageMgr.filter);
      CmMessageMgr.filter = NULL;
    }

  CmMessageMgr.filter = RegExpNew (pattern);
  
  condition = CmConnectWait ();

  if (CmMessageMgr.filter != NULL)
    {
      RegExpDelete (CmMessageMgr.filter);
      CmMessageMgr.filter = NULL;
    }

  return (condition);
}

/*----------------------------------------------------------------*/
CmConnectCondition CmMessageServerWaitWithTimeout (char* pattern,
                                                   double seconds)
/*----------------------------------------------------------------*/
{
  CmConnectCondition condition;
  
  if (CmMessageMgr.filter != NULL)
    {
      RegExpDelete (CmMessageMgr.filter);
      CmMessageMgr.filter = NULL;
    }

  CmMessageMgr.filter = RegExpNew (pattern);
  
  condition = CmConnectWaitWithTimeout (seconds);

  if (CmMessageMgr.filter != NULL)
    {
      RegExpDelete (CmMessageMgr.filter);
      CmMessageMgr.filter = NULL;
    }

  return (condition);
}

/*----------------------------------------------------------------*/
void CmMessageTraceOn (int level)
/*----------------------------------------------------------------*/
{
  if ((level < 1) && (level > 3)) return;
  
  CmMessageMgr.traceOn = level;
}

/*----------------------------------------------------------------*/
void CmMessageTraceOff ()
/*----------------------------------------------------------------*/
{
  CmMessageMgr.traceOn = 0;
}



/* private functions ---------------------------------------------*/


/*----------------------------------------------------------------*/
static char* CmHandlerNamer (CmHandler handler)
/*----------------------------------------------------------------*/
{
  static char* nullStr = (char*) "";

  if (!handler) return (nullStr);

  return ((char*) handler->type);
}

/*----------------------------------------------------------------*/
static char* CmHandlerBuildName (char* type)
/*----------------------------------------------------------------*/
{
  static char temp[CmMessageTypeSize];
  static char* nullStr = (char*) "";
  int length;
  int maxLength;

  if (!type) return (nullStr);

  maxLength = CmMessageTypeSize - 1;
  length = strlen (type);
  if (length > maxLength) length = maxLength;
  strncpy (temp, type, length);
  temp[length] = '\0';

  return (temp);
}

/*----------------------------------------------------------------*/
static CmHandler CmHandlerNew (char* type, CmMessageHandler function)
/*----------------------------------------------------------------*/
{
  CmHandler handler;

  handler = (CmHandler) malloc (sizeof(CmHandlerRec));
  if (handler == NULL)
    {
      DoPrint ("Cm> Malloc error on Message handler.\n");
      return (NULL);
    }
  
  CmHandlerCount++;
  
  strcpy (handler->type, CmHandlerBuildName (type));

  handler->address = function;

  GCObjectNew (handler, CmHandlerGCClass);

  return (handler);
}

/*----------------------------------------------------------------*/
static CmHandler CmHandlerFind (char* type)
/*----------------------------------------------------------------*/
{
  CmHandler handler;
  char* name;

  if (!type) return (NULL);

  name = CmHandlerBuildName (type);
  handler = (CmHandler) CHasherFind (CmMessageMgr.handlers, name);
  
  return (handler);
}

/*----------------------------------------------------------------*/
static void CmHandlerDelete (CmHandler handler)
/*----------------------------------------------------------------*/
{
  if (!handler) return;

  free (handler);
  CmHandlerCount--;
}

/*----------------------------------------------------------------*/
static CmMessageStatus MessageDebugControl (CmMessage message,
                                            char* sender,
                                            char* serverName)
/*----------------------------------------------------------------*/
{
  char* command;

  while (CmMessageGetItemType (message) == CmMessageItemText)
    {
      command = CmMessageGetText (message);
      
      if (!strcmp (command, "TraceOn")) 
        {
          int level;
          
          level = CmMessageGetInt (message);
          CmMessageTraceOn (level);
        }
      else if (!strcmp (command, "TraceOff")) 
        {
          CmMessageTraceOff ();
        }
      else if (!strcmp (command, "DebugOff")) 
        {
          CmConnectDebugOff ();
        }
      else if (!strcmp (command, "DebugOn")) 
        {
          CmConnectDebugOn ();
        }
      else if (!strcmp (command, "Transactions")) 
        {
          CmDumpTransactions ();
        }
      else if (!strcmp (command, "TerminateTransaction")) 
        {
          int id;

          id = CmMessageGetInt (message);
          CmTerminateTransaction (id);
        }
    }
  return (CmMessageOk);
}

/*----------------------------------------------------------------*/
static char* SynchronizePointer (char* pointer)
/*----------------------------------------------------------------*/
{
  int offset;
  
  offset = *pointer;
  pointer += offset + 1;
  return (pointer);
}

/*----------------------------------------------------------------*/
static void ArrayDump (Cvt cvt,
                       unsigned char* pointer,
                       CmMessageArrayType arrayType,
                       int elements)
/*----------------------------------------------------------------*/
{
  int i;
  int j;
  unsigned int c;
  short s;
  int n;
  float f;
  double d;
  
  for (i = 0; i < elements; i += 16)
    {
      DoPrint ("      ");
      for (j = 0; (j < 16) && ((i + j) < elements); j++)
        {
          switch (arrayType)
            {
              case CmMessageChar:
                c = *((unsigned char*) pointer);
                DoPrint ("%3.3d ", c);
                break;
              case CmMessageShort:
                if (cvt) s = CvtGetShort (cvt, (unsigned char*) pointer);
                else s = *((short*) pointer);

                DoPrint ("%8.8d ", s);
                break;
              case CmMessageInt:
                if (cvt) n = CvtGetInt (cvt, (unsigned char*) pointer);
                else n = *((int*) pointer);

                DoPrint ("%8.8d ", n);
                break;
              case CmMessageFloat:
                if (cvt) f = CvtGetFloat ((unsigned char*) pointer);
                else f = *((float*) pointer);

                DoPrint ("%+08.6g ", f);
                break;
              case CmMessageDouble:
                if (cvt) d = CvtGetDouble ((unsigned char*) pointer);
                else d = *((double*) pointer);

                DoPrint ("%+08.6lg ", d);
                break;
            }
          pointer += Size[arrayType];
        }
      DoPrint ("\n");
    }
}

/*----------------------------------------------------------------*/
static void CmMessageDump (CmMessage message, char* name, int from)
/*----------------------------------------------------------------*/
{
  int length;
  int magic;
  char* messageType;
  char* pointer;
  Cvt cvt;
  Entry* entry;
  
  if (!message) return;
  
  cvt = CmCvtHandleGet (&message->cvt);
  
  if (from)
    DoPrint ("CmMessage received from ");
  else
    DoPrint ("Sending CmMessage to ");

  if (name) DoPrint ("%s" , name);
  else DoPrint ("<unknown>");
  
  magic  = CmMessageHeadGetMagic ((CmMessageHead) message->start, cvt);
  length = CmMessageHeadGetLength ((CmMessageHead) message->start, cvt);
  messageType   = CmMessageHeadGetType ((CmMessageHead) message->start);
  DoPrint (" magic = %08x", magic);
  DoPrint (" length = %d bytes", length);
  DoPrint (" type = '%s'\n", messageType);

  if (CmMessageMgr.traceOn < 2) return;
  
  DoPrint ("  data:\n");

  pointer = message->start + sizeof (CmMessageHeadRec);

  entry = message->arrayRefList.first;

  for (;;)
    {
      CmMessageItemType type;
      int offset;
      
      type = ItemTypes[*pointer];
      pointer += sizeof(CmMessageItem);

      pointer = SynchronizePointer (pointer);
      
      switch (type)
        {
          case CmMessageItemChar :
            {
              char value;

              value = *((char*) pointer);
              DoPrint ("    char    : %c\n", value);
              pointer += sizeof(char);
            }
            break;
          case CmMessageItemShort :
            {
              short value;

              value = CvtGetShort (cvt, (unsigned char*) pointer);
              DoPrint ("    short   : %d\n", value);
              pointer += sizeof(short);
            }
            break;
          case CmMessageItemInt :
            {
              int value;

              value = CvtGetInt (cvt, (unsigned char*) pointer);
              DoPrint ("    int     : %d\n", value);
              pointer += sizeof(int);
            }
            break;
          case CmMessageItemLong :
            {
              long value;
              int bytes;

              bytes = strlen(pointer);
              sscanf (pointer, "%ld", &value);
              DoPrint ("    long    : %ld\n", value);
              pointer += bytes + 1;
            }
            break;
          case CmMessageItemFloat :
            {
              float value;

              value = CvtGetFloat ((unsigned char*) pointer);
              DoPrint ("    float   : %+08.6g\n", value);
              pointer += sizeof(float);
            }
            break;
          case CmMessageItemDouble :
            {
              double value;

              value = CvtGetDouble ((unsigned char*) pointer);
              DoPrint ("    double  : %+08.6lg\n", value);
              pointer += sizeof(double);
            }
            break;
          case CmMessageItemArray :
            {
              CmArray array = NULL;
              CmArrayRef arrayRef = NULL;
              CmMessageArrayType arrayType;
              int elements;
              int bytes;
              int internal;

              array = (CmArray) pointer;
              arrayType = (CmMessageArrayType) CvtGetInt (cvt,
                                                          array->type);
              elements = CvtGetInt (cvt, array->elements);
              bytes = elements * Size[arrayType];
              
              pointer += sizeof (CmArrayRec);
              pointer = SynchronizePointer (pointer);
      
              if (entry != NULL)
                {
                    /* There are some external arrays.                 */
                    /* We check whether this array matches one of them */
                  
                  arrayRef = (CmArrayRef) entry->reference;

                  offset = ((char*) array) - message->data;
                  if (offset == arrayRef->offset)
                    {
                      internal = 0;
                      entry = entry->next;
                    }
                  else 
                    {
                      internal = 1;
                    }
                }
              else 
                {
                  internal = 1;
                }

              if (internal)
                {

                  DoPrint ("    internal array[%d]\n", elements);

                  if (CmMessageMgr.traceOn == 3)
                    {
                       ArrayDump (cvt,
                                  (unsigned char*) pointer,
                                  arrayType,
                                  elements);
                    }

                  pointer += bytes;

                  offset = CmMessageAlign (bytes);
                  pointer += offset;
                }
              else
                {

                  DoPrint ("    external array[%d]\n", elements);
                  if (CmMessageMgr.traceOn == 3)
                    {
                       ArrayDump (NULL,
                                  (unsigned char*) arrayRef->address,
                                  arrayType,
                                  elements);
                    }
                }
            }
            break;
          case CmMessageItemText :
            {
              char* value;
              int bytes;

                /* A Bytes item is stored */
              type = ItemTypes[*pointer];
              pointer += sizeof(CmMessageItem);
              pointer = SynchronizePointer (pointer);
      
                /* A int item is stored */
              type = ItemTypes[*pointer];
              pointer += sizeof(CmMessageItem);
              pointer = SynchronizePointer (pointer);
      
              bytes = CvtGetInt (cvt, (unsigned char*) pointer);
              pointer += sizeof (int);
              
                /* Then the string itself */
              if (bytes > 0)
                {
                  value = pointer;
 
                  DoPrint ("    text    : %s\n", value);
                  pointer += bytes;
                }
              else
                {
                  DoPrint ("    text    : (NULL)\n");
                }
            }
            break;
          case CmMessageItemBytes :
            {
              int bytes;

                /* A int item is stored */
              type = ItemTypes[*pointer];
              pointer += sizeof(CmMessageItem);
              pointer = SynchronizePointer (pointer);
      
              bytes = CvtGetInt (cvt, (unsigned char*) pointer);
              pointer += sizeof (int);
              
                /* Then the string itself */

              DoPrint ("    %d bytes", bytes);
              if ((bytes > 0) && (CmMessageMgr.traceOn == 3))
                {
                  int i;
                  int j;
                  unsigned int c;
                  
                  for (i = 0; i < bytes; i += 16)
                    {
                      if ((i % 16) == 0) DoPrint ("\n      ");
                      for (j = 0; (j < 16) && ((i + j) < bytes); j++)
                        {
                          c = *((unsigned char*) pointer);
                          DoPrint ("%3.3d ", c);
                          pointer++;
                        }
                    }
                  DoPrint ("\n");
                }
              else pointer += bytes; 
            }
            break;
          case CmMessageItemTail :
            break;
	case CmMessageItemUnknown :
            break;
        }
      if (type == CmMessageItemTail) break;
    }
  
  DoPrint ("---------------------------\n");
}











/*----------------------------------------------------------------*/
/*                                                                */
/*   Message contents manipulations.                              */
/*                                                                */
/*----------------------------------------------------------------*/




/*----------------------------------------------------------------*/
CmMessageItemType CmMessageGetItemType (CmMessage message)
/*----------------------------------------------------------------*/
{
  CmMessageItem* pointer;
  CmMessageItemType type;

  pointer = (CmMessageItem*) CmMessageMove (message, 0);
  if (!pointer) return (CmMessageItemUnknown);

  type = ItemTypes[*pointer];
  return (type);
}

/*----------------------------------------------------------------*/
void CmMessagePutChar (CmMessage message, char value)
/*----------------------------------------------------------------*/
{
  char* pointer;

  CmMessagePutItemType (message, CmMessageItemChar);
  pointer = (char*) CmMessageExtend (message, sizeof(char));
  if (pointer) *pointer = value;
}

/*----------------------------------------------------------------*/
char CmMessageGetChar (CmMessage message)
/*----------------------------------------------------------------*/
{
  char* pointer;
  char value = 0;

  if (!CmMessageCheckItemType (message, CmMessageItemChar)) return (0);
  
  pointer = (char*) CmMessageMove (message, sizeof(char));
  if (pointer) value = *pointer;
  return (value);
}

/*----------------------------------------------------------------*/
void CmMessagePutShort (CmMessage message, short value)
/*----------------------------------------------------------------*/
{
  unsigned char* pointer;

  CmMessagePutItemType (message, CmMessageItemShort);
  pointer = (unsigned char*) CmMessageExtend (message, sizeof(short));
  if (pointer) CvtPutShort (value, pointer);
}

/*----------------------------------------------------------------*/
short CmMessageGetShort (CmMessage message)
/*----------------------------------------------------------------*/
{
  unsigned char* pointer;
  short value = 0;
  Cvt cvt = CmCvtHandleGet (&message->cvt);

  if (!CmMessageCheckItemType (message, CmMessageItemShort)) return (0);
  pointer = (unsigned char*) CmMessageMove (message, sizeof(short));
  value = CvtGetShort (cvt, pointer);
  return (value);
}

/*----------------------------------------------------------------*/
void CmMessagePutInt (CmMessage message, int value)
/*----------------------------------------------------------------*/
{
  unsigned char* pointer;

  CmMessagePutItemType (message, CmMessageItemInt);
  pointer = (unsigned char*) CmMessageExtend (message, sizeof(int));
  if (pointer) CvtPutInt (value, pointer);
}

/*----------------------------------------------------------------*/
int CmMessageGetInt (CmMessage message)
/*----------------------------------------------------------------*/
{
  unsigned char* pointer;
  int value = 0;
  Cvt cvt = CmCvtHandleGet (&message->cvt);

  if (!CmMessageCheckItemType (message, CmMessageItemInt)) return (0);
  pointer = (unsigned char*) CmMessageMove (message, sizeof(int));
  value = CvtGetInt (cvt, pointer);
  return (value);
}

/*----------------------------------------------------------------*/
void CmMessagePutLong (CmMessage message, long value)
/*----------------------------------------------------------------*/
{
  static char string[20];
  char* pointer;

  CmMessagePutItemType (message, CmMessageItemLong);
  sprintf (string, "%ld", value);
  pointer = (char*) CmMessageExtend (message, strlen(string) + 1);
  if (pointer) strcpy (pointer, string);
}

/*----------------------------------------------------------------*/
long CmMessageGetLong (CmMessage message)
/*----------------------------------------------------------------*/
{
  char* pointer;
  int length;
  long value = 0;

  if (!CmMessageCheckItemType (message, CmMessageItemLong)) return (0);
  pointer = (char*) CmMessageMove (message, 0);
  if (pointer)
    {
      length = strlen(pointer);
      pointer = (char*) CmMessageMove (message, length + 1);
      if (pointer) sscanf (pointer, "%ld", &value);
    }
  return (value);
}

/*----------------------------------------------------------------*/
void CmMessagePutFloat (CmMessage message, float value)
/*----------------------------------------------------------------*/
{
  char* pointer;

  CmMessagePutItemType (message, CmMessageItemFloat);
  pointer = (char*) CmMessageExtend (message, sizeof(float));
  if (pointer) CvtPutFloat (value, (unsigned char*) pointer);
}

/*----------------------------------------------------------------*/
float CmMessageGetFloat (CmMessage message)
/*----------------------------------------------------------------*/
{
  char* pointer;
  float value = (float) 0.0;

  if (!CmMessageCheckItemType (message, CmMessageItemFloat)) return (value);
  pointer = (char*) CmMessageMove (message, sizeof(float));
  value = CvtGetFloat ((unsigned char*) pointer);

  return (value);
}

/*----------------------------------------------------------------*/
void CmMessagePutDouble (CmMessage message, double value)
/*----------------------------------------------------------------*/
{
  char* pointer;

  CmMessagePutItemType (message, CmMessageItemDouble);
  pointer = (char*) CmMessageExtend (message, sizeof(double));
  if (pointer) CvtPutDouble (value, (unsigned char*) pointer);
}

/*----------------------------------------------------------------*/
double CmMessageGetDouble (CmMessage message)
/*----------------------------------------------------------------*/
{
  char* pointer;
  double value = 0;

  if (!CmMessageCheckItemType (message, CmMessageItemDouble)) return (0);
  pointer = (char*) CmMessageMove (message, sizeof(double));
  value = CvtGetDouble ((unsigned char*) pointer);

  return (value);
}

/*----------------------------------------------------------------*/
void CmMessagePutArray (CmMessage message, CmMessageArrayType type,
                        int elements, void* address)
/*----------------------------------------------------------------*/
{
  char* pointer;
  int length;
  CmArray array;

  CmMessagePutItemType (message, CmMessageItemArray);
  array = (CmArray) CmMessageExtend (message, sizeof(CmArrayRec));
  if (!array) return;

  CvtPutInt ((int) type, array->type);
  CvtPutInt (elements, array->elements);

  CmMessagePutSynchro (message);

  length = elements * Size[type];
  pointer = (char*) CmMessageExtend (message, length);
  if (!pointer) return;

  memcpy (pointer, address, length);

  switch (type)
    {
    case CmMessageChar :
    case CmMessageShort :
    case CmMessageInt :
      break;
    case CmMessageFloat :
      CvtPutFloats ((unsigned char*) pointer, elements);
      break;
    case CmMessageDouble :
      CvtPutDoubles ((unsigned char*) pointer, elements);
      break;
    }

  {
    int offset;

    offset = CmMessageAlign (length);
    if (offset) CmMessageExtend (message, offset);
  }
}

/*----------------------------------------------------------------*/
void CmMessagePutExtArray (CmMessage message, CmMessageArrayType type,
                           int elements, void* address)
/*----------------------------------------------------------------*/
{
  int bytes;
  int offset;
  CmArray array;
  CmArrayRef ref;
  Cvt cvt = CmCvtHandleGet (&message->cvt);

  CmMessagePutItemType (message, CmMessageItemArray);
  array = (CmArray) CmMessageExtend (message, sizeof(CmArrayRec));
  if (!array) return;

  CvtPutInt ((int) type, array->type);
  CvtPutInt (elements, array->elements);

  CmMessagePutSynchro (message);

  bytes = elements * Size[type];

  CmMessageHeadAddToLength ((CmMessageHead) message->start, cvt, bytes);

  offset = ((char*) array) - message->data;

  ref = CmArrayRefNew (offset, address);
  CmMessageInstallArrayRef (message, ref);

  {
    int offset;

    offset = CmMessageAlign (bytes);
    if (offset)
      {
        CmMessageHeadAddToLength ((CmMessageHead) message->start, 
                                  cvt,
                                  offset);
      }
  }
}

/*----------------------------------------------------------------*/
void* CmMessageGetArray (CmMessage message, CmMessageArrayType* typeRef,
                         int* elementsRef)
/*----------------------------------------------------------------*/
{
  CmMessageArrayType type;
  int elements;
  char* pointer;
  int length;
  CmArray array;
  int size;
  Cvt cvt = CmCvtHandleGet (&message->cvt);

  if (!CmMessageCheckItemType (message, CmMessageItemArray)) return (0);
  array = (CmArray) CmMessageMove (message, sizeof(CmArrayRec));
  if (!array) return (0);

  type = (CmMessageArrayType) CvtGetInt (cvt, array->type);
  elements = CvtGetInt (cvt, array->elements);
  
  CmMessageGetSynchro (message);
  
  length = elements * Size[type];
  pointer = (char*) CmMessageMove (message, length);
  if (!pointer) return (0);

  if (typeRef) *typeRef = type;
  if (elementsRef) *elementsRef = elements;

  size = Size[type];

  switch (type)
    {
    case CmMessageChar :
      break;
    case CmMessageShort :
      {
        int elem;
        unsigned char* ptr;
        short* shortArray;

        ptr = (unsigned char*) pointer;
        shortArray = (short*) pointer;

        for (elem = 0; elem < elements; elem++)
          {
            shortArray[elem] = CvtGetShort (cvt, ptr);
            ptr += size;
          }
      }
      break;
    case CmMessageInt :
      {
        int elem;
        unsigned char* ptr;
        int* intArray;

        ptr = (unsigned char*) pointer;
        intArray = (int*) pointer;

        for (elem = 0; elem < elements; elem++)
          {
            intArray[elem] = CvtGetInt (cvt, ptr);
            ptr += size;
          }
      }
      break;
    case CmMessageFloat :
      CvtGetFloats ((unsigned char*) pointer, elements);
      break;
    case CmMessageDouble :
      CvtGetDoubles ((unsigned char*) pointer, elements);
      break;
    }

  {
    int offset;

    offset = CmMessageAlign (length);
    if (offset) CmMessageMove (message, offset);
  }

  return ((void*) pointer);
}

/*----------------------------------------------------------------*/
void CmMessagePutText (CmMessage message, char* value)
/*----------------------------------------------------------------*/
{
  CmMessagePutItemType (message, CmMessageItemText);
  
  if (value != NULL)
    CmMessagePutBytes (message, value, strlen (value)+1);
  else
    CmMessagePutBytes (message, value, 0);
}

/*----------------------------------------------------------------*/
char* CmMessageGetText (CmMessage message)
/*----------------------------------------------------------------*/
{
  int length;
  char* address;

  if (!CmMessageCheckItemType (message, CmMessageItemText)) return (0);
  address = CmMessageGetBytes (message, &length);
  if (address == NULL) return (NULL);
  if (length == 0) return (NULL);

  if (address[length-1] != 0)
    {
      DoPrint ("Cm> String not terminated !\n");
      message->status = MessageBadData;
      address[length-1] = 0;
    }
  return (address);
}

/*----------------------------------------------------------------*/
void CmMessagePutBytes (CmMessage message, char* value, int length)
/*----------------------------------------------------------------*/
{
  char* pointer;

  CmMessagePutItemType (message, CmMessageItemBytes);
  CmMessagePutInt (message, length);
  if ((value != NULL) && (length > 0))
    {
      pointer = (char*) CmMessageExtend (message, length);
      if (pointer) memcpy (pointer, value, length);
    }
}

/*----------------------------------------------------------------*/
char* CmMessageGetBytes (CmMessage message, int* returnedLength)
/*----------------------------------------------------------------*/
{
  int  length = 0;
  char* address = NULL;

  if (!CmMessageCheckItemType (message, CmMessageItemBytes)) return (0);
  
  length = CmMessageGetInt (message);

  if (length > 0)
  {
    address = (char*) CmMessageMove (message, length);
  }

  if (returnedLength) *returnedLength = length;
  return (address);
}


/* private functions ---------------------------------------------*/


/*----------------------------------------------------------------*/
static char* CmMessageHeadGetSwap (CmMessageHead head)
/*----------------------------------------------------------------*/
{
  if (!head) return (NULL);

  return ((char*) head->swap);
}

/*----------------------------------------------------------------*/
static int CmMessageHeadGetMagic (CmMessageHead head, Cvt cvt)
/*----------------------------------------------------------------*/
{
  int magic;

  if (!head) return (0);

  magic = CvtGetInt (cvt, head->magic);
  return (magic);
}

/*----------------------------------------------------------------*/
static int CmMessageHeadGetLength (CmMessageHead head, Cvt cvt)
/*----------------------------------------------------------------*/
{
  int length;

  if (!head) return (0);

  length = CvtGetInt (cvt, head->length);
  return (length);
}

/*----------------------------------------------------------------*/
static void CmMessageHeadAddToLength (CmMessageHead head, Cvt cvt, int bytes)
/*----------------------------------------------------------------*/
{
  int length;

  if (!head) return;

  length = CvtGetInt (cvt, head->length);
  length += bytes;
  CvtPutInt (length, head->length);
}

/*----------------------------------------------------------------*/
static void CmMessageHeadSetTail (CmMessageHead head, int offset)
/*----------------------------------------------------------------*/
{
  if (!head) return;

  CvtPutInt (offset, head->tail);
}

/*----------------------------------------------------------------*/
static int CmMessageHeadGetTail (CmMessageHead head, Cvt cvt)
/*----------------------------------------------------------------*/
{
  int offset;

  if (!head) return (0);

  offset = CvtGetInt (cvt, head->tail);
  return (offset);
}

/*----------------------------------------------------------------*/
static char* CmMessageHeadGetType (CmMessageHead head)
/*----------------------------------------------------------------*/
{
  if (!head) return (0);

  return (head->type);
}

/*----------------------------------------------------------------*/
static void CmMessageHeadSetType (CmMessageHead head, char* type)
/*----------------------------------------------------------------*/
{
  if (!head) return;

  strcpy (head->type, CmHandlerBuildName (type));
}

/*----------------------------------------------------------------*/
static int CmMessageTailGetMagic (CmMessageTail tail, Cvt cvt)
/*----------------------------------------------------------------*/
{
  int magic;

  if (!tail) return (0);

  magic = CvtGetInt (cvt, tail->magic);
  return (magic);
}








/*----------------------------------------------------------------*/
static void CmMessagePutItemType (CmMessage message, CmMessageItemType type)
/*----------------------------------------------------------------*/
{
  CmMessageItem* pointer;

  pointer = (CmMessageItem*) CmMessageExtend (message, sizeof(CmMessageItem));
  if (!pointer) return;

  *pointer = type;

  CmMessagePutSynchro (message);
}

/*----------------------------------------------------------------*/
static int CmMessageAlign (int offset)
/*----------------------------------------------------------------*/
{
  offset %= ALIGNMENT;
  if (offset) offset = ALIGNMENT - offset;
  return (offset);
}

/*----------------------------------------------------------------*/
static void CmMessagePutSynchro (CmMessage message)
/*----------------------------------------------------------------*/
{
  char* offsetPtr;
  char* ptr;
  int offset;

  if (message->state == CmMessageIsDead) return;
  if (message->state == CmMessageIsSending) return;
  
  offsetPtr = (char*) CmMessageExtend (message, sizeof(char));
  if (!offsetPtr) return;

  ptr = (char*) CmMessageExtend (message, 0);
  offset = CmMessageAlign (ptr - message->start);

  *offsetPtr = offset;

  if (offset)
    {
      ptr = (char*) CmMessageExtend (message, offset);
      memset (ptr, 0, offset);
    }
}

/*----------------------------------------------------------------*/
static int CmMessageGetSynchro (CmMessage message)
/*----------------------------------------------------------------*/
{
  char* offsetPtr;
  int offset;

  offsetPtr = (char*) CmMessageMove (message, sizeof(char));
  if (!offsetPtr) return (0);

  offset = *offsetPtr;
  
  if (offset)
    {
      CmMessageMove (message, offset);
    }

  return (1);
}

/*----------------------------------------------------------------*/
static int CmMessageCheckItemType (CmMessage message, CmMessageItemType type)
/*----------------------------------------------------------------*/
{
  CmMessageItemType actualType;
  static char* typeName[] = {
    (char*) "Char",
    (char*) "Short",
    (char*) "Int",
    (char*) "Long",
    (char*) "Float",
    (char*) "Double",
    (char*) "Array",
    (char*) "Text",
    (char*) "Bytes",
    (char*) "Tail"
  };

  actualType = CmMessageGetItemType (message);
  if (actualType == CmMessageItemUnknown) return (0);

  CmMessageMove (message, sizeof(CmMessageItem));
  
  if (actualType == type)
    {
      return (CmMessageGetSynchro (message));
    }
    
  DoPrint ("CmMessage>>> Warning : %s found instead of %s\n",
           typeName[actualType], typeName[type]);
  
  switch (actualType)
    {
    case CmMessageItemChar :
      CmMessageGetChar (message);
      break;
    case CmMessageItemShort :
      CmMessageGetShort (message);
      break;
    case CmMessageItemInt :
      CmMessageGetInt (message);
      break;
    case CmMessageItemLong :
      CmMessageGetLong (message);
      break;
    case CmMessageItemFloat :
      CmMessageGetFloat (message);
      break;
    case CmMessageItemDouble :
      CmMessageGetDouble (message);
      break;
    case CmMessageItemArray :
      CmMessageGetArray (message, NULL, NULL);
      break;
    case CmMessageItemText :
      CmMessageGetText (message);
      break;
    case CmMessageItemBytes :
      CmMessageGetBytes (message, NULL);
      break;
    case CmMessageItemTail :
      message->state = CmMessageIsClosed;
      break;
    case CmMessageItemUnknown :
      break;
    }

  return (0);
}

/*----------------------------------------------------------------*/
static void CmMessagePutHeader (CmMessage message)
/*----------------------------------------------------------------*/
{
  CmMessageHead head;

  if (!message) return;
  if (message->state != CmMessageIsNew) return;

  message->state = CmMessageIsNotEmpty;

  head = (CmMessageHead) CmMessageExtend (message, sizeof (CmMessageHeadRec));
  if (head)
    {
      memcpy ((char*) head->swap, CvtGetSwap (), sizeof(CvtSwap));
      CvtPutInt (MAGIC_WORD, head->magic);
      CvtPutInt (sizeof (CmMessageHeadRec), head->length);
      CvtPutInt (0, head->tail);
      memset( head->type, '\0', CmMessageTypeSize );
    }
}

/*----------------------------------------------------------------*/
static void CmMessagePutTail (CmMessage message)
/*----------------------------------------------------------------*/
{
  CmMessageTail tail;
  char* pointer;
  int offset;
  Cvt cvt = CmCvtHandleGet (&message->cvt);

  if (message->state != CmMessageIsNotEmpty) return;

  CmMessagePutItemType (message, CmMessageItemTail);

  offset = CmMessageHeadGetLength ((CmMessageHead) message->start, cvt);
  pointer = (char*) CmMessageExtend (message, sizeof (CmMessageTailRec));
  if (!pointer) return;

  CmMessageHeadSetTail ((CmMessageHead) message->start, offset);

  tail = (CmMessageTail) pointer;

  CvtPutInt (MAGIC_WORD, tail->magic);
  CmMessagePutSynchro (message);

  message->state = CmMessageIsClosed;
}

/*----------------------------------------------------------------*/
static void CmMessageSkipHeader (CmMessage message)
/*----------------------------------------------------------------*/
{
  CmMessageHead head;
  int magic;
  Cvt cvt = CmCvtHandleGet (&message->cvt);

  message->state = CmMessageIsNotEmpty;

  message->current = message->start;
  
  head = (CmMessageHead) CmMessageMove (message, sizeof (CmMessageHeadRec));
  if (!head) return;

  magic = CmMessageHeadGetMagic (head, cvt);
  if (magic != MAGIC_WORD) 
    {
      DoPrint ("Cm> Header format error.\n");
    }
  CmMessageHeadGetLength (head, cvt);
}

/*----------------------------------------------------------------*/
static CmArrayRef CmArrayRefNew (int offset, void* address)
/*----------------------------------------------------------------*/
{
  CmArrayRef array_ref;

  array_ref = (CmArrayRef) malloc (sizeof(CmArrayRefRec));
  if (array_ref == NULL)
    {
      DoPrint ("Cm> Malloc error on array ref.\n");
      return (NULL);
    }
  CmArrayRefCount++;
  
  array_ref->offset = offset;
  array_ref->address = address;
  
  return (array_ref);
}

/*----------------------------------------------------------------*/
static void CmArrayRefDelete (CmArrayRef array_ref)
/*----------------------------------------------------------------*/
{
  if (!array_ref) return;

  array_ref->offset = 0;
  array_ref->address = 0;

  free (array_ref);
  CmArrayRefCount--;
}

/*----------------------------------------------------------------*/
static void CmMessageInstallArrayRef (CmMessage message, CmArrayRef ref)
/*----------------------------------------------------------------*/
{
  if (!message) return;
  if (!ref) return;

  ListAdd (&message->arrayRefList, ref);
}

/*----------------------------------------------------------------*/
static CmMessageIterator CmMessageIteratorNew (CmMessage message,
                                               CmConnect connect,
                                               CmMessageHandler handler)
/*----------------------------------------------------------------*/
{
  CmMessageIterator iterator;

  iterator = (CmMessageIterator) malloc (sizeof(CmMessageIteratorRec));
  if (iterator == NULL)
    {
      DoPrint ("Cm> Malloc error on MessageIterator.\n");
      return (NULL);
    }

  CmMessageIteratorCount++;
  
  CmMessageHandleInstall (&iterator->message);
  CmConnectHandleInstall (&iterator->connect);
  iterator->start        = NULL;
  iterator->bytes        = 0;
  iterator->entry        = NULL;
  iterator->prefix_sent  = 0;
  iterator->finished     = 1;
  iterator->dead         = 0;
  iterator->handler      = handler;
  CmMessageHandleInstall (&iterator->message_copy);
  CmConnectHandleInstall (&iterator->connect_copy);
  
  CmMessageHandleSet (&iterator->message, message);
  CmConnectHandleSet (&iterator->connect, connect);

  CmMessageHandleSet (&iterator->message_copy, message);
  CmConnectHandleSet (&iterator->connect_copy, connect);

  CmMessageIteratorFirst (iterator);
  
  return (iterator);
}

/*----------------------------------------------------------------*/
void CmMessageIteratorDelete (CmMessageIterator iterator)
/*----------------------------------------------------------------*/
{
  if (!iterator) return;

  iterator->finished = 1;

  CmMessageHandleUninstall (&iterator->message);
  CmConnectHandleUninstall (&iterator->connect);

  CmMessageHandleUninstall (&iterator->message_copy);
  CmConnectHandleUninstall (&iterator->connect_copy);

  iterator->start = NULL;
  iterator->bytes = 0;
  iterator->entry = NULL;
  iterator->prefix_sent = 0;

  CmIteratorHandleClearAll (iterator);
  
  free (iterator);
  CmMessageIteratorCount--;
}

/*----------------------------------------------------------------*/
void CmMessageIteratorStop (CmMessageIterator iterator)
/*----------------------------------------------------------------*/
{
  CmMessage message;
  
  if (!iterator) return;

  if (CmMessageMgr.traceOn > 3) 
    {
      DoPrint ("CmMessageIteratorStop-%lx> finished=%d dead=%d\n",
               iterator,
               iterator->finished,
               iterator->dead);
    }

  if (iterator->finished) return;

  iterator->finished = 1;

  message = CmMessageHandleGet (&iterator->message);
  if (message != NULL)
    {
      message->iterators--;
      if (message->iterators <= 0)
        {
          message->iterators = 0;
          message->state = CmMessageIsClosed;
        }
    }

  iterator->start = NULL;
  iterator->bytes = 0;
  iterator->entry = NULL;
  iterator->prefix_sent = 0;
}

/*----------------------------------------------------------------*/
static void CmMessageIteratorFirst (CmMessageIterator iterator)
/*----------------------------------------------------------------*/
{
  CmMessage message;
  CmMessageState state;
  
  if (!iterator) return;

  iterator->finished = 1;

  message = CmMessageHandleGet (&iterator->message);
  
  if (message == NULL) return;
  if (CmConnectHandleGet (&iterator->connect) == NULL) return;

  state = CmMessageGetState (message);
  if ((state != CmMessageIsClosed) &&
      (state != CmMessageIsSending)) return;

  message->state = CmMessageIsSending;
  message->iterators++;

  iterator->finished = 0;

  iterator->start = message->start;
  iterator->bytes = 0;
  iterator->entry = message->arrayRefList.first;
  if (iterator->entry == NULL)
    {
      iterator->bytes = message->bytes;
    }
  else
    {
      char* pointer;
      CmArrayRef ref;
      int offset;

      ref = (CmArrayRef) iterator->entry->reference;
      pointer = message->data + ref->offset;
      pointer += sizeof(CmArrayRec);

         /* Synchro for ArrayRec */
          
      offset = *pointer;
      pointer++;
      pointer += offset;

         /*
            Sending from previous pointer up to
            the end of the array descriptor.
            */
          
      iterator->bytes = pointer - iterator->start;
      iterator->prefix_sent = 0;
    }
}

/*----------------------------------------------------------------*/
CmMessageStatus CmMessageIteratorCallHandler (CmMessageIterator iterator)
/*----------------------------------------------------------------*/
{
  CmMessageStatus result = CmMessageOk;
  CmConnect connect;
  CmMessage message;

  if (iterator == NULL) return (CmMessageOk);
  if (iterator->handler != NULL)
    {
      connect = CmConnectHandleGet (&iterator->connect);
      message = CmMessageHandleGet (&iterator->message);

      result = iterator->handler (message,
                                  CmConnectGetName (connect),
                                  CmConnectGetName (CmConnectGetServer (connect)));
    }

  CmMessageHandleClear (&iterator->message);
  CmMessageHandleClear (&iterator->message_copy);
  CmConnectHandleClear (&iterator->connect);
  CmConnectHandleClear (&iterator->connect_copy);

  return (result);
}

/*----------------------------------------------------------------*/
int CmMessageIteratorFinished (CmMessageIterator iterator)
/*----------------------------------------------------------------*/
{
  CmConnect connect;
  CmMessage message;
  CmConnect connect_copy;
  CmMessage message_copy;

  if (!iterator) return (1);

  connect = CmConnectHandleGet (&iterator->connect);
  message = CmMessageHandleGet (&iterator->message);

  if (CmMessageMgr.traceOn > 3) 
    {
      DoPrint ("CmMessageIteratorFinished-%lx> finished=%d dead=%d "
               "message=%lx connect=%lx\n", 
               iterator,
               iterator->finished,
               iterator->dead,
               message,
               connect);
    }

  if (message == NULL) iterator->finished = 1;
  if (connect == NULL) iterator->finished = 1;

  message_copy = CmMessageHandleGet (&iterator->message_copy);
  if (message_copy != message) 
    {
      if (CmMessageMgr.traceOn > 0) 
        {
          DoPrint ("CmMessageIteratorFinished> bad message copy\n");
        }

      iterator->finished = 1;
    }

  connect_copy = CmConnectHandleGet (&iterator->connect_copy);

  if (connect_copy != connect) 
    {
      if (CmMessageMgr.traceOn > 0) 
        {
          DoPrint ("CmMessageIteratorFinished> bad connect copy\n");
        }

      iterator->finished = 1;
    }
  if (iterator->dead) iterator->finished = 1;
  
  return (iterator->finished);
}

/*----------------------------------------------------------------*/
void CmMessageIteratorForth (CmMessageIterator iterator)
/*----------------------------------------------------------------*/
{
  CmMessage message;
  CmMessageArrayType type;
  CmArrayRef ref;
  CmArray array;
  char* pointer;
  int elements;
  int offset;
  int bytes;
  CmConnect connect;
  Cvt cvt;
  
  if (CmMessageIteratorFinished (iterator)) return;

  if (CmMessageMgr.traceOn > 3) 
    {
      DoPrint ("CmMessageIteratorForth-%lx> finished=%d dead=%d\n",
               iterator,
               iterator->finished,
               iterator->dead);
    }

  if (iterator->bytes == 0)
    {
      CmMessageIteratorStop (iterator);
      return;
    }

    /* We always have something to send. */

  connect = CmConnectHandleGet (&iterator->connect);
  if (connect != NULL)
    {
      bytes = CmConnectSend (connect, iterator->start, iterator->bytes);
      if (bytes < 0)
        {
          if (CmConnectDebug > 0)
            {
              DoPrint ("CmMessageIteratorForth> Connect %s dead\n",
                       CmConnectGetName (connect));
            }
          if( bytes == -2 )
            {
              DoPrint("Cm> CmMessageIteratorForth> SIGPIPE received or"
		      " errno set EPIPE, close %s CmConnect \n",
		      CmConnectGetName (connect) );
            } 
          else
	    {
              DoPrint ("CmMessageIteratorForth> Connect %s dead - errno %d\n",
                       CmConnectGetName (connect), errno );
	    }
          iterator->dead     = 1;
          CmMessageIteratorStop (iterator);
          CmConnectKill (connect);
          return;
        }

      if( (bytes == 0) && (errno == EAGAIN) )
        DoPrint("Cm> CmMessageIteratorForth> Connect %s temporaly "
		"unavailable\n", CmConnectGetName (connect) );
    }
  else
    {
      if (CmConnectDebug > 0)
        {
          DoPrint ("CmMessageIteratorForth> No more connect\n");
        }
          
      iterator->dead     = 1;
      CmMessageIteratorStop (iterator);
      return;
    }
        
  iterator->bytes -= bytes;

  if (iterator->bytes > 0)
    {
      iterator->start += bytes;
      return;
    }

    /*
      The packet is entirely sent. Thus we have to setup the
      next packet.
      */
      
  if (iterator->entry == NULL)
    {
        /* No more packet. */
      CmMessageIteratorStop (iterator);
      return;
    }

  ref = (CmArrayRef) iterator->entry->reference;
  if (ref == NULL)
    {
        /* No more packet. */
      CmMessageIteratorStop (iterator);
      return;
    }

  message = CmMessageHandleGet (&iterator->message);
  if (message == NULL)
    {
      if (CmConnectDebug > 0)
        {
          DoPrint ("CmMessageIteratorForth> No more message\n");
        }
          
      iterator->dead     = 1;
      CmMessageIteratorStop (iterator);
      return;
    }

  pointer = message->data + ref->offset;
  if (pointer == NULL)
    {
        /* No more packet. */
      CmMessageIteratorStop (iterator);
      return;
    }

  array = (CmArray) pointer;
  pointer += sizeof(CmArrayRec);

    /* Synchro for ArrayRec */
          
  offset = *pointer;
  pointer++;
  pointer += offset;

  cvt = CmCvtHandleGet (&message->cvt);

  type = (CmMessageArrayType) CvtGetInt (cvt, array->type);
  elements = CvtGetInt (cvt, array->elements);
  bytes = elements * Size[type];

  if (!iterator->prefix_sent)
    {
        /*
          We have just sent the packet that stands before the next
          external array.

          When there is no next external array the packet we have
          just sent was the end of the message buffer.

          Here we compute the parameters for the next packet
          that will be the external buffer.
          */

      iterator->prefix_sent = 1;
    
      switch (type)
        {
          case CmMessageFloat :
            CvtPutFloats ((unsigned char*) ref->address, elements);
            break;
          case CmMessageDouble :
            CvtPutDoubles ((unsigned char*) ref->address, elements);
            break;
          default :
            break;
        }
      
      if (bytes > 0)
        {
          iterator->start = (char*) ref->address;
          iterator->bytes = bytes;
        
          return;
        }
        
        /*
          The external buffer has a null length. Thus we consider
          it is finished.

          We act as if it would be sent and we move to the next
          packet.
          */
    }
  
  switch (type)
    {
      case CmMessageFloat :
        CvtPutFloats ((unsigned char*) ref->address, elements);
        break;
      case CmMessageDouble :
        CvtPutDoubles ((unsigned char*) ref->address, elements);
        break;
      default :
        break;
    }
  
    /*
      The previous external array is finished.
      We must move to the next message packet which is actually
      the portion of the message buffer between the previous
      external array and the next external array or the end of
      the message buffer.
      */
  
  {
    int offset;

    offset = CmMessageAlign (bytes);
    if (offset) pointer -= offset;
  }

  iterator->start = pointer;
  if (iterator->entry != NULL)
    {
      iterator->entry = iterator->entry->next;
    }

  if (iterator->entry == NULL)
    {
        /* No more external array. */

      bytes = pointer - message->data;
      iterator->bytes = message->bytes - bytes;
    }
  else
    {
      ref = (CmArrayRef) iterator->entry->reference;
      pointer = message->data + ref->offset;
      array = (CmArray) pointer;
      pointer += sizeof(CmArrayRec);
      
        /* Synchro for ArrayRec */
      
      offset = *pointer;
      pointer++;
      pointer += offset;
      
        /*
          Sending from previous pointer up to
          the end of the array descriptor.
          */
      
      bytes = pointer - iterator->start;
      
      iterator->bytes = bytes;
      iterator->prefix_sent = 0;
    }
}



/*----------------------------------------------------------------*/
/*                                                                */
/*   Message reception.                                           */
/*                                                                */
/*----------------------------------------------------------------*/

/*----------------------------------------------------------------*/
CmConnectStatus CmMessageCallHandler (CmMessage message)
/*----------------------------------------------------------------*/
{
  char* type;
  CmHandler handler;
  char* name;
  CmMessageStatus message_status;
  char* serverName;
  CmConnect server;
  CmConnect connect;

  if (message == NULL) return (CmConnectStatusNoData);

  connect = CmConnectHandleGet (&message->connect);
  server = CmConnectGetServer (connect);
  serverName = CmConnectGetName (server);

  if (CmMessageMgr.filter != NULL)
    {
      if (!RegExpMatch (CmMessageMgr.filter, serverName))
        {
          return (CmConnectStatusData);
        }
    }

  type = CmMessageGetType (message);
  handler = CmHandlerFind (type);

  name = CmConnectGetName (connect);

  message_status = CmMessageOk;
  
  CmMessageSkipHeader (message);
  if (CmMessageMgr.traceOn > 0) CmMessageDump (message, name, 1);

  if (handler)
    {
      message_status = handler->address (message, name, serverName);
    }
  else
    {
      if (CmMessageMgr.defaultHandler)
        {
          message_status = CmMessageMgr.defaultHandler (message,
                                                        name, serverName);
        }
    }
      
  CmCvtHandleClear (&message->cvt);

  if (message_status == CmMessageBreak) return (CmConnectStatusAbort);

  return (CmConnectStatusNoData);
}

/*----------------------------------------------------------------*/
static CmConnectStatus CmMessageRearmHandler (CmConnect connect)
/*----------------------------------------------------------------*/
{
  if (CmConnectTopMessage (connect) != NULL)
    {
      return (CmConnectStatusData);
    }
  else
    {
      return (CmConnectStatusNoData);
    }
}

/*----------------------------------------------------------------*/
static CmConnectStatus CmMessageReceiveHandler (CmConnect connect)
/*----------------------------------------------------------------*/
/*  This is THE reception handler declared to any Connect object. */
/* it will be called whenever a connection has received           */
/* something in its input channel.                                */
/*                                                                */
/*  The role of this function is to accumulate data into the      */
/* message buffer until the message frame is completed. This      */
/* may require several calls to this handler when the message     */
/* is transmitted in several packets (this may be due to the      */
/* way it has been sent or because of TCPIP buffering scheme)     */
/*                                                                */
/*----------------------------------------------------------------*/
{
  CmConnectStatus status;
  CmMessage message;

/*
  int number;
  static int n = 0;
  */

  if (connect == NULL) return (CmConnectStatusNoData);
  
  status = CmConnectCheckInputData (connect);
  if (status == CmConnectStatusData)
    {
        /*
          We ask the connect for a Message object. This object is created
          if it does not exist.
          */
      
      message = CmConnectGetMessage (connect);

      CmConnectHandleSet (&message->connect, connect);
      
        /*
          This is where data are being accumulated into the
          message buffer. The result of it may be that
          the message is not yet completed.
          */
      
        /*
          n++;
          number = n;
          
          Tab++;
          printf ("receive %d\n", number);
          */
      
      switch (CmMessageDoReceive (message, connect))
        {
          case MessageError :
          case MessageBadConnect :
          case MessageBadHead :
          case MessageBadLength :
          case MessageBadTail :
            DoPrint ("CmMessageReceiveHandler> Error on connect %s\n",
                     CmConnectGetName (connect));

            CmConnectKill (connect);
            status = CmConnectStatusNoData;
            break;
          case MessageIncompleteData :
            status = CmConnectStatusNoData;
            break;
          case MessageOk :
          default :
            CmConnectPushMessage (connect);
            status = CmConnectStatusNoData;
            break;
        }
    }

  message = CmConnectPopMessage (connect);

  if (message != NULL)
    {
      status = CmMessageCallHandler (message);
      CmMessageDelete (message);
    }

  return (status);
}

/*----------------------------------------------------------------*/
static Status CmMessageDoReceive (CmMessage message, CmConnect connect)
/*----------------------------------------------------------------*/
/*  Here we accumulate data into the message buffer. The result   */
/* of one insertion may be :                                      */
/*                                                                */
/*   o The message frame is completed.                            */
/*   o The message frame is not yet complete (we expect some      */
/*      more data to come later on)                               */
/*   o There is an error in the framing protocol.                 */
/*                                                                */
/*  This is called from the reception handler.                    */
/*                                                                */
/*----------------------------------------------------------------*/
/*----------------------------------------------------------------*/
/*  This function evaluates the current state of the message      */
/* object.                                                        */
/*                                                                */
/*  Typical situation may be detected:                            */
/*                                                                */
/*     o The buffer is not yet created.                           */
/*                MessageNoBuffer                                 */
/*                                                                */
/*     o The buffer is too small to receive even one header       */
/*                MessageUnsufficientBuffer                       */
/*                                                                */
/*     o The header got wrong format                              */
/*                MessageBadHead                                  */
/*                MessageBadLength                                */
/*                MessageBadTail                                  */
/*                                                                */
/*     o The buffer is too small to receive the full message      */
/*                MessageUnsufficientBuffer                       */
/*                                                                */
/*     o The message is not yet complete.                         */
/*                MessageIncompleteData                           */
/*                                                                */
/*     o The message is complete                                  */
/*                MessageOk                                       */
/*                                                                */
/*  This function is only used when receiving a message.          */
/*                                                                */
/*                                                                */
/*----------------------------------------------------------------*/
{
  int length;
  int available;
  int bytes;
  int offset;
  CmMessageTail tail;
  Cvt cvt;
  char *server_name;

  if (!message) 
    {
      DoPrint ("CmMessageDoReceive> No message\n");
      return (MessageError);
    }

  if (message->state == CmMessageIsDead) 
    {
      DoPrint ("CmMessageDoReceive> Message Dead\n");
      return (MessageError);
    }

  if (!connect)
    {
      DoPrint ("Cm> No connection to read from.\n");
      return (MessageBadConnect);
    }
  server_name = CmConnectGetName( connect );

  if (message->data == NULL)
    {
        /*
          The buffer is not yet created. 
          We just need at least room for a Head
          */
      if (!CmMessageExtendForReception (message, SIZEOF_HEAD))
        {
          DoPrint ("CmMessageDoReceive> cannot extend for header - from %s\n",
		   server_name );
          return (MessageNoBuffer);
        }
    }

  available = message->allocated;
  if (available < SIZEOF_HEAD)
    {
        /*
          The physical buffer is too small even to receive one header.
          */
      if (!CmMessageExtendForReception (message, SIZEOF_HEAD))
        {
          DoPrint ("CmMessageDoReceive> cannot extend for header - from %s\n",
		   server_name );
          return (MessageUnsufficientBuffer);
        }

      available = message->allocated;
    }

    /*
      'bytes' counts the data accumulated by previous ConnectReceive 
      operations.
      it may be less than 'allocated'
      */
  
  bytes = message->bytes;
  if (bytes < SIZEOF_HEAD)
    {
        /*
          The received data does not contain even one header.
          */
      if (!CmMessageReceiveData (message, connect, SIZEOF_HEAD - bytes))
        {
          return (MessageIncompleteData);
        }
      bytes = message->bytes;
    }

  if (bytes < SIZEOF_HEAD)
    {
        /*
          The received data does not contain even one header.
          */
      return (MessageIncompleteData);
    }

  cvt = CmCvtHandleGet (&message->cvt);

  if (cvt == NULL)
    {
      char* swap;
      int size;

      swap = CmMessageHeadGetSwap ((CmMessageHead) message->start);
      size = strlen(swap)+1;

      if( size > sizeof(CvtSwap) ) {
	DoPrint("CmMessageDoReceive> bad swap header - swap size %d > %d "
		" - from %s\n", size, sizeof(CvtSwap), server_name );
	return (MessageBadHead);
      }
    
      CmCvtHandleSet (&message->cvt, CvtNew (swap));
      cvt = CmCvtHandleGet (&message->cvt);
    }

  if (CmMessageHeadGetMagic ((CmMessageHead) message->start, 
                             cvt) != MAGIC_WORD)
    {
        /*
          Wrong format in the header.
          */
      DoPrint ("CmMessageDoReceive> bad header (no magic) - from %s\n",
	       server_name);
      return (MessageBadHead);
    }

  length = CmMessageHeadGetLength ((CmMessageHead) message->start, 
                                   cvt);
  if (length < SIZEOF_PROTOCOL)
    {
        /*
          Wrong format in the header.
          */
      DoPrint ("CmMessageDoReceive> bad length in header - from %s\n", 
	       server_name );
      return (MessageBadLength);
    }

  if (available < length)
    {
        /*
          The buffer is too small to receive the entire message.
          */
      if (!CmMessageExtendForReception (message, length - message->bytes))
        {
          DoPrint ("CmMessageDoReceive> cannot extend for message "
		   "- from %s\n", server_name);
          return (MessageUnsufficientBuffer);
        }
      available = message->allocated;
    }

  if (bytes < length)
    {
        /*
          The received data is shorter than the complete message.
          */
      if (!CmMessageReceiveData (message, connect, length - bytes))
        {
          return (MessageIncompleteData);
        }

      bytes = message->bytes;

      if (bytes < length) 
        {
          return (MessageIncompleteData);
        }
    }

    /*
      It seems that we have received the entire message.
      */

  offset = CmMessageHeadGetTail ((CmMessageHead) message->start, 
                                 cvt);
  tail = (CmMessageTail) (message->data + offset);

  if (CmMessageTailGetMagic (tail, cvt) != MAGIC_WORD)
    {
        /*
          The message tail has a wrong format.
          */
      DoPrint ("CmMessageDoReceive> bad tail - from %s\n", server_name);
      return (MessageBadTail);
    }

    /*
      The message is complete.
      */
  
  return (MessageOk);
}

/*----------------------------------------------------------------*/
static int CmMessageExtendForReception (CmMessage message, int needed)
/*----------------------------------------------------------------*/
/*                                                                */
/*   The situation of the message buffer is :                     */
/*       o 'allocated' represents the total number of bytes       */
/*                     available for the message.                 */
/*       o 'bytes'     represents the part of it that contains    */
/*                     previously received data (with             */
/*                     CmConnectReceive)                          */
/*                                                                */
/*   The role of this function is to extend 'allocated' by at     */
/*  least one FRAME_SIZE quantum. The result should be able       */
/*  to store at least one header and one tail.                    */
/*                                                                */
/*   Extension and available space are computed from the          */
/*  'data' index.                                                 */
/*                                                                */
/*----------------------------------------------------------------*/
{
  char* new_data;
  
  if (needed < FRAME_SIZE) needed = FRAME_SIZE;

  if (message->data != NULL)
    {
        /*
          The buffer will be physically extended.
          */

      int start_position;
      int current_position;

      start_position   = message->start - message->data;
      current_position = message->current - message->data;
              
      new_data = (char*) realloc (message->data, message->allocated + needed);
              
      if (new_data == NULL) return (0);

      message->allocated += needed;
      message->data    = new_data;
      message->start   = message->data + start_position;
      message->current = message->data + current_position;
    }
  else
    {
      message->start   = NULL;
      message->current = NULL;
      message->bytes   = 0;

      new_data = (char*) malloc (needed);
          
      if (new_data == NULL) return (0);

      CmMessageDataCount++;
      
      message->allocated = needed;
      message->data    = new_data;
      message->start   = message->data;
      message->current = message->data;
    }

  return (1);
}

/*----------------------------------------------------------------*/
static int CmMessageReceiveData (CmMessage message, CmConnect connect, 
                                 int needed)
/*----------------------------------------------------------------*/
/*  The buffer is supposed to be able to receive at least         */
/* 'needed'   worth of data from the input channel.               */
/*----------------------------------------------------------------*/
{
  char* pointer;
  int bytes;
  
  pointer   = message->data + message->bytes;
  
  bytes = CmConnectReceive (connect, pointer, needed);

  message->bytes += bytes;
  
  return (bytes);
}

/*----------------------------------------------------------------*/
static void* CmMessageMove (CmMessage message, int bytes)
/*----------------------------------------------------------------*/
{
  int  currentPosition;
  void* pointer;

  if (!message) return (NULL);
  if (message->state != CmMessageIsNotEmpty) return (0);

  currentPosition = message->current - message->data;
  if (currentPosition + bytes > message->bytes)
  {
    DoPrint ("Cm> Trying to read beyond data space.\n");
    message->status = MessageOverlay;
    message->state = CmMessageIsNotReady;
    return (NULL);
  }

  pointer = message->current;
  message->current += bytes;
  return (pointer);
}






