/*------------------------------------------------  FdCompress.c by B. Mours */
#include "FdIO.h"
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <pthread.h>
#include <time.h>
#include "FrameL.h"
#include <dirent.h>
#include <sys/resource.h>
#include <CfgVirgo.h>

/*----------------------------------------start with private definitions---*/

typedef struct FdCompress FdCompress;

struct FdCompress          
{
  int      compLevel;      /* requested compression level */
  int      gzipLevel;      /* gzip compression level, set to 0 by default */
  int      iThread;        /* used to count the threads */
  int      nThread;        /* requested number of threads */
  int      nCompTot;       /* total number of vector to compress */
  FrVect  *vect;           /* vector to be compressed */
  int     *nComp;          /* list of number of vector compressed 
			      size of the list = number of thread */
  pthread_t *tidList;      /* list of thread id */
  pthread_mutex_t lock;    /* mutex used to pass the condition */
  pthread_cond_t condFeed; /* to request one more vector */
  pthread_cond_t condComp; /* to request the compression of a vector */
  FdAction*  action;       /* action object which hold this object */
};

void FdCompressFeedVect(FdCompress *fdc, FrVect *vect);
void FdCompressProc    (FdCompress *fdc, FrameH *frame);

/*---------------------------------------------------------------------------*/
void FdCompressFeedTable(FdCompress *fdc, FrTable *table)
/*---------------------------------------------------------------------------*/
{
  if(table == NULL) return;

  if(table->next != NULL) FdCompressFeedTable(fdc, table->next);

  FdCompressFeedVect(fdc, table->column);

  return;
}
/*---------------------------------------------------------------------------*/
void FdCompressFeedVect(FdCompress *fdc, FrVect *vect)
/*---------------------------------------------------------------------------*/
/* This function passes a vector to a compression thread                     */
/*---------------------------------------------------------------------------*/
{
  if(vect == NULL) return;

  fdc->nCompTot++;
 
  /*------- wait to be called to provide a new vector to be compressed---*/
  pthread_mutex_lock (&fdc->lock);
  if(fdc->vect != NULL) {
    pthread_cond_wait  (&fdc->condFeed, &fdc->lock);}
  fdc->vect = vect;

  /*------------------------------------------call a compression thread---*/
  pthread_cond_signal  (&fdc->condComp);
  pthread_mutex_unlock (&fdc->lock);

  return;
}
/*---------------------------------------------------------------------------*/
void FdCompressThread(FdCompress *fdc)
/*---------------------------------------------------------------------------*/
{
  FrVect *vect;
  int tmo, myId;
                                     
  myId = fdc->iThread;
  fdc->iThread++;
  CfgMsgAddInfo("Start FdCompress thread id=%d", myId);

  /*-------------------------------- Set the thread as Asynchronous thread---*/
  pthread_setcancelstate( PTHREAD_CANCEL_ENABLE, &tmo );
  pthread_setcanceltype ( PTHREAD_CANCEL_ASYNCHRONOUS, &tmo );

  /*----------------- compress all vector which are passed to this process---*/
  while (1) {

    /*---------------------------------------------start critical section---*/
    pthread_mutex_lock (&fdc->lock);

    /*-----------------request a new vector and wait until it is received---*/
    while(fdc->vect == NULL) {
      pthread_cond_signal(&fdc->condFeed);
      pthread_cond_wait  (&fdc->condComp, &fdc->lock);}

    vect = fdc->vect; 
    fdc->vect = NULL;

    /*---------------------------request a new vector for the next thread---*/ 
    pthread_cond_signal(&fdc->condFeed);

    /*-----------------------------------------------end critical region---*/
    pthread_mutex_unlock (&fdc->lock);

    /*-----------------------------------------do the compression itself---*/
    FrVectCompress(vect, fdc->compLevel, fdc->gzipLevel);

    fdc->nComp[myId]++; 
  }

  return;
}
/*---------------------------------------------------------------------------*/
int FdCompressNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdCompress *fdc;
  int i;

  /*------------create the object and add it at the end of the linked list---*/
  fdc = (FdCompress*) calloc(1,sizeof(FdCompress));
  if(fdc == NULL) CfgMsgAddFatal("malloc FdCompress failed");

  fdc->action = FdActionNew((FdAction**) arg, 
			       (void*) fdc, FdCompressProc, "Compress");

  /*------------------------------------------------------------ fill info---*/
  fdc->compLevel = CfgParseGetNextDec(data);
  fdc->nThread   = CfgParseGetNextDec(data);

  CfgMsgAddInfo("Compress the frame to %d with %d thread(s)",
		fdc->compLevel, fdc->nThread);

  if(fdc->nThread <= 0)  return(CFG_OK);

  /*----------------threads have been requested create/fill local variable---*/
  fdc->nComp = (int *) calloc(fdc->nThread, sizeof(int));
  if(fdc->nComp == NULL) CfgMsgAddFatal("malloc FdCompress failed");

  fdc->tidList = (pthread_t *) calloc(fdc->nThread, sizeof(pthread_t));
  if(fdc->tidList == NULL) CfgMsgAddFatal("malloc FdCompress failed");

  /*--------------------------------------------------------create threads---*/
  pthread_mutex_init(&fdc->lock,    NULL);
  pthread_cond_init(&fdc->condFeed, NULL);
  pthread_cond_init(&fdc->condComp, NULL);

  for(i = 0; i < fdc->nThread; i++) {
    if(pthread_create((pthread_t *)&(fdc->tidList[i]), NULL,
		      (void *(*)(void *))FdCompressThread, (void *)fdc) == -1)
      CfgMsgAddFatal("FdCompressNew: pthread_create failed");

    /*--------------------wait until the thread is started before going on---*/
    while(1) {
      usleep(10);
      if(fdc->iThread == i+1) break;}

    /*-----------to wait for the end of the thread before quiting the main---*/
  /*  pthread_join(fdc->tidList[i], NULL);***************/
  }

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdCompressParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
 
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_COMPRESSION",
			 FdCompressNew, (void *) &(fdIO->actionsOut), 2,
			 CfgDec, CfgDec);

  return;
}
/*---------------------------------------------------------------------------*/
void FdCompressProcOne(FdCompress *fdc, FrameH *frame)
/*---------------------------------------------------------------------------*/
{
  FrDetector *det;
  FrAdcData  *adc;
  FrProcData *proc;
  FrSimData  *sim;
  FrSimEvent *simEvt;
  FrSerData  *ser;
  FrEvent    *evt;
  FrSummary  *sum;
  int iThread, nCompTh;

  /*------------------------------ first deal with the case without thread---*/
  if(fdc->nThread <= 0){
    FrameCompress(frame, fdc->compLevel, fdc->gzipLevel);
    return;}

  /*-------------------------------------------------------------------------*/
  /* now with thread, this function scan the frame with the same logic as the 
     FrameCompress function to search for the vectors to be compressed       */
  /*-------------------------------------------------------------------------*/

  /*-------------------first reset the nummber of vectro to be compressed---*/
  fdc->nCompTot = 0;
  for(iThread = 0; iThread < fdc->nThread; iThread++) {
    fdc->nComp[iThread] = 0;}

  /*---------------------------compress the frame starting from the header---*/
  FdCompressFeedVect (fdc, frame->type);
  FdCompressFeedVect (fdc, frame->user);
  FdCompressFeedVect (fdc, frame->auxData);
  FdCompressFeedTable(fdc, frame->auxTable);

  for(det = frame->detectProc; det != NULL; det = det->next) {
    FdCompressFeedVect (fdc, det->aux);
    FdCompressFeedTable(fdc, det->table);}

  for(det = frame->detectSim; det != NULL; det = det->next) {
    FdCompressFeedVect (fdc, det->aux);
    FdCompressFeedTable(fdc, det->table);}

  /*--------------------------------------------------compress the raw data--*/
  if(frame->rawData != NULL) {
    for(adc = frame->rawData->firstAdc; adc != NULL; adc = adc->next) {
      FdCompressFeedVect (fdc, adc->data);
      FdCompressFeedVect (fdc, adc->aux);}
    for(ser = frame->rawData->firstSer; ser != NULL; ser = ser->next) {
      FdCompressFeedVect (fdc, ser->serial);
      FdCompressFeedTable(fdc, ser->table);}
    FdCompressFeedTable(fdc, frame->rawData->firstTable);
    FdCompressFeedVect (fdc, frame->rawData->more);}

  /*-----------------------------------------------compress the other data---*/
  for(proc = frame->procData; proc != NULL; proc = proc->next) {
    FdCompressFeedVect (fdc, proc->data);
    FdCompressFeedVect (fdc, proc->aux);
    FdCompressFeedTable(fdc, proc->table);}

  for(sim = frame->simData; sim != NULL; sim = sim->next) {
    FdCompressFeedVect (fdc, sim->data);
    FdCompressFeedVect (fdc, sim->input);
    FdCompressFeedTable(fdc, sim->table);}  

  for(simEvt = frame->simEvent; simEvt != NULL; simEvt = simEvt->next) {
    FdCompressFeedVect (fdc, simEvt->data);
    FdCompressFeedTable(fdc, simEvt->table);}

  for(evt = frame->event; evt != NULL; evt = evt->next) {
    FdCompressFeedVect (fdc, evt->data);
    FdCompressFeedTable(fdc, evt->table);}

  for(sum = frame->summaryData; sum != NULL; sum = sum->next) {
    FdCompressFeedVect (fdc, sum->moments);
    FdCompressFeedTable(fdc, sum->table);}

  /*---------wait until all thread are done with their vector compression---*/
  while(1) {
    nCompTh = 0;
    for(iThread = 0; iThread < fdc->nThread; iThread++) {
       nCompTh += fdc->nComp[iThread];}
    if(nCompTh == fdc->nCompTot) break;
    usleep(1);}

  return;
}
/*---------------------------------------------------------------------------*/
void FdCompressProc(FdCompress *fdc, FrameH *frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdCompressProcOne(fdc, frame);

  /*-------------------------------call the next action in the linked list---*/
  next = fdc->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
