#define _GNU_SOURCE

#include "FdIO.h"

/*-------------------------------------------------------private functions---*/
typedef struct FdButFilter FdButFilter;
typedef struct FdFFTFilter FdFFTFilter;

struct FdButFilter{
  char*        tag;            /* input channel definition    */
  double       freqDecimation; /* new channel frequency after decimation */
  double       freqCut;        /* butterworth frequency cut off */
  int          order;          /* order of the buttherworth filter*/
  int          mode;           /* 0x1 keep input channel; 0x2 store as double*/
  FRBOOL       keepInput;      /* if FR_YES, duplicate the input channel */
  FRBOOL       storeAs64bits;  /* if FR_YES store data as double */
  double       frameDt;        /* frame length */
#ifdef LIN_FILTER
  FrvLinFilt*  filter;         /* this is the link list on filter objects*/
#else
  FrvFilterB*  filter;         /* this is the link list on filter objects*/
#endif
  FrCList   *list;
  char      *names;       /* names of the channel that are reset */
  char      *namesP;      /* current value of the namesP pointer */
  int        nResetLast;  /* last number of filters reset */
  int        nReset;      /* number of filter reseted */
  int        nDupInput;   /* number of output channel with multiple input*/
  FdAction*  action;      /* action object which hold this recast object */
};

struct FdFFTFilter{
  char*        tag;         /* input channel definition    */
  double       freqMin;     /* low frequency cutoff */
  double       freqMax;     /* high frequency cutoff */
  double       delay;       /* delay*/
  char*        tfFileName;  /* file name if a TF is provided */
  FrvFDFilter* filter;      /* this is the link list on filter objects*/
  FrameH*      lastFrame;   /* last frame on which the filtering was done*/
  FdAction*  action;      /* action object which hold this recast object */
};  

/*---------------------------------------------------------------------------*/
char* FdButNewName(char* oldName, double freq)
/*---------------------------------------------------------------------------*/
/* This function add the frequency at the end of the name.                   */
/*---------------------------------------------------------------------------*/
{
  int size;
  char *name;

  size = strlen(oldName);
  name = malloc(size+30);
  if(name == NULL) 
    CfgMsgAddFatal("FdButNewName malloc failed (%d)", size);

  memcpy(name, oldName, size);
  sprintf(name+size,"_%gHz", freq);

  return(name);
}
/*---------------------------------------------------------------------------*/
char* FdButNewNameHP(char* oldName, double freq)
/*---------------------------------------------------------------------------*/
{
  int size;
  char *name;

  size = strlen(oldName);

  /*-------------------------------------------------allocate enough space---*/
  name = malloc(size+30);
  if(name == NULL) 
    CfgMsgAddFatal("FdButNewName malloc failed (%d)", size);

  memcpy(name, oldName, size);
  sprintf(name+size,"_%gHzHP", freq);

  return(name);
}
/*---------------------------------------------------------------------------*/
char* FdButNewComment(char *name, char *oldComment, double freqCut, int order)
/*---------------------------------------------------------------------------*/
{
  int size, sizeOld;
  char *comment;

  if(oldComment != NULL) sizeOld = strlen(oldComment) +2;
  else                   sizeOld = 0;
  size = sizeOld + 128 + strlen(name);

  comment = malloc(size);
  if(comment == NULL)
    CfgMsgAddFatal("FdButNewComment malloc failed (%d)", size);

  if(oldComment != NULL) sprintf(comment,"%s; ",oldComment);

  sprintf(comment+sizeOld,"%s filtered by FdButFilted fCut=%gHz order=%d", 
          name, freqCut, order);

  return(comment);
}
/*---------------------------------------------------------------------------*/
FrCList *FrCListBldList(void *root, long offsetName, long offsetNext)
/*---------------------------------------------------------------------------*/
/* This function build an alphabetic sorted list for a linked list           */
/*---------------------------------------------------------------------------*/
{
  void *obj, **field, **next;
  FrCList *list;
  long len, maxL, pSize;
  void *current;
  char **name;

  if (root == NULL) return(NULL);

  list = (FrCList *) malloc(sizeof(FrCList));
  if(list == NULL) return (NULL);

  /*----------------------------------------------compute the table size ----*/
  list->nChannels = 0;
  maxL = 0;
  for (obj = root; obj != NULL; obj = *next) {
    (list->nChannels)++;
    name = (char**)((long)obj+offsetName);
    next = (void**)((long)obj+offsetNext);
    len = strlen(*name)+1;
    if(len > maxL) maxL = len;}

  pSize = sizeof(void *);
  list->nameSize = pSize*((maxL+pSize-1)/pSize);    /*-- align the address---*/
  list->rowSize  = pSize + list->nameSize;

  /*------------------------------------------------------ Allocate space ---*/
  list->table = calloc(1, list->nChannels * list->rowSize);
  if(list->table == NULL) return(NULL);

  /*------------------------------------------------------ Fill the table---*/
  current = list->table;
  for (obj = root; obj != NULL; obj = *next) {
    name = (char**) ((long)obj+offsetName);
    next = (void**) ((long)obj+offsetNext);
    memcpy(current, *name, strlen(*name)+1);
    field = (void **)((long)current + list->nameSize);
    *field = obj;
    current =  (void *)((long)current + list->rowSize);}

  /*---------------------------------------------------- sort the table ----*/
  qsort(list->table, list->nChannels, list->rowSize, 
	(int (*) (const void*, const void*)) strcmp);

  return(list);
}
/*---------------------------------------------------------------------------*/
FrVect* FdButFilterProcessVect(FdButFilter *f,
			       FrCList *list,
			       FrVect* vect,
			       int nGroup,
                               char *newName)
/*---------------------------------------------------------------------------*/
{
  char *data;
#ifdef LIN_FILTER
  FrvLinFilt *filter;
#else
  FrvFilterB *filter;
#endif
  FrVect *next;
  int i, j, off;

  /*------------------------------- filter already exist? If not create it---*/
#ifdef LIN_FILTER
  filter = (FrvLinFilt*) FrCListFind(list, newName);
#else
  filter = (FrvFilterB*) FrCListFind(list, newName);
#endif
  if(filter != NULL) {
    /*---if the filter already exist; check that sampling rate is the same---*/
    if(filter->output->dx[0] != vect->dx[0]) {
      f->action->state = CfgServerActive;
      if(strcmp(filter->output->name, vect->name) == 0) { 
        CfgMsgAddWarning("ButFilter: at %.0f sampling rate changed for %s from"
                         " %g to %g; reset filter for %s", 
                         vect->GTime, vect->name, 
                         filter->output->dx[0], vect->dx[0], filter->name);
        FrvFilterButReset(filter);}
      else if(filter->output->GTime + 2*f->frameDt < vect->GTime) {
        CfgMsgAddWarning("ButFilter: at %.0f sampling rate/input changed from"
                         " %g to %g; new input is %s; reset filter for %s", 
                         vect->GTime, filter->output->dx[0], 
                         vect->dx[0], vect->name, filter->name);
        FrvFilterButReset(filter);}
      else {
        if(filter->output->dx[0] < vect->dx[0]) {
	  if(filter->errorPrinted == FR_NO) {
            CfgMsgAddError("ButFilter: at %.0f more that one input vector for"
			   " %s; keep faster input channel:%s reject:%s",
			   vect->GTime, newName, filter->output->name, 
			   vect->name);
	    filter->errorPrinted = FR_YES;}
          filter->lastError = vect->GTime;
	  f->nDupInput++;
          return(NULL);}
        else if(filter->output->GTime == vect->GTime) {
          CfgMsgAddError("ButFilter: at %.0f more that one input vector for %s;"
			 " switch to faster input channel:%s reject:%s",
                         vect->GTime, newName, vect->name, filter->output->name);
          FrvFilterButReset(filter);
          filter->errorPrinted = FR_YES;}}}

    /*---------------------- if we are back to a normal situation, report it---*/
    else {
      if(filter->errorPrinted == FR_YES) {
	if(filter->lastError + 1.5*f->frameDt < vect->GTime) {
	  CfgMsgAddInfo("ButFilter: at %.0f only one input vector for %s",
			vect->GTime, newName);
	  filter->errorPrinted = FR_NO;}}}}

  /*-------------------------------------------create a new filter if needed---*/
  else {
    CfgMsgAddInfo(" Add filter But.%d order fCut=%gHz nGroup=%d fOut=%gHz "
		  "for channel %s", 
		  f->order, f->freqCut, nGroup, 
		  1./(vect->dx[0]*nGroup), vect->name);
#ifdef LIN_FILTER
    filter = FrvLinFiltButt(0., 0., 0., 0., f->order, f->freqCut*vect->dx[0]);
#else
    if(f->freqCut > 0) filter = FrvFilterButNew(f->order, 0, f->freqCut, NULL);
    else               filter = FrvFilterButNew(f->order,-f->freqCut, 0, NULL);
#endif
    if(filter == NULL) CfgMsgAddFatal("ButFilter new failed for%s",newName);
    FrStrCpy(&(filter->name), newName);
    filter->next = f->filter;
    f->filter = filter;}

  if(vect->compress != 0) FrVectExpand(vect);

  /*------------copy the auxiliary vector if the vector has missing sample---*/
  next = NULL;
  if(vect->next != NULL) {  
    if(vect->type == FR_VECT_C) { 
      if(strncmp(vect->name,"Available_data",14) == 0){
	next = FrVectCopy(vect);

	/*---------------------------------------------decimate the vector---*/
	vect->nData  = vect->nData/nGroup;
	vect->space  = vect->space/nGroup;
	vect->nx[0]  = vect->nx[0]/nGroup;
	vect->nBytes = vect->wSize * vect->nData;
	vect->dx[0]  = vect->dx[0]*nGroup;

	data = next->data;
	for(i=0; i < next->nData; i++) {
	  off = i*nGroup;
	  for(j=off+1; j<off+nGroup; j++) 
	    {if(data[j] < data[off]) data[off] = data[j];}}
      }}}

  /*--------------------------------------------------- filter the channel---*/
#ifdef LIN_FILTER
  FrvLinFiltProc(filter, vect);
  vect = vect->output;
#else 
  vect = FrvFilterButProc(filter,vect);

  if(filter->nResets != 0) {
    filter->nResets = 0;
    f->nReset++;
    if(f->nReset < 10) f->namesP += sprintf(f->namesP,"%s ",vect->name);
    else if(f->nReset == 10) sprintf(f->namesP," ...");}
#endif
  vect->next = next;

  return(vect);
}
/*---------------------------------------------------------------------------*/
FrAdcData** FdButFilterProcessAdc(FdButFilter *f,
				  FrCList *list,
				  FrAdcData** adcP)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *a, *adc, **next;
  FrVect *vect;
  char *newName;
  int i, nGroup, nbits;
  double scale, newFreq, dt;

  adc = *adcP;
  dt = adc->data->dx[0];

  /*-------skip channels whith a sampling rate lower than the resquest one---*/
  if(f->freqDecimation * dt > .5) return(&(adc->next));  

  /*-----check the consistency between sampling rate, vector and frame size---*/
  if(fabs(adc->data->nData * dt - f->frameDt) > 0.5*dt) {
    CfgMsgAddError("%.f: Inconsistent vector size: %ld*%g != %g for channel %s",
                   adc->data->GTime,adc->data->nData,dt,f->frameDt,adc->name);
    f->action->state = CfgServerActive;
    return(&(adc->next));}

  /*-------------------compute the decimation factor and the new frequency---*/
  if(f->freqCut <= 0) nGroup = 1;
  else                nGroup = (int)( 0.5 + 1./(dt*f->freqDecimation));
  newFreq = 1./(dt*nGroup);

  /*-------------------------------------------- build output channel name---*/
  if(f->freqCut <= 0) newName = FdButNewNameHP(adc->name, -f->freqCut);
  else                newName = FdButNewName  (adc->name, newFreq);

  /*------------------------------------------------------do the filtering---*/
  vect = FdButFilterProcessVect(f, list, adc->data, nGroup, newName);
  if(vect == NULL) return(&(adc->next));

  /*---------------------------------- copy the output buffer to the frame---*/
  if     (adc->data->type == FR_VECT_4R) nbits = -32;
  else if(adc->data->type == FR_VECT_8R) nbits = -64;
  else nbits = 32;
  if(f->storeAs64bits == FR_YES) nbits = -64;

  a = FrAdcDataNew(NULL,  newName, 1./(nGroup*vect->dx[0]), 
		   vect->nData/nGroup, nbits);
  if(a == NULL) CfgMsgAddFatal("FdButFilterProcessAdc malloc failed for %s",
			       newName);
  free(newName);

  if(a->units != NULL) free(a->units);
  FrStrCpy(&(a->units),adc->units);
  a->fShift    = adc->fShift;
  a->phase     = adc->phase;
  a->dataValid = adc->dataValid;

  if(vect->unitY != NULL) {
    free(a->data->unitY);
    FrStrCpy(&(a->data->unitY), vect->unitY);} 

  a->comment = FdButNewComment(adc->name, adc->comment, f->freqCut, f->order);

  /*------------------------------copy and decimate the filter output; ---
    -------------------------we take the last sample to reduce the latency---*/

  if(nbits == 32) {
    scale = sqrt(nGroup);
    a->slope = adc->slope/scale;
    a->bias = adc->bias*scale;
    for(i=0; i<a->data->nData; i++) {
      a->data->dataI[i] = floor(.5 + scale * vect->dataD[i*nGroup+nGroup-1]);}}
  else if(nbits == -32) {
    a->slope = adc->slope;
    a->bias  = adc->bias;
    for(i=0; i<a->data->nData; i++) {
      a->data->dataF[i] = vect->dataD[i*nGroup + nGroup - 1];}}
  else {
    a->slope = adc->slope;
    a->bias  = adc->bias;
    for(i=0; i<a->data->nData; i++) {
      a->data->dataD[i] = vect->dataD[i*nGroup + nGroup - 1];}}

  a->data->next = vect->next;

  /*------------------------- put the created structure in the linked list---*/
  if(f->keepInput == FR_YES) {
    a->next    = adc->next;
    a->nextOld = adc->nextOld;
    adc->next    = a;
    adc->nextOld = a;
    next = &(a->next);}

  else {   /*------------------or move the new info to the old structure----*/ 
    FrVectFree(adc->data);
    adc->data = a->data;
    a->data = NULL;

    if(adc->comment != NULL) free(adc->comment);
    adc->comment = a->comment;
    a->comment = NULL;

    free(adc->name);
    adc->name = a->name;
    a->name = NULL;

    if(a->units != NULL) free(adc->units);
    adc->units = a->units;
    a->units = NULL;

    adc->sampleRate = a->sampleRate;
    adc->slope      = a->slope;
    FrAdcDataFree(a);
    next = &(adc->next);}

  /*----------------------------return the next element in the linked list---*/
  return(next);
}
/*---------------------------------------------------------------------------*/
FrProcData** FdButFilterProcessProc(FdButFilter *f,
				    FrCList *list,
				    FrProcData** procP)
/*---------------------------------------------------------------------------*/
/* This function low pass filter a FrProcData channel. 
   It returns the next one in the linked list                                */
/*---------------------------------------------------------------------------*/
{
  FrProcData *p, *proc, **next;
  char *newName;
  int i, nGroup, nbits;
  FrVect *vect;
  double newFreq, dt;

  proc = *procP;
  dt = proc->data->dx[0];

  /*-------skip channels whith a sampling rate lower than the resquest one---*/
  if(f->freqDecimation * dt > .5) return(&(proc->next));  

  /*-----check the consistency between sampling rate, vector and frame size---*/
  if(fabs(proc->data->nData * dt - f->frameDt) > 0.5*dt) {
    CfgMsgAddError("%.f: Inconsistent vector size: %ld*%g != %g for channel %s",
                   proc->data->GTime,proc->data->nData,dt,f->frameDt,proc->name);
    f->action->state = CfgServerActive;
    return(&(proc->next));}

  /*------------------------------skip proc data which are not time series---*/
  if(proc->type > 2) return(&(proc->next));

  /*-------------------compute the decimation factor and the new frequency---*/
  if(f->freqCut <= 0) nGroup = 1;
  else                nGroup = (int)(0.5 + 1./(dt*f->freqDecimation));
  newFreq = 1./(dt*nGroup);

  /*-------------------------------------------- build output channel name---*/
  if(f->freqCut <= 0) newName = FdButNewNameHP(proc->name, -f->freqCut);
  else                newName = FdButNewName  (proc->name, newFreq);

  /*------------------------------------------------------do the filtering---*/
  vect = FdButFilterProcessVect(f, list, proc->data, nGroup, newName);
  if(vect == NULL) return(&(proc->next));

  /*------------------------------- crate the output FrProcData structuree---*/
  if  (proc->data->type == FR_VECT_8R) nbits = -64;
  else nbits = -32;
  if(f->storeAs64bits == FR_YES) nbits = -64;

  p = FrProcDataNew(NULL, newName, 1./(vect->dx[0]*nGroup), 
		    vect->nData/nGroup, nbits);
  if(p == NULL) CfgMsgAddFatal("FdButFilterProc malloc %s failed at %.0f",
			       newName, proc->data->GTime);
  free(newName);

  p->type = 1;
  FrVectSetUnitY(p->data, proc->data->unitY);
  p->fShift = proc->fShift;
  p->phase  = proc->phase;

  p->comment = FdButNewComment(proc->name, proc->comment, f->freqCut, f->order);

  /*------------------------------copy and decimate the filter output; ---
    -------------------------we take the last sample to reduce the latency---*/

  if(nbits == -64) {
    for(i=0; i < p->data->nData; i++) {
      p->data->dataD[i] = vect->dataD[i*nGroup + nGroup - 1];}}
  else {
    for(i=0; i < p->data->nData; i++) {
      p->data->dataF[i] = vect->dataD[i*nGroup + nGroup - 1];}}

  p->data->next = vect->next;

  /*------------------------- put the created structure in the linked list---*/
  if(f->keepInput == FR_YES) {
    p->next    = proc->next;
    p->nextOld = proc->nextOld;
    proc->next    = p;
    proc->nextOld = p;
    next = &(p->next);}

  else {   /*------------------or move the new info to the old structure----*/ 
    FrVectFree(proc->data);
    proc->data = p->data;
    p->data = NULL;
    if(proc->comment != NULL) free(proc->comment);
    proc->comment = p->comment;
    p->comment = NULL;
    free(proc->name);
    proc->name = p->name;
    p->name = NULL;
    FrProcDataFree(p);
    next = &(proc->next);}

  /*----------------------------return the next element in the linked list---*/
  return(next);
}
/*---------------------------------------------------------------------------*/
void FdButFilterProcessOne(FdButFilter *f, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
#ifdef LIN_FILTER
  FrvLinFilt ff;
#else
  FrvFilterB ff;
#endif
  FrProcData **proc;
  FrAdcData **adc;
  FrCList *list;
  double gtime;

  f->action->state = CfgServerGolden;
  f->frameDt = frame->dt;
  f->nDupInput = 0;
  f->nReset    = 0;
  f->namesP = f->names;

  /*--------------------build a alphabetique list of the available filters---*/
  list = FrCListBldList(f->filter, 
			(long)&(ff.name) - (long)&ff, 
			(long)&(ff.next) - (long)&ff);
 
  FrameTag(frame, f->tag);

  gtime = frame->GTimeS + 1.e-9*frame->GTimeN;

  /*------------------------------------------------------do the filtering---*/
  for(proc = &(frame->procData); *proc != NULL;) {
    if((*proc)->data == NULL) continue;
    (*proc)->data->GTime = gtime;
    proc = FdButFilterProcessProc(f, list, proc);}

  if(frame->rawData != NULL) {
    for(adc = &(frame->rawData->firstAdc); *adc != NULL;) {
      if((*adc)->data == NULL) {
        CfgMsgAddError("%d: no vector for ADC channel %s",
                       frame->GTimeS, (*adc)->name);
        f->action->state = CfgServerActive;
        continue;}
      (*adc)->data->GTime = gtime;
      adc = FdButFilterProcessAdc(f, list, adc);}}

  FrameUntag(frame);

  FrCListFree(list);

  /*---------------------------------------------report the filter reset(s)---*/
  f->action->userInfo[0] = '\0';

  if(f->nReset > 0) {
    f->action->state = CfgServerActive;
    snprintf(f->action->userInfo, f->action->userInfoSize, 
	"filter reset for: %s",f->names);}

  if(f->nReset != f->nResetLast) {
     f->nResetLast = f->nReset;
     if(f->nReset == 0) 
       CfgMsgAddInfo("At %d no more filter to reset", frame->GTimeS);
     else
      CfgMsgAddWarning("At %d filter registers reset for channel: %s",
        frame->GTimeS, f->names);}

  /*--------------------------report the number of duplicated input channels---*/
  if(f->nDupInput > 0) {
    snprintf(f->action->userInfo, f->action->userInfoSize,
	"%d filtered channels with multiple inputs", f->nDupInput);}

  return;
}
/*---------------------------------------------------------------------------*/
void FdButFilterProcess(FdButFilter *filter, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdButFilterProcessOne(filter, frame);

  next = filter->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdButFilterNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdButFilter *f;

  f = (FdButFilter*) calloc(1,sizeof(FdButFilter));
  if(f == NULL) CfgMsgAddFatal("malloc FdButFilter failed");

  f->action = FdActionNew((FdAction**) arg, 
			  (void*) f, FdButFilterProcess, "ButFilter");
  f->names = malloc(2048);

  f->order = CfgParseGetNextDec(data);
  FrStrCpy(&(f->tag), CfgParseGetNextString(data));
  f->freqCut        = CfgParseGetNextReal(data);
  f->freqDecimation = CfgParseGetNextReal(data);
  f->mode           = CfgParseGetNextDec(data);

  if(f->freqCut < 0) f->freqDecimation = 0;

  CfgMsgAddInfo("Add %d order Butterworth filter for channel(s) %s "
		"with fCut=%gHz and decimated at %gHz mode=%d",
		f->order, f->tag, f->freqCut, f->freqDecimation, f->mode);

  if     (f->mode == 0) 
    CfgMsgAddInfo("  mode=0: Replace channels content by filter output; "
		  "int are converted to float (32bits)");
  else if(f->mode == 1) 
    CfgMsgAddInfo("  mode=1: Add new channels; input channels are unchanged");
  else if(f->mode == 2) 
    CfgMsgAddInfo("  mode=2: Replace channels content by filter output"
		  " stored as double (64bits)");
  else if(f->mode == 3) 
    CfgMsgAddInfo("  mode=3: Add new channels (64 bits double); "
		  "input channels are unchanged");
  else  CfgMsgAddFatal(" valid mode values are 0, 1, 2 and 3"); 

  if((f->mode & 0x1) == 1) {f->keepInput = FR_YES;}
  else                     {f->keepInput = FR_NO;}

  if((f->mode & 0x2) == 2) {f->storeAs64bits = FR_YES;}
  else                     {f->storeAs64bits = FR_NO;}

  f->list = NULL;

#ifdef LIN_FILTER
  if(f->order > 6){
    f->order = 6;
    CfgMsgAddError(" The filter order has been reduced to 6");}
#endif

  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
void FdFFTFilterAddTF(FrvFDFilter *filter, char* fileName)
/*-----------------------------------------------------------------------------*/
{
  FrVect *tfVect;
  int   i, nLines;
  double dnu, freq, tf, tfMin, tfMax, fMin, fMax, lastFreq;
  FILE *fp;
  size_t len;
  char *line;

  CfgMsgAddInfo("  Add TF from %s", fileName);

  len = strlen(fileName);

  /*--------------------first case: the TF is stored as a text file---*/
  if(strcmp(fileName+len-4,".txt") == 0) {
    fp = fopen(fileName,"r");
    if(fp == NULL) CfgMsgAddFatal("could not open file %s",fileName);

    len = 0;
    line = NULL;
    nLines = 0;
    lastFreq = 0;
    tfMin = 1.e40;
    tfMax = 0;
    fMin = -1.;
    fMax = -1.;
    while (getline(&line, &len, fp) > 0) {
      if(sscanf(line,"%lg %lg",&freq, &tf) != 2) break;
      nLines++;
      if(nLines == 1)  CfgMsgAddInfo("  First values: TF(%g Hz)=%g",freq, tf);
      if(freq < lastFreq) CfgMsgAddFatal("Decreasing frequency %g at line %d",
					  freq, nLines);
      lastFreq = freq;
      if(tf < tfMin) {
	tfMin = tf;
	fMin = freq;}
      if(tf > tfMax) {
	tfMax = tf;
	fMax = freq;}
      FrvFDFilterAddTFBin(filter, freq ,tf ,0);}
    CfgMsgAddInfo("  last value: TF(%g Hz)=%g", freq, tf);
    CfgMsgAddInfo("  %d lines read; min: TF(%g Hz)=%g max: TF(%g Hz)=%g", 
                nLines, fMin, tfMin, fMax, tfMax);
    fclose(fp);}

  /*----------------- second option: the sensitivity is stored in an FrVect ---*/
  else {
    tfVect = FrVectLoad(fileName);
    if(tfVect == NULL) 
      CfgMsgAddFatal("Could not read vector from %s", fileName);

    FrvFDFilterFreeTFBin(filter);
    for(i=1; i<tfVect->nData; i++) {
      dnu = tfVect->dx[0];
      FrvFDFilterAddTFBin(filter,i*dnu,FrVectGetValueI(tfVect,i),0);}

    FrVectFree(tfVect);}

  return;
}
/*---------------------------------------------------------------------------*/
void FdFFTFilterProcessChannel(FdFFTFilter *f,
			       FrCList *list,
			       FrameH* lastFrame,
			       FrVect *vect,
			       double slope,
			       char *unitY)
/*---------------------------------------------------------------------------*/
{
  FrProcData *p;
  char newName[512];
  FrvFDFilter *filter;

  /*------------------------------- filter already exist? If not create it---*/
  filter = (FrvFDFilter*) FrCListFind(list, vect->name);
  if(filter == NULL) {
    CfgMsgAddInfo(" FdFFTFilterNew: add channel %s", vect->name);
    filter = FrvFDFilterNew(f->freqMax, 1., f->delay, f->freqMax);
    if(filter == NULL) CfgMsgAddFatal("FFTilter new failed for%s",vect->name);
    filter->next = f->filter;
    f->filter = filter;
    FrvFDFilterSetName(filter, vect->name);
    if(f->freqMin > 0) FrvFDFilterAddHole(filter, 0, f->freqMin, 0.,0);
    if(f->tfFileName != NULL) FdFFTFilterAddTF(filter, f->tfFileName);
    }

  /*--------------------------------------------------- filter the channel---*/
  FrvFDFilterProc(filter, vect);

  if(lastFrame == NULL) return;

  /*------ do not include old data (due to a data gap) in the output frame---*/ 
  if(filter->fOut->output->GTime != lastFrame->GTimeS) return;

  /*-------------------------------------------- build output channel name---*/
  if(f->tfFileName != NULL) {
    sprintf(newName,"%s_recolored",vect->name);}
  else if(f->delay == 0) {
    sprintf(newName,"%s_%gHz_%gHz",vect->name, f->freqMin, f->freqMax);}
  else {
    sprintf(newName,"%s_%gHz_%gHz_%.0fus",
	    vect->name, f->freqMin, f->freqMax, 1.e6*f->delay);}

  /*---------------------------------- copy the output buffer to the frame---*/
  p = FrProcDataNewV(lastFrame,FrVectCopyToF(filter->fOut->output,
					     slope,newName));
  if(p == NULL) CfgMsgAddFatal("FdFFTFilterProc malloc %s failed at %d",
			       newName, lastFrame->GTimeS);
  p->type = 1;
  if(f->tfFileName != NULL) {
    p->comment = malloc(strlen(vect->name)+20+strlen(f->tfFileName));
    sprintf(p->comment,"%s recolored using %s", vect->name, f->tfFileName);}
  else {
    p->comment = malloc(strlen(vect->name)+128);
    sprintf(p->comment,"%s Filtered by FdFFTFilterProc", vect->name);}
  FrVectSetUnitY(p->data, unitY);

  /*----------- just for debugging---
  p = FrProcDataNewV(lastFrame,FrVectCopy(filter->fOut->windowOut));
  p = FrProcDataNewV(lastFrame,FrVectCopy(filter->fOut->last));
  p = FrProcDataNewV(lastFrame,FrVectCopy(filter->fOut->fftOut));
  ------*/

  return;}

/*---------------------------------------------------------------------------*/
void FdFFTFilterProcessOne(FdFFTFilter *f,
			   FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrvFDFilter ff;
  FrProcData *proc;
  FrAdcData *adc;
  FrCList *list;

  /*-----------------in case of frame discontinuity do not fill the output---*/
  if(f->lastFrame != NULL) {
    if(f->lastFrame->GTimeS + f->lastFrame->dt != frame->GTimeS) {
      CfgMsgAddInfo("FdFFTFilter: frame discontinuty: %d + %g != %d",
		    f->lastFrame->GTimeS, f->lastFrame->dt, frame->GTimeS);}}

  /*------------------------------------------------------do the filtering---*/
  FrameTag(frame, f->tag);
  list = FrCListBldList(f->filter, 
			(long)&(ff.name) - (long)&ff, 
			(long)&(ff.next) - (long)&ff);

  for(proc = frame->procData; proc != NULL; proc = proc->next) {
    if(proc->data == NULL) continue;
    FdFFTFilterProcessChannel(f, list, f->lastFrame, proc->data,
			      1., proc->data->unitY);}

  if(frame->rawData != NULL) {
    for(adc = frame->rawData->firstAdc; adc != NULL; adc = adc->next) {
      if(adc->data == NULL) continue;
      FdFFTFilterProcessChannel(f, list, f->lastFrame, adc->data,
				adc->slope, adc->units);}}

  FrameUntag(frame);
  FrCListFree(list);

  return;}

/*---------------------------------------------------------------------------*/
void FdFFTFilterProcess(FdFFTFilter *filter, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  next = filter->action->next;

  /*-----------process the data or flush the current frame if terminating---*/
  if(frame != NULL) {
    FdFFTFilterProcessOne(filter, frame);
    if(filter->lastFrame != NULL && 
      next != NULL) next->action(next->data, filter->lastFrame);}

  else if(next != NULL) {
    next->action(next->data, filter->lastFrame);
    filter->lastFrame  = NULL;
    next->action(next->data, NULL);}

  filter->lastFrame = frame;

  return;
}
/*--------------------------------------------------------------------------*/
int FdFFTFilterNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdFFTFilter *f;

  f = (FdFFTFilter*) calloc(1,sizeof(FdFFTFilter));
  if(f == NULL) CfgMsgAddFatal("malloc FdFFTFilter failed");

  f->action = FdActionNew((FdAction**) arg, 
			  (void*) f, FdFFTFilterProcess, "FFTFilter");

  FrStrCpy(&(f->tag), CfgParseGetNextString(data));
  f->freqMin = CfgParseGetNextReal(data);
  f->freqMax = CfgParseGetNextReal(data);
  f->delay   = CfgParseGetNextReal(data);

  f->lastFrame = NULL;

  CfgMsgAddInfo("Add FFT filter on channel(s) %s [%g-%gHz] delay=%gs",
		f->tag, f->freqMin, f->freqMax, f->delay);

  return(CFG_OK);}
 
/*--------------------------------------------------------------------------*/
int FdFFTFilterRecolorNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdFFTFilter *f;

  f = (FdFFTFilter*) calloc(1,sizeof(FdFFTFilter));
  if(f == NULL) CfgMsgAddFatal("malloc FdFFTFilter failed");

  f->action = FdActionNew((FdAction**) arg, 
			  (void*) f, FdFFTFilterProcess, "FFTFilter");

  FrStrCpy(&(f->tag),        CfgParseGetNextString(data));
  FrStrCpy(&(f->tfFileName), CfgParseGetNextString(data));

  f->lastFrame = NULL;

  CfgMsgAddInfo("Add FFT recoloring on channel(s) %s TF from %s",
		f->tag, f->tfFileName);

  return(CFG_OK);}
 
/*---------------------------------------------------------------------------*/
void FdFilterParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
 
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_BUT_FILTER",
			 FdButFilterNew, (void *) &(fdIO->actionsIn), 5, 
			 CfgDec, CfgString, CfgReal, CfgReal, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_BUT_FILTER", 
			 FdButFilterNew, (void *) &(fdIO->actionsOut), 5,
			 CfgDec, CfgString, CfgReal, CfgReal, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_FFT_FILTER", 
			 FdFFTFilterNew, (void *) &(fdIO->actionsIn), 4,
			 CfgString, CfgReal, CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_FFT_FILTER", 
			 FdFFTFilterNew, (void *) &(fdIO->actionsOut), 4,
			 CfgString, CfgReal, CfgReal, CfgReal);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_RECOLOR_CHANNELS", 
			 FdFFTFilterRecolorNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_RECOLOR_CHANNELS", 
			 FdFFTFilterRecolorNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgString);

  return;
}
