#include "FdIO.h"

/*-------------------------------------------------------private functions---*/

typedef struct FdDownSample FdDownSample;

struct FdDownSample{
  FrTag*   tag;         /* tag to select channel on which we apply the down sampling*/
  int      period;      /* new period to keep the data*/
  FdAction* action;     /* this is the action process */
};

typedef struct FdHistory FdHistory;

struct FdHistory{
  char*    comment;     /* new history comment*/
  FdAction* action;     /* this is the action process */
};

typedef struct FdSelect FdSelect;

struct FdSelect{
  char*     tag;         /* selection of channel to be removed  */
  int       period;      /* period used to keep the data */
  int       duration;    /* time window without selection (if period > 0) */
  FdAction* action;      /* this is the action process */
};

typedef struct FdRename FdRename;

struct FdRename{
  char*    oldName;      /* name of the channel to be renamed  */
  char*    newName;      /* new name for this channel */
  FdAction* action;      /* this is the action process */
};

typedef struct FdSetSuffix FdSetSuffix;

struct FdSetSuffix{
  char*    suffix;      /* suffix to be set */
  char*    tag;         /* tag to select channel on which we set the suffix*/
  FdAction* action;     /* this is the action process */
};

typedef struct FdSetPrefix FdSetPrefix;

struct FdSetPrefix{
  char*    prefix;      /* prefix to be set */
  char*    tag;         /* tag to select channel on which we set the prefix*/
  FdAction* action;     /* this is the action process */
};

typedef struct FdSetProcType FdSetProcType;

struct FdSetProcType{
  char*    tag;         /* tag to select channel on which we set the type*/
  int      type;        /* new type value*/
  FdAction* action;     /* this is the action process */
};

typedef struct FdFreqCut FdFreqCut;

struct FdFreqCut{
  char*    tag;         /* tag to select channel on which we apply the cut*/
  double   freqCut;     /* maximum frequency to keep a channel*/
  FdAction* action;     /* this is the action process */
};

void FdDownSampleProcess(FdDownSample *ds, FrameH* frame);
void FdFreqCutProcess(FdFreqCut *select, FrameH* frame);
void FdHistoryProcess(FdHistory *h, FrameH* frame);
void FdRenameProcess(FdRename *rename, FrameH* frame);
void FdSetPrefixProcess(FdSetPrefix *prefix, FrameH* frame);
void FdSetSuffixProcess(FdSetSuffix *suffix, FrameH* frame);
void FdSetProcTypeProcess(FdSetProcType *setProcType, FrameH* frame);
void FdSelectProcess(FdSelect *select, FrameH* frame);

/*--------------------------------------------------------------------------*/
int FdDownSampleNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdDownSample* ds;
  char *tag;

  ds = (FdDownSample*) calloc(1,sizeof(FdDownSample));
  if(ds == NULL) CfgMsgAddFatal("malloc FdDownSample failed");

  ds->action = FdActionNew((FdAction**) arg,
                               (void*) ds, FdDownSampleProcess, "DownSample");

  tag        = CfgParseGetNextString(data);
  ds->tag    = FrTagNew(tag);
  ds->period = CfgParseGetNextDec(data);

  CfgMsgAddInfo("The 2d or 3d channel selection %s will be down sampled "
               "to 1/%d Hz", tag, ds->period);

  if(ds->period <= 0) CfgMsgAddFatal(" new period should be > 0");

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdDownSampleProcess(FdDownSample *ds, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;
  FrAdcData **adc;

  if(frame != NULL) {
    if(frame->rawData != NULL) {
      for(adc = &(frame->rawData->firstAdc); *adc != NULL;) {
        if((*adc)->data->nDim < 2) {
           adc = &((*adc)->next);
           continue;}
        if(FrTagMatch(ds->tag, (*adc)->name) == FR_NO) {
           adc = &((*adc)->next);
           continue;}
        if(frame->GTimeS % ds->period == 0) {
          if(ds->period < frame->dt) {CfgMsgAddError(
                 "Too small requested down sampling period (%d %g) for %s",
                ds->period , frame->dt, (*adc)->name); 
            ds->period = frame->dt;}
          FrVectExpand((*adc)->data);
          (*adc)->sampleRate = 1./((double)ds->period);
          (*adc)->data->dx[0] = ds->period;
          (*adc)->data->nBytes /= (*adc)->data->nx[0];
          (*adc)->data->nData  /= (*adc)->data->nx[0];
          (*adc)->data->nx[0] = 1;
          adc = &((*adc)->next);}
        else {
          *adc = FrAdcDataFreeOne(*adc);}
      }}}

  next = ds->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdHistoryNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdHistory* h;

  h = (FdHistory*) calloc(1,sizeof(FdHistory));
  if(h == NULL) CfgMsgAddFatal("malloc FdHistory failed");

  h->action = FdActionNew((FdAction**) arg,
                               (void*) h, FdHistoryProcess, "History");

  FrStrCpy(&h->comment, CfgParseGetNextString(data));

  CfgMsgAddInfo("All history records will be replaced by one with comment:%s ",
               h->comment);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdHistoryProcess(FdHistory *h, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  if(frame != NULL) {
    FrHistoryFree(frame->history);
    int mytime = FrGetCurrentGPS();
    frame->history = FrHistoryNew(NULL, mytime, h->comment);}

  FdAction *next = h->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdRenameNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdRename *rename;

  rename = (FdRename*) calloc(1,sizeof(FdRename));
  if(rename == NULL) CfgMsgAddFatal("malloc FdRename failed");

  rename->action = FdActionNew((FdAction**) arg, 
			       (void*) rename, FdRenameProcess, "Rename");

  FrStrCpy(&(rename->oldName),  CfgParseGetNextString(data));
  FrStrCpy(&(rename->newName),  CfgParseGetNextString(data));

  CfgMsgAddInfo("Add rename channel:  %s to %s",
		rename->oldName, rename->newName);

  return(CFG_OK);}

 
/*---------------------------------------------------------------------------*/
void FdRenameProcessOne(FdRename *rename, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  FrProcData *proc;
  char *comment;

  for(proc = frame->procData; proc != NULL; proc = proc->next) {
    if(strcmp(rename->oldName, proc->name) != 0) continue;

    if(proc->comment != NULL) {
      if(proc->comment[0] == '\0') {
        free(proc->comment);
        proc->comment = NULL;}}

    if(proc->comment != NULL) {
      comment = malloc(12+strlen(proc->name)+strlen(proc->comment));
      sprintf(comment,"%s; %s renamed",proc->comment, proc->name);
      free(proc->comment);
      proc->comment = comment;}

    if(proc->comment == NULL) {
            proc->comment = malloc(12+strlen(proc->name));
       if(proc->comment != NULL)
          sprintf(proc->comment,"%s renamed",proc->name);}

    free(proc->name);
    FrStrCpy(&proc->name, rename->newName);
    FrVectSetName(proc->data, rename->newName);
    return;}

  if(frame->rawData == NULL) return;

  for(adc = frame->rawData->firstAdc; adc != NULL; adc = adc->next) {
    if(strcmp(rename->oldName, adc->name) != 0) continue;

    if(adc->comment != NULL) {
      if(adc->comment[0] == '\0') {
        free(adc->comment);
        adc->comment = NULL;}}

    if(adc->comment != NULL) {
      comment = malloc(12+strlen(adc->name)+strlen(adc->comment));
      sprintf(comment,"%s; %s renamed",adc->comment, adc->name);
      free(adc->comment);
      adc->comment = comment;}

    if(adc->comment == NULL) {
	    adc->comment = malloc(12+strlen(adc->name));
       if(adc->comment != NULL)
          sprintf(adc->comment,"%s renamed",adc->name);}

    free(adc->name);
    FrStrCpy(&adc->name, rename->newName);
    FrVectSetName(adc->data, rename->newName);
    return;}

 return;}

/*---------------------------------------------------------------------------*/
void FdRenameProcess(FdRename *rename,
		     FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdRenameProcessOne(rename, frame);

  /*-------------------------------call the next action in the linked list---*/
  next = rename->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdSetPrefixNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdSetPrefix* prefix;

  prefix = (FdSetPrefix*) calloc(1,sizeof(FdSetPrefix));
  if(prefix == NULL) CfgMsgAddFatal("malloc FdSetPrefix failed");

  prefix->action = FdActionNew((FdAction**) arg, 
			       (void*) prefix, FdSetPrefixProcess, "Prefix");

  FrStrCpy(&(prefix->prefix),  CfgParseGetNextString(data));
  FrStrCpy(&(prefix->tag),     CfgParseGetNextString(data));

  if(prefix->tag == NULL) 
	CfgMsgAddInfo("Add prefix  %s to all channels", prefix->prefix);
  else	CfgMsgAddInfo("Add prefix  %s to %s channels",  prefix->prefix,
		prefix->tag);

  return(CFG_OK);
}
/*--------------------------------------------------------------------------*/
int FdSetPrefixObsolete(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  CfgMsgAddFatal("FDOUT_CHPREFIX and FDOUT_SET_PREFIX are obsolete and "
                 "replaced by FDOUT_ADD_PREFIX");

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdSetPrefixProcess(FdSetPrefix *prefix, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) {
    FrameTag(frame, prefix->tag);
    FrameSetPrefix(frame, prefix->prefix);
    FrameUntag(frame);}

  next = prefix->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdSetSuffixNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdSetSuffix* suffix;

  suffix = (FdSetSuffix*) calloc(1,sizeof(FdSetSuffix));
  if(suffix == NULL) CfgMsgAddFatal("malloc FdSetSuffix failed");

  suffix->action = FdActionNew((FdAction**) arg, 
			       (void*) suffix, FdSetSuffixProcess, "Suffix");

  FrStrCpy(&(suffix->suffix),  CfgParseGetNextString(data));
  FrStrCpy(&(suffix->tag),     CfgParseGetNextString(data));

  if(suffix->tag) 
	CfgMsgAddInfo("Add suffix  %s to all channels", suffix->suffix);
  else	CfgMsgAddInfo("Add suffix  %s to %s channels",  suffix->suffix, 
		suffix->tag);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdSetSuffixProcess(FdSetSuffix *suffix, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) {
    FrameTag(frame, suffix->tag);
    FrameSetSuffix(frame, suffix->suffix);
    FrameUntag(frame);}

  next = suffix->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdSetProcTypeNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdSetProcType* setProcType;

  setProcType = (FdSetProcType*) calloc(1,sizeof(FdSetProcType));
  if(setProcType == NULL) CfgMsgAddFatal("malloc FdSetProcType failed");

  setProcType->action = FdActionNew((FdAction**) arg, (void*) setProcType, 
				    FdSetProcTypeProcess, "ProcType");

  FrStrCpy(&(setProcType->tag),     CfgParseGetNextString(data));
  setProcType->type = CfgParseGetNextDec(data);

  CfgMsgAddInfo("The type of the FrProcData %s will be set to %d",
		setProcType->tag, setProcType->type);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdSetProcTypeProcess(FdSetProcType *setProcType, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;
  FrProcData *proc;

  if(frame != NULL) {
    FrameTag(frame, setProcType->tag);

    for(proc=frame->procData; proc != NULL; proc = proc->next) {
      proc->type = setProcType->type;}

    FrameUntag(frame);}

  next = setProcType->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdFreqCutNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdFreqCut* freqCut;

  freqCut = (FdFreqCut*) calloc(1,sizeof(FdFreqCut));
  if(freqCut == NULL) CfgMsgAddFatal("malloc FdFreqCut failed");

  freqCut->action = FdActionNew((FdAction**) arg, 
			       (void*) freqCut, FdFreqCutProcess, "FreqCut");

  FrStrCpy(&(freqCut->tag), CfgParseGetNextString(data));
  freqCut->freqCut = CfgParseGetNextReal(data);

  CfgMsgAddInfo("Add freqCut at %gHz to %s channels",
		freqCut->freqCut, freqCut->tag);

  return(CFG_OK);
}
/*--------------------------------------------------------------------------*/
int FdFreqCutObsolete(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  CfgMsgAddFatal("FDOUT_CHFREQCUT is obsolete and replaced by "
		 "FDOUT_FREQUENCY_CUT");

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdFreqCutProcess(FdFreqCut *freqCut, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) {
    FrameTag(frame, freqCut->tag);
    FrameFreqCut(frame, freqCut->freqCut);
    FrameUntag(frame);}

  next = freqCut->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdSelectNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdSelect* select;

  select = (FdSelect*) calloc(1,sizeof(FdSelect));
  if(select == NULL) CfgMsgAddFatal("malloc FdSelect failed");

  select->action = FdActionNew((FdAction**) arg, 
			       (void*) select, FdSelectProcess, "SelectCh");

  FrStrCpy(&(select->tag), CfgParseGetNextString(data));
  select->period   = CfgParseGetNextDec(data);
  select->duration = CfgParseGetNextDec(data);

  if(select->period == 0)
    CfgMsgAddInfo("The channel selection %s will be always applied",
                  select->tag);
  else
    CfgMsgAddInfo("The channel selection %s will be apply except every %d"
		  " seconds for %d seconds", select->tag,
		  select->period,select->duration);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdSelectProcess(FdSelect *select, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;
  double tPart;

  if(frame != NULL) {
    if(select->period > 0) tPart = frame->GTimeS % select->period;
    else        	   tPart = select->duration;
    if(tPart >= select->duration) {
      FrameTag(frame, select->tag);
      FrameRemoveUntagged(frame);
      FrameUntag(frame);}}

  next = select->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
void FdRenameParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
 
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_DOWN_SAMPLING",
			 FdDownSampleNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_DOWN_SAMPLING",
			 FdDownSampleNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_HISTORY",
			 FdHistoryNew, (void *) &(fdIO->actionsIn), 1,
			 CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_HISTORY",
			 FdHistoryNew, (void *) &(fdIO->actionsOut), 1,
			 CfgString);

  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_FREQUENCY_CUT",
			 FdFreqCutNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgReal);

  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_MAX_FREQUENCY",
			 FdFreqCutObsolete, NULL, 1, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_ADD_PREFIX",
			 FdSetPrefixNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_ADD_PREFIX",
			 FdSetPrefixNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_SET_PREFIX",
			 FdSetPrefixObsolete, NULL, 1, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_ADD_SUFFIX",
			 FdSetSuffixNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_ADD_SUFFIX",
			 FdSetSuffixNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgString);

  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_CHPREFIX",
			 FdSetPrefixObsolete, NULL, 1, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_RENAME_CHANNEL",
			 FdRenameNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_RENAME_CHANNEL",
			 FdRenameNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgString);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_SELECT_CHANNELS",
			 FdSelectNew, (void *) &(fdIO->actionsIn), 3,
			 CfgString, CfgDec, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_SELECT_CHANNELS",
			 FdSelectNew, (void *) &(fdIO->actionsOut), 3,
			 CfgString, CfgDec, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_SET_PROC_TYPE",
			 FdSetProcTypeNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_SET_PROC_TYPE",
			 FdSetProcTypeNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgDec);

  return;
}
