#include "FdIO.h"

typedef struct FdRescale  FdRescale;
typedef struct FdNTCT  FdNTCT;

struct FdNTCT{ // this is for Negative Temperature Coefficient (NTCT) Thermistors
  char*      outName;     // output channel name 
  int        rate;        // output rate        
  char*      inName;      // input channel name
  double     ADC_R;       // ADC input impedance
  double     NTCT_R;      // NTCT resistor      
  double     NTCT_V;      // Voltage apply to the NTCT  
  double     beta;        // NTCT beta coef.  
  double     Tref;        // reference temperature (deg C) like 25 
  char*      RName;       // resistor channel name if requested
  FRBOOL     present;     // to control the "present" error message
  FdAction*  action;      // action object which hold this recast object  
};

struct FdRescale{
  char*      output;      /* output channel name                             */
  double     scale;       /* scaling factor for input                        */
  char*      input;       /* input channel name                              */
  char*      units;       /* new units if the channel content has been changed
			     or NULL if the units are unchanged              */
  FRBOOL     present;     /* to control the "present" error message          */
  FRBOOL     alreadyThere;/* to control the "laready there" error message    */
  FdAction*  action;      /* action object which hold this recast object     */
};
 
/*-----------------------------------------------declare private functions---*/
int  FdNTCTNew(void *arg, CfgDataType_t *data);
void FdNTCTProcess(FdNTCT* list, FrameH* frame);
 
int  FdRescaleNew(void *arg, CfgDataType_t *data);
void FdRescaleProcess(FdRescale* list, FrameH* frame);
 
/*---------------------------------------------------------------------------*/
int FdNTCTNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdNTCT *ntct;

  /*------------create the object and add it at the end of the linked list---*/
  ntct = (FdNTCT*) calloc(1,sizeof(FdNTCT));
  if(ntct == NULL) CfgMsgAddFatal("malloc FdNTCT failed");

  ntct->action = FdActionNew((FdAction**) arg, 
			       (void*) ntct, FdNTCTProcess, "NTCT");

  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&ntct->outName, CfgParseGetNextString(data));
  ntct->rate  =            CfgParseGetNextDec(data);
  FrStrCpy(&ntct->inName,  CfgParseGetNextString(data));
  ntct->ADC_R  =           CfgParseGetNextReal(data);
  ntct->NTCT_R =           CfgParseGetNextReal(data);
  ntct->NTCT_V =           CfgParseGetNextReal(data);
  ntct->beta   =           CfgParseGetNextReal(data);
  ntct->Tref   =           CfgParseGetNextReal(data);
  FrStrCpy(&ntct->RName,   CfgParseGetNextString(data));
  ntct->present = FR_YES;

  CfgMsgAddInfo("New NTCT channel %s", ntct->outName);

  ntct->Tref += 273.15; // go from deg. C to K.
 
  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdNTCTProcessOne(FdNTCT* ntct, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  FrVect *vectT, *vectR;
  int nGroup, i;
  double volts;

  //---------------------------------get input channel
  adc = FrAdcDataFind(frame, ntct->inName);
  if(adc == NULL) {
    if(ntct->present == FR_YES) {
      ntct->present = FR_YES;
      CfgMsgAddError("%d: %s is missing", frame->GTimeS, ntct->inName);}
    return;}
  
  if(ntct->present == FR_NO) {
    CfgMsgAddInfo("%d: channel %s is back", frame->GTimeS, ntct->inName);
    ntct->present = FR_YES;}

  //------------------------------compute output channel
  vectT = FrVectCopyToF(adc->data, adc->slope, ntct->outName);
  nGroup = 1./(vectT->dx[0] * ntct->rate);
  FrVectDecimate(vectT, nGroup, NULL);
  vectR = FrVectCopyToF(vectT, 0, ntct->RName);
 
  for(i=0; i<vectT->nData; i++) {
    volts = vectT->dataF[i] + adc->bias;
    if(volts < 1.e-3) volts = 1.e-3;
    vectR->dataF[i] = ntct->ADC_R * (ntct->NTCT_V - volts) / volts;
    if(vectR->dataF[i] < 1) vectR->dataF[i] = 1;
    vectT->dataF[i] = 1./(log(vectR->dataF[i]/ntct->NTCT_R)/ntct->beta + 1./ntct->Tref);
    vectT->dataF[i]-= 273.15;}

  //--------------------------------save results in frame
  FrVectSetUnitY(vectT,"T deg. C");
  FrProcDataNewVT(frame, vectT, 1);

  if(ntct->RName != NULL) {
    FrVectSetUnitY(vectR,"R [Ohms]");
    FrProcDataNewVT(frame, vectR, 1);}
  else FrVectFree(vectR);

  return;
}
/*---------------------------------------------------------------------------*/
void FdNTCTProcess(FdNTCT *ntct,
		     FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdNTCTProcessOne(ntct, frame);

  next = ntct->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
int FdRescaleNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdRescale *r;

  /*------------create the object and add it at the end of the linked list---*/
  r = (FdRescale*) calloc(1,sizeof(FdRescale));
  if(r == NULL) CfgMsgAddFatal("malloc FdRescale failed");

  r->action = FdActionNew((FdAction**) arg, 
			       (void*) r, FdRescaleProcess, "Rescale");
  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&r->output, CfgParseGetNextString(data));
  r->scale =           CfgParseGetNextReal(data);
  FrStrCpy(&r->input,  CfgParseGetNextString(data));
  FrStrCpy(&r->units,  CfgParseGetNextString(data));
  r->present = FR_YES;
  r->alreadyThere = FR_NO;

  CfgMsgAddInfo("New output channel %s = %g*%s units=%s",
		r->output, r->scale, r->input, r->units);
 
  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdRescaleProcessOne(FdRescale* r,
			FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  FrProcData *proc;
  FRBOOL present, alreadyThere;

  present = FR_YES;

  /*-----------------first deal with the case were the channel is unchanged---*/
  if(strcmp(r->input, r->output) == 0) {

    if((adc = FrAdcDataFind(frame, r->input)) != NULL) {
      adc->slope *= r->scale;
      if(r->units != NULL) {
        free(adc->units);
        FrStrCpy(&(adc->units),r->units);}}
    
    else if((proc = FrProcDataFind(frame, r->input)) != NULL) {
      FrvScale(r->scale, proc->data, proc->data, NULL);
      if(r->units != NULL) FrVectSetUnitY(proc->data, r->units);}      

    else {present = FR_NO;}}

  /*-------create a new channel; first check that it is not already there---*/
  else {
    alreadyThere = FR_NO;
    if((FrAdcDataFind(frame, r->output) != NULL) ||
       (FrProcDataFind(frame, r->output) != NULL)){
      alreadyThere = FR_YES;}

    else if((adc = FrAdcDataFind(frame, r->input)) != NULL) {
      adc = FrAdcDataCopy(adc, frame);
      if(adc == NULL) CfgMsgAddFatal("FdRescaleProcessOne: malloc failed");
      free(adc->name);
      FrStrCpy(&(adc->name),r->output);
      FrVectSetName(adc->data, r->output);
      adc->slope *= r->scale;
      if(r->units != NULL) {
        free(adc->units);
        FrStrCpy(&(adc->units),r->units);}}

    else if((proc = FrProcDataFind(frame, r->input)) != NULL) {
      proc = FrProcDataCopy(proc, frame);
      if(proc == NULL) CfgMsgAddFatal("FdRescaleProcessOne: malloc failed");
      free(proc->name);
      FrStrCpy(&(proc->name),r->output);
      FrVectSetName(proc->data, r->output);
      FrvScale(r->scale, proc->data, proc->data, NULL);
      if(r->units != NULL) FrVectSetUnitY(proc->data, r->units);}
 
    else {present = FR_NO;}

    if(alreadyThere != r->alreadyThere) {
      if(alreadyThere == FR_YES) 
        CfgMsgAddError("%d: channel %s is already there, could not rescale it",
                        frame->GTimeS, r->output);
      else
        CfgMsgAddInfo("%d: channel %s is no more already there, back for rescaling",
                    frame->GTimeS, r->output);
      r->alreadyThere = alreadyThere;}}

  /*-------------------------------------------------------error messages---*/
  if(present != r->present) {
    if(present == FR_NO) 
      CfgMsgAddError("%d: channel %s is missing, could not rescale it",
                      frame->GTimeS, r->input);
    else
      CfgMsgAddInfo("%d: channel %s is back for rescaling",
                    frame->GTimeS, r->input);
    r->present = present;}

  return;
}
/*---------------------------------------------------------------------------*/
void FdRescaleProcess(FdRescale *combine,
		     FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdRescaleProcessOne(combine, frame);

  next = combine->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
void FdRescaleParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_CONVERT_NTCT",
			 FdNTCTNew, (void *) &(fdIO->actionsIn), 9,
			 CfgString, CfgDec, CfgString, CfgReal,
			 CfgReal, CfgReal, CfgReal, CfgReal, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_CONVERT_NTCT",
			 FdNTCTNew, (void *) &(fdIO->actionsOut), 9,
			 CfgString, CfgDec, CfgString, CfgReal,
			 CfgReal, CfgReal, CfgReal, CfgReal, CfgString);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_RESCALE_CHANNEL",
			 FdRescaleNew, (void *) &(fdIO->actionsIn), 4,
			 CfgString, CfgReal, CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_RESCALE_CHANNEL",
			 FdRescaleNew, (void *) &(fdIO->actionsOut), 4,
			 CfgString, CfgReal, CfgString, CfgString);

  return;
}
