#include "FdIO.h"

/*-----------------------------------------------declare private functions---*/
typedef struct FdCombCh  FdCombCh;

struct FdCombCh{
  char*      output;      /* output channel name */
  double     scale1;      /* scaling factor for input 1 */
  char*      input1;      /* input channel name 1 */
  double     scale2;      /* scaling factor for input 2 */
  char*      input2;      /* input channel name 2   */
  int        errorNData;  /* tell if the error message has been printed */
  FdAction*  action;      /* action object which hold this recast object */
};
 
typedef struct FdCombFlag FdCombFlag;

struct FdCombFlag{
  char*      output;      /* output channel name */
  int        shift1;      /* shift for input 1 */
  char*      input1;      /* input channel name 1 */
  int        shift2;      /* shiftfor input 2 */
  char*      input2;      /* input channel name 2   */
  FdAction*  action;      /* action object which hold this recast object */
};
 
typedef struct FdBitOp  FdBitOp;

struct FdBitOp{
  char*        output;      /* output channel name */
  char*        input;       /* input channel name 1 */
  char*        input2;      /* input channel name 2 */
  char*        operator;    /* bitwise operator */
  int          outFreq;     /* output frequency or zero if unchanged */
  FRBOOL       error[5]  ;  /* tell if the error message has been printed */
  FdAction*    action;      /* action object which hold this recast object */
};

typedef struct FdFlagMinMax FdFlagMinMax;

struct FdFlagMinMax{
  char*        inputName;   /* input channel */
  char*        flagName;    /* name of the produced channel */
  int          sampleRate;  /* sampling rate of the flag channel */
  double       min;         /* minimum requested value */
  double       max;         /* maximum requested value */
  double       padding;     /* minimal requested time before switching the flag to 1*/
  FRBOOL       error;       /* tell if the error message has been printed */
  double       lastBadTime; /* time of the last bad case */
  FdAction*    action;      /* action object which hold this recast object */
};

typedef struct FdSegments FdSegments;

struct FdSegments{
  char*        fileName;    /* name of the segments file */
  char*        channelName; /* channel on which the segments are applied */
  FrSegList*   segList;     /* segment list */
  FRBOOL       error;       /* tell if the error message has been printed */
  FdAction*    action;      /* action object which hold this recast object */
};

typedef struct FdVetos FdVetos;

struct FdVetos{
  char*        fileName;    /* name of the vetos file */
  char*        channelName; /* channel on which the vetos are applied */
  FrSegList*   segList;     /* segment list */
  FRBOOL       error;       /* tell if the error message has been printed */
  FdAction*    action;      /* action object which hold this recast object */
};

void FdCombChProcess  (FdCombCh*   c, FrameH* frame);
void FdCombFlagProcess(FdCombFlag* c, FrameH* frame);
void FdBitOpProcess   (FdBitOp*    b, FrameH* frame);
void FdFlagMinMaxProcess(FdFlagMinMax *f, FrameH* frame);
void FdSegProcess     (FdSegments* s, FrameH* frame);
void FdVetoProcess    (FdVetos*    v, FrameH* frame);
 
/*---------------------------------------------------------------------------*/
int FdCombChNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdCombCh *combine;

  /*------------create the object and add it at the end of the linked list---*/
  combine = (FdCombCh*) calloc(1,sizeof(FdCombCh));
  if(combine == NULL) CfgMsgAddFatal("malloc FdCombCh failed");

  combine->action = FdActionNew((FdAction**) arg, 
			       (void*) combine, FdCombChProcess, "CombCh");
  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&combine->output, CfgParseGetNextString(data));
  combine->scale1 =          CfgParseGetNextReal(data);
  FrStrCpy(&combine->input1, CfgParseGetNextString(data));
  combine->scale2 =          CfgParseGetNextReal(data);
  FrStrCpy(&combine->input2, CfgParseGetNextString(data));

  CfgMsgAddInfo("New output channel %s = %g*%s + %g*%s",
		combine->output, 
		combine->scale1, combine->input1,
		combine->scale2, combine->input2);

 
  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdCombChProcessOne(FdCombCh* c,
			 FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrVect *input1, *input2, *output;
  double scale1, scale2;
  FRBOOL new1, new2;
  FrAdcData *adc1, *adc2;

  input1 = FrameFindVect(frame, c->input1);
  if(input1 == NULL) return;
  scale1 = c->scale1; 
  adc1 = FrAdcDataFind(frame, c->input1);
  if(adc1 != NULL) scale1 *= adc1->slope;

  input2 = FrameFindVect(frame, c->input2);
  if(input2 == NULL) return;
  scale2 = c->scale2; 
  adc2 = FrAdcDataFind(frame, c->input2);
  if(adc2 != NULL) scale2 *= adc2->slope;

  if(input1->nData != input2->nData) {
    if(c->errorNData == 0) 
      CfgMsgAddError("Could not combine %s and %s; nData missmatch: %"FRLLD"-%"FRLLD"",
		     c->input1, c->input2, input1->nData, input2->nData);
    c->errorNData = 1;
    return;}
   else if(c->errorNData != 0) {
    CfgMsgAddInfo("Could combine again %s and %s; nData: %"FRLLD"",
		     c->input1, c->input2, input1->nData);
    c->errorNData = 1;}

  /*---------output type is a double, unleass both input are 4 bytes float---*/
  new1 = FR_NO;
  new2 = FR_NO;
  if((input1->type != FR_VECT_4R) ||
     (input2->type != FR_VECT_4R)) {
    if(input1->type != FR_VECT_8R) {
      input1 = FrVectCopyToD(input1, 1., NULL);
      new1 = FR_YES;}
    if(input2->type != FR_VECT_8R) {
      input2 = FrVectCopyToD(input2, 1., NULL);
      new2 = FR_YES;}}

  if      (strcmp(c->input1, c->output) == 0) output = input1;
  else if (strcmp(c->input2, c->output) == 0) output = input2;
  else   {
    output = FrVectCopy(input1);
    FrVectSetName(output, c->output);
    FrProcDataNewVT(frame, output, 1);}

  FrvCombine2(scale1, input1, scale2, input2, output, NULL);

  if(new1 == FR_YES) FrVectFree(input1);
  if(new2 == FR_YES) FrVectFree(input2);
 
  return;
}
/*---------------------------------------------------------------------------*/
void FdCombChProcess(FdCombCh *combine,
		     FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdCombChProcessOne(combine, frame);

  next = combine->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
int FdCombFlagNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdCombFlag *c;

  /*------------create the object and add it at the end of the linked list---*/
  c = (FdCombFlag*) calloc(1,sizeof(FdCombFlag));
  if(c == NULL) CfgMsgAddFatal("malloc FdCombFlag failed");

  c->action = FdActionNew((FdAction**) arg, 
			       (void*) c, FdCombFlagProcess, "CombFlag");
  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&c->output, CfgParseGetNextString(data));
  FrStrCpy(&c->input1, CfgParseGetNextString(data));
  c->shift1 =           CfgParseGetNextDec(data);
  FrStrCpy(&c->input2, CfgParseGetNextString(data));
  c->shift2 =          CfgParseGetNextDec(data);

  if(c->shift1 >= 0 && c->shift2 >= 0) 
    CfgMsgAddInfo("New output channel %s = %s << %d + %s << %d",
		c->output, c->input1, c->shift1, c->input2, c->shift2);
  else if(c->shift1 >= 0 && c->shift2 < 0) 
    CfgMsgAddInfo("New output channel %s = %s << %d + %s >> %d",
		c->output, c->input1, c->shift1, c->input2, -c->shift2);
  else if(c->shift1 < 0 && c->shift2 >= 0) 
    CfgMsgAddInfo("New output channel %s = %s >> %d + %s << %d",
		c->output, c->input1, -c->shift1, c->input2, c->shift2);
  else if(c->shift1 < 0 && c->shift2 < 0) 
    CfgMsgAddInfo("New output channel %s = %s >> %d + %s >> %d",
		c->output, c->input1, -c->shift1, c->input2, -c->shift2);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdCombFlagProcessOne(FdCombFlag* c,
			 FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrVect *input1, *input2, *output;
  int ratio, i, j, k, value1, value2;

  /*-------------------------------get data, check type and relative rate---*/
  input1 = FrameFindVect(frame, c->input1);
  if(input1 == NULL) return;
  if(input1->type != FR_VECT_4U && input1->type != FR_VECT_4S) 
    CfgMsgAddFatal("FdCombineFlag: %s is not a flag", c->input1);

  input2 = FrameFindVect(frame, c->input2);
  if(input2 == NULL) return;
  if(input2->type != FR_VECT_4U && input2->type != FR_VECT_4S) 
    CfgMsgAddFatal("FdCombineFlag: %s is not a flag", c->input2);

  if(input1->nData < input2->nData) 
    CfgMsgAddFatal("FdCombineFlag: %s is slower than %s", c->input1, c->input2);
     
  ratio = input1->nData/input2->nData;
  if(ratio*input2->nData != input1->nData)
    CfgMsgAddFatal("FdCombineFlag: rate of %s is not a multiple of %s", 
          c->input1, c->input2);

  /*--------------------------prepare output vector which could be input1---*/
  if(strcmp(c->output, c->input1) == 0) {
    output = input1;}
  else {
    output = FrVectCopy(input1);
    if(output == NULL) CfgMsgAddFatal("FdCombineFlag malloc failed");
    FrVectSetName(output, c->output);
    FrProcDataNewVT(frame, output, 1);}
 
  /*--------------------------------------------------------combine data---*/ 
  k = 0;
  for(i=0; i<input2->nData; i++) {
    for(j=0; j<ratio; j++) {
      k++;
      if(c->shift1 > 0) value1 = input1->dataUI[k] <<  c->shift1;
      else              value1 = input1->dataUI[k] >> -c->shift1;
      if(c->shift2 > 0) value2 = input2->dataUI[i] <<  c->shift2;
      else              value2 = input2->dataUI[i] >> -c->shift2;
      output->dataUI[k] = value1 + value2;}}
 
  return;
}
/*---------------------------------------------------------------------------*/
void FdCombFlagProcess(FdCombFlag *combine,
		     FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdCombFlagProcessOne(combine, frame);

  next = combine->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
int FdBitOpNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdBitOp *bitop;
  int i;

  /*------------create the object and add it at the end of the linked list---*/
  bitop = (FdBitOp*) calloc(1,sizeof(FdBitOp));
  if(bitop == NULL) CfgMsgAddFatal("malloc FdBitOp failed");

  bitop->action = FdActionNew((FdAction**) arg, 
			       (void*) bitop, FdBitOpProcess, "BitOp");
  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&bitop->output,  CfgParseGetNextString(data));
  FrStrCpy(&bitop->input,   CfgParseGetNextString(data));
  FrStrCpy(&bitop->operator,CfgParseGetNextString(data));
  FrStrCpy(&bitop->input2,  CfgParseGetNextString(data));
//  bitop->mask = CfgParseGetNextHex(data);
  bitop->outFreq = CfgParseGetNextDec(data);

  if ((strcmp(bitop->operator,"NOT")  == 0) &&
      (strcmp(bitop->operator,"AND")  == 0) &&
      (strcmp(bitop->operator,"OR" )  == 0) &&
      (strcmp(bitop->operator,"XOR")  == 0) &&
      (strcmp(bitop->operator,"MASK") == 0) && 
      (strcmp(bitop->operator,"<<" )  == 0) && 
      (strcmp(bitop->operator,">>" )  == 0))
    CfgMsgAddFatal("%s is not a valid LOGICAL_OPERATOR", bitop->operator);

  if(bitop->input2 == NULL) 
     CfgMsgAddFatal("FdBitOp: last parameter for %s is missing",bitop->output);

  if(bitop->outFreq == 0)
    CfgMsgAddInfo("New output channel %s = %s %s %s",
		bitop->output, bitop->input, bitop->operator, bitop->input2);
  else
    CfgMsgAddInfo("New output channel %s = %s %s %s output frequency: %d",
		bitop->output, bitop->input, bitop->operator, bitop->input2,
		bitop->outFreq);

  for(i= 0; i< 5; i++) {bitop->error[i] = FR_NO;}

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdBitOpProcessOne(FdBitOp* b, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrVect *input, *input2, *output, *extended;
  int i, nData, nTimes;
  unsigned int *in, *in2, *out, mask;
  FrProcData *proc;

/*-------------------------------------------------------get input vector---*/
  input = FrameFindVect(frame, b->input);
  if(input == NULL) {
    if(b->error[0] == FR_NO)
	CfgMsgAddError("FdBitOp: at %d missing channel %s",
                               frame->GTimeS, b->input);
    b->error[0] = FR_YES;
    return;}
  if(b->error[0] == FR_YES) {
    CfgMsgAddInfo("FdBitOp: %s is back at %d",b->input, frame->GTimeS);
    b->error[0] = FR_NO;}

  if(input->type != FR_VECT_4U &&
     input->type != FR_VECT_4S) {
    if(b->error[1] == FR_NO) CfgMsgAddError(
           "FdBitOp: channel %s is not of the right type (%d)",
                    b->input, input->type);
    b->error[1] = FR_YES;
    return;}
  b->error[1] = FR_NO;

  /*----------------------------------get second input vector or mask value---*/
  if(isdigit(b->input2[0])) {
    sscanf(b->input2, "%x",&mask);
    input2 = NULL;}
  else {
    input2 = FrameFindVect(frame, b->input2);
    if(input2 == NULL) {
      if(b->error[2] == FR_NO)  
         CfgMsgAddError("FdBitOp: at %d missing channel %s",
                               frame->GTimeS, b->input2);
      b->error[2] = FR_YES;
      return;}
    if(b->error[2] == FR_YES) {
      CfgMsgAddInfo("FdBitOp: %s is back at %d",b->input2, frame->GTimeS);
      b->error[2] = FR_NO;}

    if(input2->type != FR_VECT_4U &&
       input2->type != FR_VECT_4S) {
      if(b->error[3] == FR_NO)
	  CfgMsgAddError("FdBitOp: channel %s is not of the right type (%d)",
                    b->input2, input2->type);
      b->error[3] = FR_YES;
      return;}
    b->error[3] = FR_NO;}

  /*--------------------------------------------------get output vector---*/
  if(strcmp(b->output, b->input) != 0) {
    if(FrameFindVect(frame, b->output) != NULL) {
      if(b->error[4] == FR_NO)
         CfgMsgAddError("FdBitOp: channel %s already exist",b->output);
      b->error[4] = FR_YES;
      return;}
    b->error[4] = FR_NO;

    output = FrVectCopy(input);
    if(output == NULL) return;}

  else {
    output = input;}

  /*------------------------------------------------------------apply logic---*/
  nData = output->nData;
  in  = input ->dataUI;
  out = output->dataUI;

  if(input2 == NULL) {
    if      (b->operator[0]=='N') for(i=0; i<nData; i++) {out[i] =(~in[i])& mask;}
    else if (b->operator[0]=='A') for(i=0; i<nData; i++) {out[i] =  in[i] & mask;}
    else if (b->operator[0]=='O') for(i=0; i<nData; i++) {out[i] =  in[i] | mask;}
    else if (b->operator[0]=='X') for(i=0; i<nData; i++) {out[i] =  in[i] ^ mask;}
    else if (b->operator[0]=='<') for(i=0; i<nData; i++) {out[i] =  in[i] << mask;}
    else if (b->operator[0]=='>') for(i=0; i<nData; i++) {out[i] =  in[i] >> mask;}
    else if (b->operator[0]=='M') for(i=0; i<nData; i++) {out[i] = (in[i] & mask)==mask;}
   }
  else {
    in2 = input2->dataUI;
    if      (b->operator[0]=='N') for(i=0; i<nData; i++) {out[i] = ~in[i];}
    else if (b->operator[0]=='A') for(i=0; i<nData; i++) {out[i] =  in[i] & in2[i];}
    else if (b->operator[0]=='O') for(i=0; i<nData; i++) {out[i] =  in[i] | in2[i];}
    else if (b->operator[0]=='X') for(i=0; i<nData; i++) {out[i] =  in[i] ^ in2[i];}
    else if (b->operator[0]=='<') for(i=0; i<nData; i++) {out[i] =  in[i] <<in2[i];}
    else if (b->operator[0]=='>') for(i=0; i<nData; i++) {out[i] =  in[i] >>in2[i];}
    else if (b->operator[0]=='M') for(i=0; i<nData; i++) {out[i] = (in[i] & in2[i])==in2[i];}
  }

  /*-----------------extend output vector if requested (if output != input)---*/
  if(output == input) return;

  nTimes = b->outFreq * output->dx[0];
  if(nTimes > 1) {
    extended = FrVectExtend(output, nTimes, NULL, NULL);
    FrVectFree(output);
    output = extended;}

  /*---------------------------------------------attach result to the frame---*/
  FrVectSetName(output, b->output);
  proc = FrProcDataNewVT(frame, output, 1);
  if(proc != NULL) {
    proc->comment = malloc(strlen(output->name) + 50 + strlen(b->input2));
    if(proc->comment != NULL) sprintf(proc->comment,"%s %s %s",
           output->name, b->operator, b->input2);}

  return;
}
/*---------------------------------------------------------------------------*/
void FdBitOpProcess(FdBitOp *bitop, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdBitOpProcessOne(bitop, frame);

  next = bitop->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
int FdFlagMinMaxNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdFlagMinMax *flag;

  /*------------create the object and add it at the end of the linked list---*/
  flag = (FdFlagMinMax*) calloc(1,sizeof(FdFlagMinMax));
  if(flag == NULL) CfgMsgAddFatal("malloc FdFlagMinMax failed");

  flag->action = FdActionNew((FdAction**) arg, 
	     (void*) flag, FdFlagMinMaxProcess, "FlagMinMax");
  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&flag->flagName,  CfgParseGetNextString(data));
  flag->sampleRate =         CfgParseGetNextDec(data);
  FrStrCpy(&flag->inputName, CfgParseGetNextString(data));
  flag->min =                CfgParseGetNextReal(data);
  flag->max =                CfgParseGetNextReal(data);
  flag->padding =            CfgParseGetNextReal(data);

  CfgMsgAddInfo("New flag channel %s (%d Hz) set to 1 if %s < %g or > %g padding = %g",
		flag->flagName, flag->sampleRate, flag->inputName,
                flag->min,      flag->max,        flag->padding);

  flag->lastBadTime = 0;

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdFlagMinMaxProcessOne(FdFlagMinMax* f,
			 FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrVect *input;
  FrAdcData *flag;
  int nData, i, j, jMin, jMax;
  double value;

  nData = frame->dt * f->sampleRate;
  flag = FrAdcDataNew(frame, f->flagName, f->sampleRate, nData, 32);
  for(i=0; i<nData; i++) {flag->data->dataI[i] = 2;}

  input = FrameFindVect(frame, f->inputName);
  if(input == NULL) {
    if(f->error == FR_NO) { 
      CfgMsgAddError("FdFlagMinMax: %s start to be missing at %d", 
            f->inputName, frame->GTimeS);
      f->error = FR_YES;}
    return;}

  if(f->error == FR_YES) { 
    CfgMsgAddInfo("FdFlagMinMax: %s is back at %d", 
            f->inputName, frame->GTimeS);
     f->error = FR_NO;}

  jMin = 0;
  for(i=0; i<nData; i++) {
    flag->data->dataI[i] = 0;
    jMax = (input->nData*(i+1))/nData;
    for(j=jMin; j<jMax; j++) {
      value = FrVectGetValueI(input, j);
      if(value > f->min && value < f->max) continue;
      flag->data->dataI[i] = 1;
      f->lastBadTime = input->GTime + input->dx[0]*j;}
    if(input->GTime + i*flag->data->dx[0] < f->lastBadTime + f->padding) 
			flag->data->dataI[i] = 1;
    jMin = jMax;}

  return;
}
/*---------------------------------------------------------------------------*/
void FdFlagMinMaxProcess(FdFlagMinMax *flag, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdFlagMinMaxProcessOne(flag, frame);

  next = flag->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
int FdSegNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdSegments *s;

  /*------------create the object and add it at the end of the linked list---*/
  s = (FdSegments*) calloc(1,sizeof(FdSegments));
  if(s == NULL) CfgMsgAddFatal("malloc FdSegments failed");

  s->action = FdActionNew((FdAction**) arg, 
			       (void*) s, FdSegProcess, "Segments");
  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&s->channelName, CfgParseGetNextString(data));
  FrStrCpy(&s->fileName,    CfgParseGetNextString(data));

  CfgMsgAddInfo("Read segments for %s from %s", s->channelName, s->fileName);

  s->error   = FR_NO;
  s->segList = FrSegListRead(s->fileName);
  if(s->segList == NULL) CfgMsgAddFatal("Could not read segments file");

  FrSegListDump(s->segList, stdout, 1);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdSegProcessOne(FdSegments *s, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrVect *input;
  double gtime0, gtime, dt;
  int i;

/*-------------------------------------------------------get input vector---*/
  input = FrameFindVect(frame, s->channelName);
  if(input == NULL) {
    if(s->error == FR_NO) CfgMsgAddError(
              "%d: %s is missing", frame->GTimeS, s->channelName);
    s->error = FR_YES;
    return;}
  
  if(s->error == FR_YES) {
    CfgMsgAddInfo("%d: %s is back",frame->GTimeS, s->channelName);
    s->error = FR_NO;}

  if(input->type != FR_VECT_4U &&
     input->type != FR_VECT_4S) 
    CfgMsgAddFatal("Channel: %s is not of the right type (%d)",
                     s->channelName, input->type);

 /*---------------apply segment:reject (set to 0) if NOT in a segment time---*/
  dt = input->dx[0];
  gtime0 = frame->GTimeS + 1.e-9*frame->GTimeN;

  for(i=0; i<input->nData; i++) {
    gtime = gtime0 + i*dt;
    if(FrSegListFind(s->segList, gtime) < 0) input->dataI[i] = 0;}
 
  return;
}
/*---------------------------------------------------------------------------*/
void FdSegProcess(FdSegments *seg, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdSegProcessOne(seg, frame);

  next = seg->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
int FdVetoNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdVetos *v;

  /*------------create the object and add it at the end of the linked list---*/
  v = (FdVetos*) calloc(1,sizeof(FdVetos));
  if(v == NULL) CfgMsgAddFatal("malloc FdVetos failed");

  v->action = FdActionNew((FdAction**) arg, 
			       (void*) v, FdVetoProcess, "Vetos");
  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&v->channelName, CfgParseGetNextString(data));
  FrStrCpy(&v->fileName,    CfgParseGetNextString(data));

  CfgMsgAddInfo("Read vetos for %s from %s", v->channelName, v->fileName);

  v->error   = FR_NO;
  v->segList = FrSegListRead(v->fileName);
  if(v->segList == NULL) CfgMsgAddFatal("Could not read vetos file");

  FrSegListDump(v->segList, stdout, 1);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdVetoProcessOne(FdVetos *v, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrVect *input;
  double gtime0, gtime, dt;
  int i;

/*-------------------------------------------------------get input vector---*/
  input = FrameFindVect(frame, v->channelName);
  if(input == NULL) {
    if(v->error == FR_NO) CfgMsgAddError("%d: %s is missing for veto", 
           frame->GTimeS, v->channelName);
    v->error = FR_YES;
    return;}

  if(v->error == FR_YES) {
    CfgMsgAddInfo("%d: %s is back",frame->GTimeS, v->channelName);
    v->error = FR_NO;}

  if(input->type != FR_VECT_4U &&
     input->type != FR_VECT_4S)
    CfgMsgAddFatal("Veto: %s is not of the right type (%d)",
                     v->channelName, input->type);

 /*------------------apply veto:reject (set to 0) if inside a segment time---*/
  dt = input->dx[0];
  gtime0 = frame->GTimeS + 1.e-9*frame->GTimeN;

  for(i=0; i<input->nData; i++) {
    gtime = gtime0 + i*dt;
    if(FrSegListFind(v->segList, gtime) >= 0) input->dataI[i] = 0;}
 
  return;
}
/*---------------------------------------------------------------------------*/
void FdVetoProcess(FdVetos *veto, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdVetoProcessOne(veto, frame);

  next = veto->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
void FdCombineParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
 
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_COMBINE_CHANNELS",
			 FdCombChNew, (void *) &(fdIO->actionsIn), 5,
			 CfgString, CfgReal, CfgString, CfgReal, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_COMBINE_CHANNELS",
			 FdCombChNew, (void *) &(fdIO->actionsOut), 5,
			 CfgString, CfgReal, CfgString, CfgReal, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_COMBINE_FLAGS",
			 FdCombFlagNew, (void *) &(fdIO->actionsIn), 5,
			 CfgString, CfgString, CfgDec, CfgString, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_COMBINE_FLAGS",
			 FdCombFlagNew, (void *) &(fdIO->actionsOut), 5,
			 CfgString, CfgString, CfgDec, CfgString, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_BITOP",
			 FdBitOpNew, (void *) &(fdIO->actionsIn), 5,
			 CfgString, CfgString, CfgString, CfgString, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_BITOP",
			 FdBitOpNew, (void *) &(fdIO->actionsOut), 5,
			 CfgString, CfgString, CfgString, CfgString, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_FLAG_MIN_MAX",
			 FdFlagMinMaxNew, (void *) &(fdIO->actionsIn), 6,
			 CfgString, CfgDec, CfgString, CfgReal, CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_SEGMENTS",
			 FdSegNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_SEGMENTS",
			 FdSegNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_VETOS",
			 FdVetoNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_VETOS",
			 FdVetoNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgString);

  return;
}
