#include "FdIO.h"
#include "Frv.h"

/*-----------------------------------------------declare private functions---*/
typedef struct FdDemod      FdDemod;
typedef struct FdFShift     FdFShift;
typedef struct FdGetModulus FdGetModulus;
typedef struct FdGetPhase   FdGetPhase;

struct FdDemod{
  char*       outP;        /* name of the ACp output channel */
  char*       outQ;        /* name of the ACq output channel */
  char*       input;       /* input channel name */
  double      fShift;      /* demodulation frequency */
  char*       phase;       /* phase shift at GPS=0 */
  int         fOut;        /* sampling frequency of the output signals */
  double      f3dB;        /* 3dB frequency cut of the output low pass filter */
  int         order;       /* order of the output low pass filter */
  FRBOOL      inputStatus; /* flag telling if the input signal is available*/
  FRBOOL      phiStatus;   /* flag telling if the phase signal is available*/
  FrvFilterB* filterP;     /* output filter for ACp*/
  FrvFilterB* filterQ;     /* output filter for ACq*/
  double      tStart;      /* GPS time of the first processed data */
  double      phiStart;    /* phase offset at tStart */
  FdAction*   action;      /* action object which hold this demod object */
};

struct FdFShift{
  char*     outP;       /* name of the ACp output channel */
  char*     outQ;       /* name of the ACq output channel */
  char*     inP;        /* name of the ACp input channel */
  char*     inQ;        /* name of the ACq ouput channel */
  double    fShift;     /* frequency shift */
  char*     phase;      /* phase shift at GPS=0 */
  FRBOOL    inPStatus;  /* flag telling if the input signal is available*/
  FRBOOL    inQStatus;  /* flag telling if the input signal is available*/
  FRBOOL    phiStatus;  /* flag telling if the phase signal is available*/
  double    tStart;     /* GPS time of the first processed data */
  double    phiStart;   /* phase offset at tStart */
  FdAction* action;     /* action object which hold this demod object */
};

struct FdGetModulus{
  char*     output;     /* name of the output channel */
  char*     inP;        /* name of the ACp input channel */
  char*     inQ;        /* name of the ACq ouput channel */
  FRBOOL    inPStatus;  /* flag telling if the input signal is available*/
  FRBOOL    inQStatus;  /* flag telling if the input signal is available*/
  FdAction* action;     /* action object which hold this demod object */
};

struct FdGetPhase{
  char*     output;     /* name of the output channel */
  char*     inP;        /* name of the ACp input channel */
  char*     inQ;        /* name of the ACq ouput channel */
  FRBOOL    inPStatus;  /* flag telling if the input signal is available*/
  FRBOOL    inQStatus;  /* flag telling if the input signal is available*/
  double    rawPhase;   /* last computed phase value (within -pi; +pi) */
  double    phase;      /* total number of two pi */   
  FdAction* action;     /* action object which hold this demod object */
};

/*---------------------------------------------------------------------------*/
void FdDemodProcessOne(FdDemod* d,
		       FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *adcIn, *adcP, *adcQ;
  FrVect *input, *phiVect;
  double p, phi0, nCycles, ip, tNowInt, tNowRes, dPhi, freqRes, phiStart, phi;
  long nGroup, freqInt, i, fIn;

  /*----------------------------------------------first get input channels---*/
  adcIn = FrAdcDataFind(frame, d->input);
  if(adcIn != NULL) input = adcIn->data;
  else              input = FrameFindProcVect(frame, d->input);
  if(input == NULL) {
    if(d->inputStatus == FR_YES) {
      CfgMsgAddError("FdDemod: %d: channel %s start to be missing",
		     frame->GTimeS,d->input);
      d->inputStatus = FR_NO;}
    return;}
  else if(d->inputStatus != FR_YES) {
    CfgMsgAddInfo("FdDemod: %d: channel %s is back",frame->GTimeS, d->input);
    d->inputStatus = FR_YES;}

  /*-------------------------------------------create the output channels---*/
  fIn = .5+1./input->dx[0];
  if(d->fOut > fIn) {
    d->fOut = fIn;
    CfgMsgAddError("FdDemod: The sampling rate of %s/q will be limited to %d Hz",
      d->outP, d->fOut);}

  adcP = FrAdcDataNew(frame, d->outP, fIn, input->nData, -64);
  adcQ = FrAdcDataNew(frame, d->outQ, fIn, input->nData, -64);
  if(adcQ  == NULL) CfgMsgAddFatal("FdDemodProc malloc %s failed at %d",
				   d->outQ, frame->GTimeS);
  adcP->fShift = d->fShift;
  adcQ->fShift = d->fShift;
  adcP->comment = malloc(strlen(d->input)+128);
  adcQ->comment = malloc(strlen(d->input)+128);
  if(adcQ->comment == NULL) 
    CfgMsgAddFatal("FdDemodProc malloc comment failed at %d",frame->GTimeS);
  sprintf(adcP->comment, "produced by FdDemod from %s", d->input);
  sprintf(adcQ->comment, "produced by FdDemod from %s", d->input);
  FrVectSetUnitY(adcP->data, input->unitY);
  FrVectSetUnitY(adcQ->data, input->unitY);

  if(adcIn != NULL) {
    adcP->slope = adcIn->slope;
    adcQ->slope = adcIn->slope;
    FrStrCpy(&adcP->units, adcIn->units);
    FrStrCpy(&adcQ->units, adcIn->units);}

  /*------------------------get phase: could be a vector or just a number---*/
  if(isdigit(d->phase[0]) || d->phase[0] == '-') {
    phi0 = atof(d->phase);
    phiVect = NULL;
    adcP->phase  = phi0;
    adcQ->phase  = phi0;}
  else {
    phi0 = 0;
    phiVect = FrameFindVect(frame, d->phase);
    if(phiVect == NULL) {
      CfgMsgAddError("FdDemod: missing phase channel: %s; set to zero",
        d->phase);}
    else if(phiVect->nData != input->nData) {
      CfgMsgAddFatal("FdDemod: input rates missmatch for %s and %s",
         d->input, d->phase);}}

  /*-----------------------------get the phase at the start of the vector---*/
  /* To avoid numerical problems, we split the modulation frequency in an 
     integer part, which cancel at each new second and a residual part------*/
  freqInt = (long) d->fShift;
  freqRes = d->fShift - freqInt;

  if(d->tStart == 0) {
    d->tStart = frame->GTimeS;  /*---we keep only the integer part---*/
    nCycles = freqRes * frame->GTimeS;
    d->phiStart = FRTWOPI * modf(nCycles, &ip) + phi0;}

  tNowInt = frame->GTimeS - d->tStart;
  tNowRes = 1.e-9*frame->GTimeN;
  nCycles = freqInt * tNowRes + freqRes * (tNowInt + tNowRes);
  phiStart =  FRTWOPI * modf(nCycles, &ip) + d->phiStart + phi0;

  /*--------------------------------------------- fill the output vectors---*/
  dPhi = FRTWOPI * input->dx[0] * d->fShift;
  for(i=0; i<input->nData; i++) {
    phi = phiStart + i * dPhi;
    if(phiVect != NULL) phi += FrVectGetValueI(phiVect, i);
    p = 2. * FrVectGetValueI(input, i);
    adcP->data->dataD[i] = cos(phi) * p;
    adcQ->data->dataD[i] = sin(phi) * p;}

  /*------------------output low pass filter----*/
  if(d->order > 0) {
    FrvFilterButProc(d->filterP, adcP->data);
    FrvFilterButProc(d->filterQ, adcQ->data);}

  nGroup  = input->nData/(d->fOut*frame->dt);
  FrAdcDataDecimate(adcP, -nGroup);
  FrAdcDataDecimate(adcQ, -nGroup);

  return;
}
/*---------------------------------------------------------------------------*/
void FdDemodProcess(FdDemod *demod,
		    FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdDemodProcessOne(demod, frame);

  next = demod->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
int FdDemodNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdDemod *demod;
  char *outName;

  /*------------create the object and add it at the end of the linked list---*/
  demod = (FdDemod*) calloc(1,sizeof(FdDemod));
  if(demod == NULL) CfgMsgAddFatal("malloc FdDemod failed");

  demod->action = FdActionNew((FdAction**) arg, 
			      (void*) demod, FdDemodProcess, "Demod");
  /*------------------------------------------------------------ fill info---*/
  outName = CfgParseGetNextString(data);
  FrStrCpy(&demod->input,  CfgParseGetNextString(data));
  demod->fShift  =         CfgParseGetNextReal(data);
  FrStrCpy(&demod->phase,  CfgParseGetNextString(data));
  demod->fOut  =           CfgParseGetNextDec(data);
  demod->f3dB  =           CfgParseGetNextReal(data);
  demod->order =           CfgParseGetNextDec(data);

  CfgMsgAddInfo("Demodulation: output=%sp/q input=%s fDemod=%g phase=%s",
		outName, demod->input, demod->fShift, demod->phase);
  if(isalpha(demod->phase[0])) CfgMsgAddInfo(" phase is from a live vector");
  CfgMsgAddInfo(" output sampling rate:%d LP filter cut:%gHz order=%d",
		demod->fOut, demod->f3dB, demod->order);

  demod->outP = malloc(strlen(outName)+5);
  demod->outQ = malloc(strlen(outName)+5);
  sprintf(demod->outP,"%sp", outName);
  sprintf(demod->outQ,"%sq", outName);
  demod->inputStatus = FR_YES;
  demod->tStart   = 0;
  demod->phiStart = 0;

  if(demod->order > 0) {
    demod->filterP = FrvFilterButNew(demod->order, 0, demod->f3dB, NULL);
    demod->filterQ = FrvFilterButNew(demod->order, 0, demod->f3dB, NULL);}
 
  /*------------- is the phase offset a static value or from a signal?---*/
  if(!isdigit(demod->phase[0]) && demod->phase[0] != '-') 
   CfgMsgAddInfo(" phase is from a live channel");

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdFShiftProcessOne(FdFShift *fs,
			FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *inP, *inQ, *outP, *outQ;
  FrVect  *phiVect;
  double p, q, phi0, nCycles, ip, tNowInt, tNowRes, dPhi, freqRes, phiStart, phi;
  long nData, freqInt, i;

  /*----------------------------------------------first get input channels---*/
  inP = FrAdcDataFind(frame, fs->inP);
  if(inP == NULL) {
    if(fs->inPStatus == FR_YES) {
      CfgMsgAddError("FdFShift: %d: ADC channel %s start to be missing",
		     frame->GTimeS,fs->inP);
      fs->inPStatus = FR_NO;}
    return;}
  else if(fs->inPStatus != FR_YES) {
    CfgMsgAddInfo("FdFShift: %d: ADC %s is back",frame->GTimeS, fs->inP);
    fs->inPStatus = FR_YES;}

  inQ = FrAdcDataFind(frame, fs->inQ);
  if(inQ == NULL) {
    if(fs->inQStatus == FR_YES) {
      CfgMsgAddError("FdFShift: %d: ADC channel %s start to be missing",
		     frame->GTimeS,fs->inQ);
      fs->inQStatus = FR_NO;}
    return;}
  else if(fs->inQStatus != FR_YES) {
    CfgMsgAddInfo("FdFShift: %d: ADC %s is back",frame->GTimeS, fs->inQ);
    fs->inQStatus = FR_YES;}

  /*---------------------------------------------------check sampling rate---*/
  if(inP->data->nData != inQ->data->nData) 
    CfgMsgAddFatal("FdFShift: channel length missmatch: %"FRLLD"!=%"FRLLD"",
		   inP->data->nData, inQ->data->nData);
    
  /*------------ get the phase channel and check if it size is compatible---*/
  if(isdigit(fs->phase[0]) || fs->phase[0] == '-') {
    phi0 = atof(fs->phase);
    phiVect = NULL;}
  else {
    phi0 = 0;
    phiVect = FrameFindVect(frame, fs->phase);
    if(phiVect != NULL) {
      if(phiVect->nData != inP->data->nData) {
	CfgMsgAddFatal("FShift: phase vector %s missmatch (%"FRLLD"!=%"FRLLD")",
		       fs->phase,phiVect->nData,inP->data->nData);}
      if(fs->phiStatus != FR_YES) {
	CfgMsgAddInfo("FShift: %d: phase channel %s is back",
		      frame->GTimeS, fs->phase);
	fs->phiStatus = FR_YES;}}
    else {
      if(fs->phiStatus == FR_YES) {
	CfgMsgAddError("FShift: %d: channel %s start to be missing",
		       frame->GTimeS, fs->phase);
	fs->phiStatus = FR_NO;}
      return;}}

  /*-----------------------------------------create the output channel(s)---*/
  nData = inP->data->nData;
  outP = FrAdcDataNew(frame, fs->outP, inP->sampleRate, nData, -64);
  outQ = FrAdcDataNew(frame, fs->outQ, inQ->sampleRate, nData, -64);
  if(outQ  == NULL) CfgMsgAddFatal("FdFShiftProc malloc %s failed at %d",
				   fs->outQ, frame->GTimeS);
  outP->fShift = inQ->fShift + phi0;
  outQ->fShift = inP->fShift + phi0;
  outP->comment = malloc(strlen(fs->inP)+128);
  outQ->comment = malloc(strlen(fs->inQ)+128);
  if(outQ->comment == NULL) 
    CfgMsgAddFatal("FdFShiftProc malloc coment failed at %d",frame->GTimeS);
  sprintf(outP->comment, "produced by FdFShift from %s", fs->inP);
  sprintf(outQ->comment, "produced by FdFShift from %s", fs->inQ);
  FrVectSetUnitY(outP->data, inP->data->unitY);
  FrVectSetUnitY(outQ->data, inQ->data->unitY);

  outP->slope = inP->slope;
  outQ->slope = inQ->slope;
  FrStrCpy(&outP->units, inP->units);
  FrStrCpy(&outQ->units, inQ->units);

  /*-----------------------------get the phase at the start of the vector---*/
  /* To avoid numerical problems, we split the modulation frequency in an 
     integer part, which cancel at each new second and a residual part------*/
  freqInt = (long) fs->fShift;
  freqRes = fs->fShift - freqInt;

  if(fs->tStart == 0) {
    fs->tStart = frame->GTimeS;  /*---we keep only the integer part---*/
    nCycles = freqRes * frame->GTimeS;
    fs->phiStart = FRTWOPI * modf(nCycles, &ip) + phi0;}

  tNowInt = frame->GTimeS - fs->tStart;
  tNowRes = 1.e-9*frame->GTimeN;
  nCycles = freqInt * tNowRes + freqRes * (tNowInt + tNowRes);
  phiStart =  FRTWOPI * modf(nCycles, &ip) + fs->phiStart + phi0;

  /*--------------------------------------------- fill the output vectors---*/
  dPhi = FRTWOPI * inP->data->dx[0] * fs->fShift;
  for(i=0; i<inP->data->nData; i++) {
    phi = phiStart + i * dPhi;
    if(phiVect != NULL) phi += FrVectGetValueI(phiVect, i);
    p = FrVectGetValueI(inP->data, i);
    q = FrVectGetValueI(inQ->data, i);
    outP->data->dataD[i] = cos(phi) * p + sin(phi) * q;
    outQ->data->dataD[i] = sin(phi) * p + cos(phi) * q;}

  return;
}
/*---------------------------------------------------------------------------*/
void FdFShiftProcess(FdFShift *fs,
		     FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdFShiftProcessOne(fs, frame);

  next = fs->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdFShiftNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdFShift *fs;
  char *input, *output;

  fs = (FdFShift*) calloc(1,sizeof(FdFShift));
  if(fs == NULL) CfgMsgAddFatal("malloc FdFShift failed");

  fs->action = FdActionNew((FdAction**) arg, 
			   (void*) fs, FdFShiftProcess, "FdFShift");

  output =  CfgParseGetNextString(data);
  input  =  CfgParseGetNextString(data);
  fs->fShift =           CfgParseGetNextReal(data);
  FrStrCpy(&(fs->phase), CfgParseGetNextString(data));

  fs->inP  = malloc(strlen(input)+5);
  fs->inQ  = malloc(strlen(input)+5);
  fs->outP = malloc(strlen(output)+5);
  fs->outQ = malloc(strlen(output)+5);
  sprintf(fs->inP,  "%sp",input);
  sprintf(fs->inQ,  "%sq",input);
  sprintf(fs->outP, "%sp",output);
  sprintf(fs->outQ, "%sq",output);
  fs->inPStatus = FR_YES;
  fs->inQStatus = FR_YES;
  fs->phiStatus = FR_YES;
  fs->tStart = 0;

  CfgMsgAddInfo("Add frequency shift:  new channel: %s %s from %s %s "
		" with fShift=%gHz phi=%s",
		fs->outP, fs->outQ, fs->inP, fs->inQ, fs->fShift, fs->phase);

  /*------------- is the phase offset a static value or from a signal?---*/
  if(!isdigit(fs->phase[0]) && fs->phase[0] != '-') 
   CfgMsgAddInfo(" phase is from a live channel");

  return(CFG_OK);
} 
/*---------------------------------------------------------------------------*/
void FdGetModulusProcessOne(FdGetModulus *mod,
			    FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *inP, *inQ;
  FrProcData *out;
  double p, q, slope, bias;
  long nData, i;

  /*----------------------------------------------first get input channels---*/
  inP = FrAdcDataFind(frame, mod->inP);
  if(inP == NULL) {
    if(mod->inPStatus == FR_YES) {
      CfgMsgAddError("FdGetModulus: %d: ADC channel %s start to be missing",
		     frame->GTimeS, mod->inP);
      mod->inPStatus = FR_NO;}
    return;}
  else if(mod->inPStatus != FR_YES) {
    CfgMsgAddInfo("FdGetModulus: %d: ADC %s is back",frame->GTimeS, mod->inP);
    mod->inPStatus = FR_YES;}

  inQ = FrAdcDataFind(frame, mod->inQ);
  if(inQ == NULL) {
    if(mod->inQStatus == FR_YES) {
      CfgMsgAddError("FdGetModulus: %d: ADC channel %s start to be missing",
		     frame->GTimeS,mod->inQ);
      mod->inQStatus = FR_NO;}
    return;}
  else if(mod->inQStatus != FR_YES) {
    CfgMsgAddInfo("FdGetModulus: %d: ADC %s is back",frame->GTimeS, mod->inQ);
    mod->inQStatus = FR_YES;}

  /*---------------------------------------------------check sampling rate---*/
  if(inP->data->nData != inQ->data->nData) 
    CfgMsgAddFatal("FdGetModulus: channel length missmatch: %"FRLLD"!=%"FRLLD"",
		   inP->data->nData, inQ->data->nData);
    
  /*--------------------------------------------create the output channel---*/
  nData = inP->data->nData;
  out = FrProcDataNew(frame, mod->output, inP->sampleRate, nData, -64);
  if(out  == NULL) CfgMsgAddFatal("FdGetModulus Proc malloc %s failed at %d",
				  mod->output, frame->GTimeS);
  out->comment = malloc(strlen(mod->inQ)+128);
  if(out->comment == NULL) 
    CfgMsgAddFatal("FdGetModulusProc malloc comment failed at %d",frame->GTimeS);
  sprintf(out->comment, "produced by FdGetModulus from %sp/q", mod->inP);
  FrVectSetUnitY(out->data, inP->data->unitY);
  out->type = 1;

  slope = inP->slope;
  bias  = inP->bias;

  /*---------------------------------------------- fill the output vector---*/
  for(i=0; i<out->data->nData; i++) {
    p = slope*FrVectGetValueI(inP->data, i) - bias;
    q = slope*FrVectGetValueI(inQ->data, i) - bias;
    out->data->dataD[i] = sqrt(p*p + q*q);}

  return;
}
/*---------------------------------------------------------------------------*/
void FdGetModulusProcess(FdGetModulus *mod,
			 FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdGetModulusProcessOne(mod, frame);

  next = mod->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdGetModulusNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdGetModulus *mod;
  char *input;

  mod = (FdGetModulus*) calloc(1,sizeof(FdGetModulus));
  if(mod == NULL) CfgMsgAddFatal("malloc FdGetModulus failed");

  mod->action = FdActionNew((FdAction**) arg, 
			    (void*) mod, FdGetModulusProcess, "FdGetModulus");

  FrStrCpy(&(mod->output), CfgParseGetNextString(data));
  input  =  CfgParseGetNextString(data);

  mod->inP  = malloc(strlen(input)+2);
  mod->inQ  = malloc(strlen(input)+2);
  sprintf(mod->inP, "%sp",input);
  sprintf(mod->inQ, "%sq",input);
  mod->inPStatus = FR_YES;
  mod->inQStatus = FR_YES;

  CfgMsgAddInfo("Add GET_MODULUS:  new channel: %s from %s %s ",
		mod->output, mod->inP, mod->inQ);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdGetPhaseProcessOne(FdGetPhase *ph,
		          FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *inP, *inQ;
  FrProcData *out;
  double p, q, dPhi, slope, bias, rawPhase;
  long nData, i;

  /*----------------------------------------------first get input channels---*/
  inP = FrAdcDataFind(frame, ph->inP);
  if(inP == NULL) {
    if(ph->inPStatus == FR_YES) {
      CfgMsgAddError("FdGetPhase: %d: ADC channel %s start to be missing",
		     frame->GTimeS, ph->inP);
      ph->inPStatus = FR_NO;}
    return;}
  else if(ph->inPStatus != FR_YES) {
    CfgMsgAddInfo("FdGetPhase: %d: ADC %s is back",frame->GTimeS, ph->inP);
    ph->inPStatus = FR_YES;}

  inQ = FrAdcDataFind(frame, ph->inQ);
  if(inQ == NULL) {
    if(ph->inQStatus == FR_YES) {
      CfgMsgAddError("FdGetPhase: %d: ADC channel %s start to be missing",
		     frame->GTimeS,ph->inQ);
      ph->inQStatus = FR_NO;}
    return;}
  else if(ph->inQStatus != FR_YES) {
    CfgMsgAddInfo("FdGetPhase: %d: ADC %s is back",frame->GTimeS, ph->inQ);
    ph->inQStatus = FR_YES;}

  /*---------------------------------------------------check sampling rate---*/
  if(inP->data->nData != inQ->data->nData) 
    CfgMsgAddFatal("FdGetPhase: channel length missmatch: %"FRLLD"!=%"FRLLD"",
		   inP->data->nData, inQ->data->nData);
    
  /*--------------------------------------------create the output channel---*/
  nData = inP->data->nData;
  out = FrProcDataNew(frame, ph->output, inP->sampleRate, nData, -64);
  if(out  == NULL) CfgMsgAddFatal("FdGetPhase Proc malloc %s failed at %d",
				  ph->output, frame->GTimeS);
  out->comment = malloc(strlen(ph->inQ)+128);
  if(out->comment == NULL) 
    CfgMsgAddFatal("FdGetPhaseProc malloc comment failed at %d",frame->GTimeS);
  sprintf(out->comment, "produced by FdGetPhase from %sp/q", ph->inP);
  FrVectSetUnitY(out->data, "rad");
  out->type = 1;

  /*---------------------------------------------- fill the output vector---*/
  slope = inP->slope;
  bias  = inP->bias;

  for(i=0; i<out->data->nData; i++) {
    p = slope*FrVectGetValueI(inP->data, i) - bias;
    q = slope*FrVectGetValueI(inQ->data, i) - bias;
    rawPhase = atan2(p, q);
    dPhi = rawPhase - ph->rawPhase;
    if(dPhi >  FRPI) ph->phase -= FRTWOPI;
    if(dPhi < -FRPI) ph->phase += FRTWOPI;
    out->data->dataD[i] = rawPhase + ph->phase;
    ph->rawPhase = rawPhase;}

  return;
}
/*---------------------------------------------------------------------------*/
void FdGetPhaseProcess(FdGetPhase *ph,
		       FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdGetPhaseProcessOne(ph, frame);

  next = ph->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdGetPhaseNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdGetPhase *ph;
  char *input;

  ph = (FdGetPhase*) calloc(1,sizeof(FdGetPhase));
  if(ph == NULL) CfgMsgAddFatal("malloc FdGetPhase failed");

  ph->action = FdActionNew((FdAction**) arg, 
			   (void*) ph, FdGetPhaseProcess, "FdGetPhase");

  FrStrCpy(&(ph->output), CfgParseGetNextString(data));
  input  =  CfgParseGetNextString(data);

  ph->inP  = malloc(strlen(input)+2);
  ph->inQ  = malloc(strlen(input)+2);
  sprintf(ph->inP, "%sp",input);
  sprintf(ph->inQ, "%sq",input);
  ph->inPStatus = FR_YES;
  ph->inQStatus = FR_YES;
  ph->rawPhase = 0;
  ph->phase    = 0;

  CfgMsgAddInfo("Add GET_PHASE: new channel: %s from %s %s ",
		ph->output, ph->inP, ph->inQ);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdDemodParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_DEMOD",
			 FdDemodNew, (void *) &(fdIO->actionsIn), 7,
			 CfgString, CfgString, CfgReal, CfgString, 
                         CfgDec, CfgReal, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_DEMOD",
			 FdDemodNew, (void *) &(fdIO->actionsOut), 7,
			 CfgString, CfgString, CfgReal, CfgString, 
                         CfgDec, CfgReal, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_FSHIFT",
			 FdFShiftNew, (void *) &(fdIO->actionsIn), 4,
			 CfgString, CfgString, CfgReal, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_FSHIFT",
			 FdFShiftNew, (void *) &(fdIO->actionsOut), 4,
			 CfgString, CfgString, CfgReal, CfgString);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_GET_MODULUS",
			 FdGetModulusNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_GET_MODULUS",
			 FdGetModulusNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgString);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_GET_PHASE",
			 FdGetPhaseNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_GET_PHASE",
			 FdGetPhaseNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgString);

  return;
} 
