#include "FdIO.h"

/*-----------------------------------------------declare private functions---*/
typedef struct FdPlayback  FdPlayback;

struct FdPlayback{
  char *fileName;      /* name of the input playback file              */
  int   start;         /* start time used in the playback file         */
  int   len;           /* length used in the playback file             */
  int   offset;        /* time offset when replaying the data          */
  int   nReplay;       /* number of time the playback file is replayed */
  int   iReplay;       /* current number of replay                     */
  int   frLen;         /* length of the playback frames                */
  int   latency;       /* playback frame are produced with this latency*/
  int   inputTime;     /* time of the frame to read                    */
  FrFile*    iFile;    /* input frame file                              */
  FdAction*  action;   /* action object which hold this playback object */
};
 
int  FdPlaybackNew(void *arg, CfgDataType_t *data);
void FdPlaybackProcess(FdPlayback* list, FrameH* frame);
 
/*---------------------------------------------------------------------------*/
int FdPlaybackNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdPlayback* p;
  FrameH* frame;
  long currentGPS;

  /*------------create the object and add it at the end of the linked list---*/
  p = (FdPlayback*) calloc(1,sizeof(FdPlayback));
  if(p == NULL) CfgMsgAddFatal("malloc FdPlayback failed");

  p->action = FdActionNew((FdAction**) arg, (void*) p, 
 	                   FdPlaybackProcess, "FDIN_PLAYBACK");

  if(p->action != *((FdAction**) arg)) 
    CfgMsgAddFatal("FDIN_PLAYBACK  could not be use with the %s key",
                    (*((FdAction**) arg))->type);

  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&p->fileName, CfgParseGetNextString(data));
  p->start   = CfgParseGetNextDec(data);
  p->len     = CfgParseGetNextDec(data);
  p->offset  = CfgParseGetNextDec(data);
  p->nReplay = CfgParseGetNextDec(data);
  p->latency = CfgParseGetNextDec(data);

  CfgMsgAddInfo("We will replay file %s", p->fileName);
  CfgMsgAddInfo(" start=%d len=%d offset=%d nReplay=%d latency=%d",
		p->start, p->len, p->offset, p->nReplay, p->latency);

  /*----------------------------------open file and check parameters---*/
  p->iFile = FrFileINew(p->fileName);
  if(p->iFile == NULL) CfgMsgAddFatal("Cannot open playback file");
  p->iFile->compress = FdIOGetCompressIn();

  if(p->start <= 0) p->start = FrFileITStart(p->iFile);
  if(p->len   <= 0) p->len = FrFileITEnd(p->iFile) - FrFileITStart(p->iFile);

  /*------------------------read the first frame to get the frame duration---*/
  frame = FrameHReadT(p->iFile, p->start);
  if(frame == NULL) 
    CfgMsgAddFatal("Could not read first frame from playback file");
  p->frLen = frame->dt;
  if(p->frLen != frame->dt) 
    CfgMsgAddFatal("Input frame duration is not an integer number of s:%g",
		   frame->dt);
  FrameFree(frame);

  /*----adjust start time and duration to a multiple of the frame duration---*/
  p->start  = p->frLen*(((p->start-1)/p->frLen)+1);
  p->len    = p->frLen*    (p->len   /p->frLen);
  p->offset = p->frLen*    (p->offset/p->frLen);
  CfgMsgAddInfo("Open playback file: %s",p->fileName);
  CfgMsgAddInfo(" Use playback file from %ds. for %ds, offset:%ds. nReplay:%d"
		" latency:%d", p->start,  p->len, 
		p->offset, p->nReplay, p->latency);

  /*----------------- compute the original time of the first frame to read---*/
  currentGPS = FrGetCurrentGPS();

  p->iReplay = (currentGPS - (p->start + p->offset))/ p->len;
  p->inputTime = currentGPS - p->iReplay*p->len -  p->offset;
  p->inputTime = p->frLen*(((int)p->inputTime)/p->frLen);
 
  CfgMsgAddInfo(" Set reading file at %d, gps now:%ld iReplay=%d",
		p->inputTime, currentGPS, p->iReplay);

  if(p->nReplay > 0 && p->iReplay < 0)  
    CfgMsgAddFatal("  Playback could only start in %ld seconds",
		   p->start + p->offset - currentGPS);
  if(p->nReplay > 0 && (p->iReplay > p->nReplay)) { 
    CfgMsgAddFatal("  Requested playback time is over; No frame read");}
  
 
  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
FrameH *FdPlaybackProcOne(FdPlayback *p)
/*-----------------------------------------------------------------------------*/
{
  FrameH *frame;
  long  currentGPS, nFramesLate, wait, offset;
  char comment[128];
  FrSerData *ser;
  double nBytes;
  unsigned int i;
  FrAdcData *adc;

  /*------read the next expected frame; unlesss we replay the data a limited 
    number of times and we have not yet reach the start of the data segment---*/
  nBytes = p->iFile->nBytes;

  if((p->nReplay <= 0) ||
     (p->nReplay > 0 && 
     (p->iReplay >= 0 && p->iReplay < p->nReplay))) {
    if(p->action->tag == NULL) {
      frame = FrameReadT(p->iFile, p->inputTime);}
    else {
      frame = FrameReadTChnl(p->iFile, p->inputTime, p->action->tag);
      if(frame != NULL) {
        frame->event = FrEventReadTF(p->iFile, p->action->tag, 
                                     frame->GTimeS + 1.e-9*frame->GTimeN, 
                                     frame->dt, 1, 0);}}
    nBytes = p->iFile->nBytes - nBytes;}
  else {
    frame = NULL;
    nBytes = 0;}

  if(frame != NULL) {
    if(p->frLen != frame->dt) {
      CfgMsgAddFatal("At %d input frame length has changed from %d to %g",
		     frame->GTimeS, p->frLen, frame->dt);}}

  /*-----------------------------------------------------report frame size---*/
  snprintf(p->action->serData, p->action->serDataSize,
    	"iReplay %d units MB nBytesIn %g units counts", 
	p->iReplay, nBytes/1024/1024); 
  if(nBytes < 1024*128) {
    snprintf(p->action->userInfo, p->action->userInfoSize,
	" nBytesIn %.2f kB", nBytes/1024);}
  else {
    snprintf(p->action->userInfo, p->action->userInfoSize,
	" nBytes %.2f MB", nBytes/1024/1024);}

  /*-----------------------------wait some time if the frame arrive too soon---*/
  currentGPS = FrGetCurrentGPS();
  offset = p->iReplay * p->len + p->offset;
  wait = p->inputTime + offset - currentGPS + p->latency;
  if(wait > 0) sleep(wait);

  /*------------if we cannot keep up in real time; skip a bunch of frames---*/
  nFramesLate = wait/p->len - 1;
  if(nFramesLate > 10) {
    p->inputTime += nFramesLate*p->frLen;
    CfgMsgAddWarning("Playback is too slow at %ld; jumps %ld frames",
		     currentGPS, nFramesLate);}

  /*----------------------Do we need to restart the file from the begining?---*/
  p->inputTime += p->frLen;
  if(p->inputTime >= p->start + p->len) {
    p->inputTime -= p->len;
    p->iReplay ++;
    if((p->iReplay >= p->nReplay) && p->nReplay > 0) {
      CfgTerminate();
      CfgMsgAddInfo("Last playback frame read from %d",p->inputTime);}}

  /*---------------------------------------------------------update GPS time---*/
  if(frame != NULL) {
    sprintf(comment, "FdPlayback: original GPS:%d",frame->GTimeS);
    FrHistoryAdd(frame,comment);

    adc = FrAdcDataNew(frame, "V1:FdPlayback_original_GPS", 1, frame->dt, 32);
    if(adc == NULL) CfgMsgAddFatal("FdPlaybackProcOne dt=%g", frame->dt);
    for(i=0; i<adc->data->nData; i++) {
      adc->data->dataUI[i] = frame->GTimeS+i;}

    frame->GTimeS += offset;
    frame->ULeapS = FrGetLeapS(frame->GTimeS);

    if(frame->rawData != NULL) {
      for(ser = frame->rawData->firstSer; ser != NULL; ser = ser->next) {
	ser->timeSec += offset;}}}

  return(frame);
}

/*---------------------------------------------------------------------------*/
void FdPlaybackProcess(FdPlayback *plbck,
		       FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  frame = FdPlaybackProcOne(plbck);

  next = plbck->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
void FdPlaybackParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_PLAYBACK",
			 FdPlaybackNew, (void *) &(fdIO->actionsIn), 6,
			 CfgString, CfgDec, CfgDec, CfgDec, CfgDec, CfgDec);

  return;
}
