/*----------------------------------------------------------------------------*/
#include "FdIO.h"

typedef struct FdRangeGating FdRangeGating;

struct FdRangeGating {
  char*   hoftChannel;     /* name of the h(t) channel                        */
  double  mass;            /* mass used for the range computation             */
  double  freqMin;         /* starting frequency for the horizon computation  */
  int     rangeRate;       /* sample rate at which the range is computed      */
  double  dtFFT;           /* FFT duration                                    */
  double  dtTrans;         /* Time to go from 0 to 1 for the window           */
  double  minRange;        /* minimal range to apply gating                   */
  int     tRangeMoni;      /* time used to monitor the range                  */
  char*   extGateName;     /* name of the external gate channel               */
  /*-------------------------------------------end of configuration parameters*/
  int     nDataTrans;      /* number of data point for the transition         */
  double  rangePart;       /* fraction of the current range to get minRange   */
  double  rangeMedian;     /* range for the median PSD                        */
  double  median;          /* last median range value                         */
  int     nFrames;         /* number of frames processed                      */
  int     nFramesMedian;   /* number of frames used to compute the median     */
  int     overSampling;    /* number of sub-steps to compute the ranges       */
  FRBOOL   savePSD;        /* if FR_YES save the PSD in frames                */
  FrVect*  gate;           /* gate as function of time, same rate as hoft     */
  FrVect*  gateUp;         /* vector holding a transition from 0 to 1         */
  FrVect*  gateDown;       /* vector holding a transition from 1 to 0         */
  FrVect*  gateFlag;       /* vector for the gate flag accross two frames     */
  FrVect*  gateFlagRange;  /* vector for the gate flag only for the range     */
  FrVect*  rangeV;         /* vector holding the range over two frames        */
  FrvRFFT* fft;
  FrVect** psd;
  FrVect*  psdMedian;
  FrVect*  psdLast; 
  int      nPsdPerFrame;   /* number of PSD per frame */
  int      nSamplesPerFFT;
  FrvFilterB* shapingFilter;/* high pass shaping filter to reduce the dynamic */
  double   shapingFreq;    /* cut of frequency for the shaping filter         */
  int      shapingOrder;   /* order of the shaping filter                     */
  FrVect*  hoftHP;         /* last output of the shaping filter               */
  double* valuesAtOneFreq;
  FrameH* previousFrame;
  FdAction* action;       /* this is the action process                       */
};

void FdRangeGatingProcess(FdRangeGating *rg, FrameH* frame);

static int compare_double( const void *p1, const void *p2 )
{
  double x1 = *(const double *)p1;
  double x2 = *(const double *)p2;
  return (x1 > x2) - (x1 < x2);
}

/*----------------------------------------------------------------------------*/
double FdRGGetRangeOne(
   FrVect *psd,  /* pointer to PSD vector                                     */
   double mass,
   double freqMin,
   double freqShaping,
   int    orderShaping)
/*----------------------------------------------------------------------------*/
{
  double pi, c, G, Mpc, Msun, m1, m2, xLSO, nuLSO, norm, K, df;
  double h2, snr, freq, eta, range, freqRatio, shaping;
  int i;

  pi = acos(-1.);
  c = 299792458.;
  G = 6.6726e-11;
  Mpc = 3.0856775807e22;
  Msun = 1.98892e30;

  /*--------------------------------------------- stop at last stable orbit---*/
  m1 = mass;
  m2 = mass;
  eta = m1*m2/(m1+m2)/(m1+m2);
  xLSO = (2.*(-9. - eta + sqrt(1539. - 1008.*eta + 19.*eta*eta)))/
         (3.*(81. - 57.*eta + eta*eta));
  nuLSO = 1./(pi*(m1+m2)*4.92554e-6)*pow(xLSO,3./2.);

  norm = 5./4.*pi/6.*pow(G,5./3.)/c/c/c*pow(Msun,5./3.)/Mpc/Mpc*pow(pi,-7./3.);
  K = m1*m2/(m1+m2)*pow(m1+m2,2./3.);
  norm = norm*K;

  if(nuLSO > 1024) nuLSO =1024;

  df = psd->dx[0];
  snr = 0;
  for(i=1; i<psd->nData; i++) {
    freq = i*df;
    if(freq < freqMin) continue;
    if(freq > nuLSO) break;
    h2 = norm*pow(freq,-7./3.);
    freqRatio = pow(freqShaping/freq,orderShaping);
    shaping = 1./(1+freqRatio*freqRatio);
    snr += h2*shaping/psd->dataD[i];}

  snr = sqrt(snr * 4. * df);
  
  range = snr/(8.*2.26);

  return(range);
}
/*----------------------------------------------------------------------------*/
void FdRGInit(
   FrVect *hoft, /* pointer to PSD vector                                     */
   FdRangeGating *rg)
/*----------------------------------------------------------------------------*/
{
  int i, nPSD, dtFrame, nBits;
  double dnu, rate, b;
  char name[128];

  /*-------------create the FFT object to save the PSD---*/
  rg->nSamplesPerFFT = rg->dtFFT/hoft->dx[0];
  rg->fft = FrvRFFTNewT("HN",hoft->dx[0]*rg->nSamplesPerFFT, 1.e9);    
  if(rg->fft == NULL) CfgMsgAddFatal("FdHProcess: Could not create FFT");

  /*------------------ create storage for the PSD---*/
  dtFrame = 0.5*hoft->dx[0]*hoft->nData;
  rg->nFramesMedian = rg->tRangeMoni/dtFrame;
  if(rg->nFramesMedian <= 0) CfgMsgAddFatal("Gating: too short dtMedian (%d)"
     " for the frame duration (%d)", rg->tRangeMoni, dtFrame); 
  rg->nPsdPerFrame = rg->rangeRate*dtFrame;
  nPSD = rg->nFramesMedian * rg->nPsdPerFrame; 
  rg->psd = (FrVect**) calloc(nPSD, sizeof(FrVect*));
  rg->valuesAtOneFreq = (double*) calloc(nPSD, sizeof(double));

  dnu = 1./(rg->nSamplesPerFFT*hoft->dx[0]);
  for(i = 0; i < nPSD; i++) {
     sprintf(name,"PSD_%d",i);
     rg->psd[i] = FrVectNew1D(name, FR_VECT_8R, rg->nSamplesPerFFT/2,
                              dnu,"frequency[Hz]","h^{2}/sqrt(Hz)");}
  rg->psdMedian = FrVectNew1D("PSD_median", FR_VECT_8R, rg->nSamplesPerFFT/2,
                               dnu,"frequency[Hz]","h^{2}/sqrt(Hz)");
  rg->psdLast   = FrVectNew1D("PSD_last",   FR_VECT_8R, rg->nSamplesPerFFT/2,
                               dnu,"frequency[Hz]","h^{2}/sqrt(Hz)");

  /*---------------------------------------- create vector for range values---*/
  sprintf(name,"%s_GateRange",rg->hoftChannel);
  rg->rangeV = FrVectNewTS(name, rg->rangeRate, 2*rg->nPsdPerFrame, -32);
  FrVectSetUnitY(rg->rangeV,"Mpc");

  /*------------------------------------create the vector for the gate flag---*/
  sprintf(name,"%s_GateFlag",rg->hoftChannel);
  rg->gateFlag = FrVectNewTS(name,rg->rangeRate, 2*rg->nPsdPerFrame, 8);
  FrVectSetUnitY(rg->gateFlag,"flag 1=OK, 0=gated");

  /*------------------------------------create the vector for the gate flag---*/
  if(rg->extGateName != NULL) {
    sprintf(name,"%s_GateFlagRange",rg->hoftChannel);
    rg->gateFlagRange = FrVectNewTS(name,rg->rangeRate, 2*rg->nPsdPerFrame, 8);
    FrVectSetUnitY(rg->gateFlagRange,"flag 1=OK, 0=gated");}

  /*-------------------------------------create storage for the gate vector---*/
  if(hoft->type == FR_VECT_8R) nBits = -64;
  else                         nBits = -32;
  sprintf(name,"%s_Gate",hoft->name);
  rate = 1./hoft->dx[0];
  rg->gate = FrVectNewTS(name, rate, hoft->nData, nBits);
  memset(rg->gate->data, 0, rg->gate->nBytes);
  FrVectSetUnitY(rg->gate,"weight");

  /*-------------------------create the transition vector with Tukey window---*/
  rg->nDataTrans = rg->dtTrans / hoft->dx[0];
  if(2*rg->nDataTrans + rg->nSamplesPerFFT/2 > hoft->nData/2) 
    CfgMsgAddFatal("Too long trantion (%d, %d, %ld)",
        rg->nDataTrans, rg->nSamplesPerFFT, hoft->nData);

  b = FRTWOPI/rg->nDataTrans;
  rg->gateDown = FrVectNewTS("Gate_going_down", rate, rg->nDataTrans, nBits);
  for(i = 0; i<rg->nDataTrans; i++) {
    if(nBits == -64) rg->gateDown->dataD[i] = 0.5*(1+cos(b*(i*.5)));
    else             rg->gateDown->dataF[i] = 0.5*(1+cos(b*(i*.5)));}

  rg->gateUp = FrVectNewTS("Gate_going_up", rate, rg->nDataTrans, nBits);
  for(i = 0; i<rg->nDataTrans; i++) {
    if(nBits == -64) rg->gateUp->dataD[i] = 0.5*(1-cos(b*(i*.5)));
    else             rg->gateUp->dataF[i] = 0.5*(1-cos(b*(i*.5)));}

  return;
}
/*----------------------------------------------------------------------------*/
void FdRGGetRange(
   FdRangeGating *rg,
   FrameH *previousFrame,
   FrVect *hoft)
/*----------------------------------------------------------------------------*/
{
  int i,j, iBin, nZeros, nFrames, start, step, nPSD, iOff, nBytes, iRangeOffset;
  char name[256];
  FrVect *psd;
  double *fftD, corr;

  /*--------------------------------------------shift previous range values---*/
  nBytes = rg->rangeV->nBytes/2;
  memcpy(rg->rangeV->data, rg->rangeV->data + nBytes, nBytes);

  /*-------------if we have several (> 4) bins set to zero, h(t) is invalid---*/
  nZeros = 0;
  if     (hoft->type == FR_VECT_8R)
    for(i=0; i<hoft->nData; i++) {if(hoft->dataD[i] == 0) nZeros++;}
  else if(hoft->type == FR_VECT_4R)
    for(i=0; i<hoft->nData; i++) {if(hoft->dataF[i] == 0) nZeros++;}
  else return;

  if(nZeros > 4) { /*--set to zero new range values and save previous value---*/
    for(i = rg->rangeV->nData/2; i<rg->rangeV->nData; i++) {
      rg->rangeV->dataF[i] = 0;}

    FrVectZoomInI(rg->rangeV, 0, rg->rangeV->nData/2);
    FrProcDataNewVT(previousFrame, FrVectCopy(rg->rangeV),1);
    FrVectZoomOut(rg->rangeV);

    return;}

  /*---------------------------------------compute the PSD for each subset---*/
  step   = 1./(rg->rangeRate*hoft->dx[0]);
  start  = hoft->nData - rg->nSamplesPerFFT - (rg->nPsdPerFrame-1) * step; 
  start -= 0.5 * step; /* 0.5 is to align the range when making plots*****/
  iOff = rg->nPsdPerFrame * (rg->nFrames % rg->nFramesMedian);

  for(i = 0; i < rg->nPsdPerFrame; i++) {
    FrVectZoomInI(hoft, start + i*step, rg->nSamplesPerFFT);
    FrvRFFTFor(rg->fft, hoft);
    fftD = rg->fft->output->dataD;
    for(j=0; j<rg->fft->output->nData; j++) {
      rg->psd[i+iOff]->dataD[j] = fftD[2*j  ]*fftD[2*j] + 
                                  fftD[2*j+1]*fftD[2*j+1];}
    if(rg->savePSD == FR_YES) {
      psd = FrVectCopy(rg->psd[i+iOff]);
      sprintf(name,"%s_PSD_%d", hoft->name, i);
      FrVectSetName(psd, name);
      FrProcDataNewVT(previousFrame, psd, 2);}
    FrVectZoomOut(hoft);}

  /*-------to smooth PSD, take the max of the local value and median value---*/
  /*--------------at the beggining, we have spectrums for only few frames---*/ 
  rg->nFrames++;
  if(rg->nFrames < rg->nFramesMedian) nFrames = rg->nFrames;
  else                                nFrames = rg->nFramesMedian;
  nPSD = nFrames*rg->nPsdPerFrame;

  /*--------------------------------------------------build the median PSD---*/
  for(iBin = 0; iBin< rg->psdMedian->nData; iBin++) {
    for(i = 0; i< nPSD; i++) { 
      rg->valuesAtOneFreq[i] = rg->psd[i]->dataD[iBin];}
    qsort(rg->valuesAtOneFreq, nPSD, sizeof(double), compare_double);
    rg->psdMedian->dataD[iBin] = rg->valuesAtOneFreq[nPSD/2];}

  /*-------------------------------------------save median PSD if requested---*/
  if(rg->savePSD == FR_YES) {
    psd = FrVectCopy(rg->psdMedian);
    sprintf(name,"%s_PSD_median", hoft->name);
    FrVectSetName(psd, name);
    FrProcDataNewVT(previousFrame, psd, 2);}

  /*------------------------------------------------------compute the range---*/
  iRangeOffset = rg->nPsdPerFrame - rg->dtFFT*rg->rangeRate/2;
  corr = 1.2;  /* a rough correction factor since the single PSD is biaised */

  for(i = 0; i< rg->nPsdPerFrame; i++) {
    /*--------build the PSD as the max of the local value and median value---*/
    for(iBin = 0; iBin< rg->psdMedian->nData; iBin++) {
      if(rg->psd[i+iOff]->dataD[iBin] < rg->psdMedian->dataD[iBin])
           rg->psdLast->dataD[iBin] = rg->psdMedian->dataD[iBin];
      else rg->psdLast->dataD[iBin] = rg->psd[i+iOff]->dataD[iBin];}

    rg->rangeV->dataF[i+iRangeOffset] =  
          corr * FdRGGetRangeOne(rg->psdLast, rg->mass, rg->freqMin, 
				 rg->shapingFreq, rg->shapingOrder);}

  /*-------------------------------------------save median PSD if requested---*/
  if(rg->savePSD == FR_YES) {
    psd = FrVectCopy(rg->psdLast);
    sprintf(name,"%s_PSD_Last", hoft->name);
    FrVectSetName(psd, name);
    FrProcDataNewVT(previousFrame, psd, 2);}

  /*-----------------duplicate the last range value to complete the vector---*/
  iRangeOffset += rg->nPsdPerFrame;
  for(i = iRangeOffset; i<rg->rangeV->nData; i++) {
    rg->rangeV->dataF[i] = rg->rangeV->dataF[iRangeOffset-1];}

  /*----------------------------------------- get range for the median PSD---*/
  rg->rangeMedian = FdRGGetRangeOne(rg->psdMedian, rg->mass, rg->freqMin,
				    rg->shapingFreq, rg->shapingOrder);

  if(rg->tRangeMoni > 0) rg->minRange = rg->rangePart * rg->rangeMedian;
    
  /*---------------------------save the range vector for the previous frame---*/
  FrVectZoomInI(rg->rangeV, 0, rg->rangeV->nData/2);
  FrProcDataNewVT(previousFrame, FrVectCopy(rg->rangeV),1);
  FrVectZoomOut(rg->rangeV);

  return;
}
/*----------------------------------------------------------------------------*/
int FdRangeGatingNew(void *arg,   CfgDataType_t *data)
/*----------------------------------------------------------------------------*/
{
  FdRangeGating *rg;

  rg = (FdRangeGating*) calloc(1,sizeof(FdRangeGating));
  if(rg == NULL) CfgMsgAddFatal("malloc FdRangeGating failed");

  rg->action = FdActionNew((FdAction**) arg, (void*) rg, 
                           FdRangeGatingProcess, "RangeGating");

  FrStrCpy(&(rg->hoftChannel), CfgParseGetNextString(data));
  rg->mass       = CfgParseGetNextReal(data);
  rg->freqMin    = CfgParseGetNextReal(data);
  rg->rangeRate  = CfgParseGetNextDec(data);
  rg->minRange   = CfgParseGetNextReal(data);
  rg->dtFFT      = CfgParseGetNextReal(data);
  rg->dtTrans    = CfgParseGetNextReal(data);
  rg->tRangeMoni = CfgParseGetNextDec(data);
  FrStrCpy(&(rg->extGateName), CfgParseGetNextString(data));

  if(rg->mass < 0) { /*-------- a dirty trick to active the saving of PSD---*/
       rg->mass = -rg->mass;
       rg->savePSD = FR_YES;} 
  else rg->savePSD = FR_NO;

  CfgMsgAddInfo("Range will be computed on %s, mass=%g from %g Hz"
		"rate=%d on FFT of %g s.",
 		rg->hoftChannel, rg->mass, rg->freqMin,
		rg->rangeRate,   rg->dtFFT);
  if(rg->minRange > 0) {
    if(rg->tRangeMoni <= 0)
      CfgMsgAddInfo(" Range gating will be applied if below %g Mpc "
		     "transition will last %g s.",
 		   rg->minRange, rg->dtTrans);
    else {
      rg->rangePart = rg->minRange;
      CfgMsgAddInfo(" Range gating will be applied if below %g the "
		    "median range computed over %d s. "
		     "transition will last %g s.",
 		   rg->minRange, rg->tRangeMoni, rg->dtTrans);}
    if(rg->extGateName != NULL) 
      CfgMsgAddInfo("  %s will be used as additional gating trigger",
		rg->extGateName);}
 
  CfgMsgAddInfo(" %s_Gated channel will be created",rg->hoftChannel);

  rg->previousFrame = NULL;

  /*-------high pass the h(t) vector to reduce signal dynamic at low freq.---*/
  rg->shapingFreq  = 50;
  rg->shapingOrder = 4;
  rg->shapingFilter = FrvFilterButNew(rg->shapingOrder,rg->shapingFreq,0,NULL);

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
int FdRangeGatingNewObs(void *arg,   CfgDataType_t *data)
/*----------------------------------------------------------------------------*/
{
  CfgMsgAddFatal("FDIN_RANGE_GATING is obsolete; use FDIN_RANGE_GATING_NEW");

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
FrameH* FdRangeGatingProcessOne(FdRangeGating *rg, FrameH *frame)
/*----------------------------------------------------------------------------*/
/*
This fonction computes the inspiral range on one frame long.
If the range is below a threshold, a gating is applied on a that data chunk 
plus two Tukey windows of half a frame each.
This is why the code is working with two frames, in order to take care of the
glitches at the overlap and apply gating with a window.
------------------------------------------------------------------------------*/
{  
  char name[256];
  int i, j, expectedGTimeS, nData, nBytes, rateRatio, minOkSize, iStart;
  int iRangeOffset, halfBin, jStart;
  FrVect *hoftHP, *hoftHPPrev, *hoftGated, *extGate, *gateV, *vect;
  FrVect *gateFlag, *hoftPrev, *hoftCurr;
  FrameH *previousFrame;
  double min, max;
  FRBOOL discontinuity;

  /*--------------------------------------------------------------------------*/
  previousFrame = rg->previousFrame;
  rg->previousFrame = frame;

  if(frame         == NULL) return(previousFrame);
  if(previousFrame == NULL) return(previousFrame);

  /*------------------------------------------------------get h(t) channels---*/
  hoftPrev = FrameFindVect(previousFrame, rg->hoftChannel);
  hoftCurr = FrameFindVect(        frame, rg->hoftChannel);
  discontinuity = FR_NO;

  FrVectFixNAN(hoftCurr);

  /*------------------------------------there was no h(t) channel up to now---*/
  if(hoftPrev == NULL && hoftCurr == NULL) return(previousFrame);

  /*---------------------------------------------detect frame discontinuity---*/
  expectedGTimeS = previousFrame->GTimeS + (int) previousFrame->dt;
  if(expectedGTimeS != frame->GTimeS) discontinuity = FR_YES;

  /*--------create a high pass filtered h(t) channel to avoid dynamic issue---*/
  /*   (the FFT for the gating are short and the h(t) LF component is large)  */
  if(hoftCurr != NULL) {
    FrVectMinMax(hoftCurr, &min, &max);
    if(max-min < 1.e-30) FrvFilterButReset(rg->shapingFilter);
    FrvFilterButProc(rg->shapingFilter,hoftCurr);
    hoftHP = FrVectCopy(rg->shapingFilter->output);}
  else {  /*------------------------ h(t) just disapear; add a dummy vector---*/
    hoftHP = FrVectCopy(rg->hoftHP);
    discontinuity = FR_YES;}

  hoftHPPrev = rg->hoftHP;
  rg->hoftHP = hoftHP;
    
  /*---------------------the previous frame do not contains an h(t) channel---*/
  if(hoftHPPrev == NULL) {
    hoftHPPrev = FrVectCopy(hoftHP);
    discontinuity = FR_YES;}
 
  /*-----------------------------------get h(t) channel over the two frames---*/
  hoftHP = hoftHPPrev;
  hoftHP->next = FrVectCopy(rg->hoftHP);
  nData = hoftHP->nData;
  hoftHP->GTime = hoftHP->next->GTime - frame->dt;
  hoftHP = FrVectConcat(hoftHP, hoftHP->GTime, 2 * nData * hoftHP->dx[0]);

  /*------------ in case of discontinuity, reset the vector to force gating---*/
  if(discontinuity == FR_YES) FrvScale(0, hoftHP, hoftHP, NULL);

  /*------------------------------------------------- initialize if needed---*/
  if(rg->nPsdPerFrame == 0) FdRGInit(hoftHP, rg);

  /*--------------------------------------------get the set of range values---*/
  FdRGGetRange(rg, previousFrame, hoftHP);
  FrVectFree(hoftHP);

  /*--------------------------------------returns if no gating is requested---*/
  snprintf(rg->action->userInfo, rg->action->userInfoSize, 
	"range: %g Mpc",rg->rangeMedian);

  if(rg->minRange < 0) return(previousFrame);

  /*------------shift previously computed gate vector and flag by one frame---*/
  nBytes = rg->gate->nBytes/2;
  memcpy(rg->gate->data, rg->gate->data + nBytes, nBytes);

  nBytes = rg->gateFlag->nBytes/2;
  memcpy(rg->gateFlag->data, rg->gateFlag->data + nBytes, nBytes);

  /*-----------------------------------------computes the range gating flag---*/
  iRangeOffset = rg->nPsdPerFrame - rg->dtFFT*rg->rangeRate/2;
  for(i=iRangeOffset; i<rg->rangeV->nData; i++) { 
    if(rg->rangeV->dataF[i] < rg->minRange) rg->gateFlag->data[i] = 0;
    else                                    rg->gateFlag->data[i] = 1;}

  /*------------ if an external gating flag is provided, merge both of them---*/
  if(rg->extGateName != NULL) {
    memcpy(rg->gateFlagRange->data, rg->gateFlagRange->data + nBytes, nBytes);
    for(i=iRangeOffset; i<rg->rangeV->nData; i++) {
      rg->gateFlagRange->data[i] = rg->gateFlag->data[i];}

    extGate = FrameFindVect(previousFrame,rg->extGateName);
    if(extGate != NULL) {
      if(extGate->dx[0] != rg->rangeV->dx[0]) CfgMsgAddFatal("FdRangeGating: "
        "sampling rate mismatch for %s",rg->extGateName);
      for(i=iRangeOffset; i<extGate->nData; i++) {
        if(FrVectGetValueI(extGate, i) != 1) 
          rg->gateFlag->data[i] = 0;}}

    extGate = FrameFindVect(frame,rg->extGateName);
    if(extGate != NULL) {
      for(i=0; i<extGate->nData; i++) {
        if(FrVectGetValueI(extGate, i) != 1) 
          rg->gateFlag->data[i+extGate->nData] = 0;}}}

  /*-----------------cluster gate time that are too close for a full window---*/
  minOkSize = 2.*rg->dtTrans/rg->rangeV->dx[0] + 1;
  for(i=0; i<rg->gateFlag->nData-1; i++) {
    if(rg->gateFlag->data[i] == 1) continue;
    jStart = minOkSize-1;
    if(jStart + i >= rg->gateFlag->nData) jStart = rg->gateFlag->nData-i-1;
    for(j=jStart; j>0; j--) {
      if(rg->gateFlag->data[i+j] == 0) break;}
    if(j == 0) continue;
    for(;j>0; j--) {
      rg->gateFlag->data[i+j] = 0;}}

  /*---------compute gate time series; first expand to the hoft sample rate---*/
  /* do not start from vector start since it was compted at the previous call */
  rateRatio = rg->gate->nData/rg->gateFlag->nData;
  iStart = rg->dtTrans/rg->gateFlag->dx[0];
  for(i=iStart; i<rg->gateFlag->nData; i++) {
    if(rg->gate->type == FR_VECT_8R) {
      for(j=0; j<rateRatio; j++) {
        rg->gate->dataD[i*rateRatio+j] = rg->gateFlag->data[i];}}
    else {
      for(j=0; j<rateRatio; j++) {
        rg->gate->dataF[i*rateRatio+j] = rg->gateFlag->data[i];}}}

  /*-------------------------------------search for transitions from 1 to 0---*/
  for(i=rg->gateDown->nData/rateRatio; i<rg->gateFlag->nData; i++) {
    if(rg->gateFlag->data[i-1] == 0) continue; 
    if(rg->gateFlag->data[i]   == 1) continue;
    memcpy(rg->gate->data + rg->gate->wSize*(i*rateRatio-rg->gateDown->nData),
           rg->gateDown->data, rg->gateDown->nBytes);}

  /*----------------now apply the transition from 0 to 1 in the gate vector---*/
  for(i=1; i<rg->gateFlag->nData-rg->gateDown->nData/rateRatio; i++) {
    if(rg->gateFlag->data[i-1] != 0) continue;
    if(rg->gateFlag->data[i]   != 1) continue;
    memcpy(rg->gate->data + rg->gate->wSize*(i*rateRatio), 
           rg->gateUp->data, rg->gateUp->nBytes);}

  /*-----------------------save the gate flag vector for the previous frame---*/
  FrVectZoomInI(rg->gateFlag, 0, rg->gateFlag->nData/2);
  gateFlag = FrVectCopy(rg->gateFlag);
  FrProcDataNewVT(previousFrame, gateFlag,1);
  FrVectZoomOut(rg->gateFlag);

  if(rg->extGateName != NULL) {
    FrVectZoomInI(rg->gateFlagRange, 0, rg->gateFlagRange->nData/2);
    FrProcDataNewVT(previousFrame, FrVectCopy(rg->gateFlagRange),1);
    FrVectZoomOut(rg->gateFlagRange);}

  /*-------------------------------------------apply gating and save result---*/
  FrVectZoomInI(rg->gate, 0, rg->gate->nData/2);

  if(hoftPrev != NULL) {
    hoftGated = FrVectCopyToD(hoftPrev, 1, NULL);
    FrvMult(hoftGated, rg->gate, hoftGated, NULL);

    sprintf(name,"%s_Gated",rg->hoftChannel);
    FrVectSetName(hoftGated, name);
    FrProcDataNewVT(previousFrame, hoftGated, 1);}

  /*-------------make a gate vector of the same type as hoft and save it------*/
  if(rg->gate->type == FR_VECT_8R) gateV = FrVectCopyToF(rg->gate, 1., NULL);
  else                             gateV = FrVectCopy(rg->gate);
  FrProcDataNewVT(previousFrame, gateV, 1);

  FrVectZoomOut(rg->gate);

  /*------------extend the gate flag vector to record also the transitions---*/
  halfBin = rateRatio/2;
  for(i=0; i<gateFlag->nData; i++) {
    if(gateV->dataF[i*rateRatio + halfBin] != 1) gateFlag->data[i] = 0;}

  /*--------------------------------------------------record minimal range---*/
  sprintf(name,"%s_GateMinRange",rg->hoftChannel);
  vect = FrVectNew1D(name, FR_VECT_4R, 1, previousFrame->dt,"time","Mpc");
  FrProcDataNewVT(previousFrame, vect, 1);
  vect->dataF[0] = rg->minRange;
  if(rg->tRangeMoni > 0) rg->minRange = rg->rangePart * rg->rangeMedian;

 return(previousFrame);
}
/*----------------------------------------------------------------------------*/
void FdRangeGatingProcess(FdRangeGating *rg, FrameH* frame)
/*----------------------------------------------------------------------------*/
{
  FdAction *next;

  frame = FdRangeGatingProcessOne(rg, frame);

  next = rg->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*----------------------------------------------------------------------------*/
void FdRangeGatingAdd(FdIO* fdIO)
/*----------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_RANGE_GATING_NEW",
			 FdRangeGatingNew, (void *) &(fdIO->actionsIn), 9,
			 CfgString, CfgReal, CfgReal, CfgDec, CfgReal, 
			 CfgReal,   CfgReal, CfgDec, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_RANGE_GATING_NEW",
			 FdRangeGatingNew, (void *) &(fdIO->actionsOut), 9,
			 CfgString, CfgReal, CfgReal, CfgDec, CfgReal, 
			 CfgReal,   CfgReal, CfgDec, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_RANGE_GATING",
			 FdRangeGatingNewObs, (void *) &(fdIO->actionsIn), 8,
			 CfgString, CfgReal, CfgDec, CfgReal, CfgDec, CfgDec,
                         CfgString, CfgHex);
 return;
}
