/*****************************************************************************/
/*  FdResize.c  Code to change the frame duration; by B. Mours               */
/*****************************************************************************/

#include <FdIO.h>

/*---------------------------------------------------------------------------*/
/* private definitions                                                       */
/*---------------------------------------------------------------------------*/

typedef struct FdResize    FdResize;
typedef struct FdFrSelect  FdFrSelect;
typedef struct FdSkipNoEvt FdSkipNoEvt;

struct FdSkipNoEvt{
  FdAction*  action;       // FdAction object 
};

struct FdResize{     
  double     newDuration;  /* new frame lenght                                */
  double     shortLength;  /* length of the intermediate short frames         */
  long long  firstGPS;     /* time of the first frame to process (nanoseconds) */
  FrameH*    oFrame;       /* store the frame parts                            */
  FdAction*  action;       /* FdAction object                                  */
};

struct FdFrSelect{     
  int        period;       // period for the selection
  int        start;        // start index for the selection
  int        end;          // end index for the selection
  FdAction*  action;       // FdAction object 
};

void FdResizeFrame(FdResize *resize, FrameH* frame);
int  FdFrSelectNew(void *arg,   CfgDataType_t *data);
void FdFrSelectFrame(FdFrSelect *s, FrameH* frame);

/*---------------------------------------------------------------------------*/
void FdSkipNoEvtFrame(FdSkipNoEvt *s, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *nextAction;

  nextAction = s->action->next;
  if(nextAction == NULL) return;

  /*--------- if no frame is passed, this is to flush the frame at the end---*/
  if(frame == NULL) {
    nextAction->action(nextAction->data, NULL);
    return;}
 
  /*--------------------------------keep processing only frames with events--*/
  if(frame->event != NULL || frame->simEvent != NULL) {
    nextAction->action(nextAction->data, frame);}

  else { /*-----------------------if not, delete frame and stop processing---*/
    FrameFree(frame);}

  return;
}
/*--------------------------------------------------------------------------*/
int FdSkipNoEvtNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{ 
  FdSkipNoEvt *skip;

  skip = (FdSkipNoEvt*) calloc(1,sizeof(FdSkipNoEvt));
  if(skip == NULL) CfgMsgAddFatal("malloc FdSkipNoEvt failed");
 
  skip->action =  FdActionNew((FdAction**) arg, (void*) skip, 
			FdSkipNoEvtFrame, "Skip frames without events");

  CfgMsgAddInfo("Add skip no event frame");

  return(CFG_OK);
}
/*--------------------------------------------------------------------------*/
int FdResizeAdd(FdAction** actionRoot, double newDuration, double firstGPS)
/*--------------------------------------------------------------------------*/
{                                             
  FdResize *resize;

  resize = (FdResize*) calloc(1,sizeof(FdResize));
  if(resize == NULL) CfgMsgAddFatal("malloc FdResize failed");
 
  resize->action =  FdActionNew(actionRoot, (void*) resize, 
				FdResizeFrame, "Set frame duration");

  resize->newDuration = newDuration;
  resize->firstGPS    = firstGPS;

  resize->shortLength = 0;
  resize->oFrame = NULL;

  CfgMsgAddInfo("Add frame resizing; new duration: %gs", resize->newDuration);
  if     (firstGPS == 1) CfgMsgAddInfo(" Will start with a full frame");
  else if(firstGPS  > 1) resize->firstGPS *= 1.e9;

  return(CFG_OK);
}
/*--------------------------------------------------------------------------*/
int FdResizeNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{ 
  double newDuration = CfgParseGetNextReal(data);
  double firstGPS    = CfgParseGetNextReal(data);        

  FdResizeAdd((FdAction**) arg, newDuration, firstGPS);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdResizeParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_SKIP_FRAMES_WITHOUT_EVENT", 
			 FdSkipNoEvtNew, (void*) &(fdIO->actionsOut), 
			 1, CfgString); // should be 0 param, but not supported by Cfg

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_FRAME_DURATION", 
			 FdResizeNew, (void*) &(fdIO->actionsIn), 
			 2, CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_FRAME_DURATION", 
			 FdResizeNew, (void*) &(fdIO->actionsOut), 
			 2, CfgReal, CfgReal);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_SELECT_FRAMES", 
			 FdFrSelectNew, (void*) &(fdIO->actionsIn), 
			 3, CfgDec, CfgDec, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_SELECT_FRAMES", 
			 FdFrSelectNew, (void*) &(fdIO->actionsOut), 
			 3, CfgDec, CfgDec, CfgDec);

  return;
}
/*---------------------------------------------------------------------------*/
void FdResizeFrame(FdResize *r, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrameH *shortFrame, *outFrame;
  int i, nSplit, maxLoop;
  double oldLength, shortLength, ratio, gap;
  FdAction *nextAction;
  long long nowNano, dNano, nFrames;

  nextAction = r->action->next;

  /*--------- if no frame is passed, this is to flush the frame at the end---*/
  if(frame == NULL) {
    FrameReshapeEnd(r->oFrame);
    if(nextAction != NULL) nextAction->action(nextAction->data, r->oFrame);
    r->oFrame = NULL;
    return;}
  /*---------------------------------output frame if no resizing is needed---*/
  if(r->newDuration <= 0 ||
     r->newDuration == frame->dt) {
    if(nextAction != NULL) nextAction->action(nextAction->data, frame);
    return;}

  /*-----if there is a frame gap such that we have to go to next frame: 
                                           output a partially built frame?---*/
  if(r->oFrame != NULL) {
    gap = (frame->GTimeS - r->oFrame->GTimeS) + 
          (frame->GTimeN - r->oFrame->GTimeN) * 1.e-9 - r->oFrame->dt;
    if(gap > 0) {
      CfgMsgAddWarning("FdResize: %g s. gap end at GPS=%d", gap, frame->GTimeS);
      if(nextAction != NULL) nextAction->action(nextAction->data, r->oFrame);
      r->oFrame = NULL;}}

  /*---------------if requested, compute the start time of the first frame---*/
  if(r->firstGPS == 1) {
    nowNano = 1.e9 * frame->GTimeS + frame->GTimeN;
    dNano   = 1.e9 * r->newDuration;
    nFrames = nowNano/dNano;
    if(nFrames*dNano == nowNano) r->firstGPS = nowNano;
    else                         r->firstGPS = dNano * (nFrames+1);
    CfgMsgAddInfo(" Frame resized to %g s., now: %.1f, first frame will start "
	"at %.1f", r->newDuration, 1.e-9*nowNano, 1.e-9*r->firstGPS);}
     
  /*---------------------------------------------resize frame if requested---*/
  /* Compute the intermediate short frame lenght. if the frame length is not
     a perfect multiple of the new frame find the longest length which is a
     submultiple of both frame length.
     case 10s->2s  need only FrameCopyPart;      nsplit=5; shortLen=2 
     case  2s->10s need only enlarge;            nSplit=1; shortLen=2 
     case 10s->16s need to go through 2s frames; nSplit=5; shortLen=2 
     case 16s->10s need to go through 2s frames; nSplit=8; shortLen=2 
     ------------------------------------------------------------------------*/

  oldLength = frame->dt;
  maxLoop = 10000;
  for(nSplit=1; nSplit<maxLoop; nSplit++) {
    shortLength = oldLength/nSplit;
    ratio = r->newDuration/shortLength;
    if(ratio - ((int)(ratio+1.e-8)) < 2.e-8) break;}

  if(nSplit == maxLoop)
    CfgMsgAddFatal("Frame resize at %d: cannot find intermediate frame "
		   "length; old length=%g requested length=%g",
		   frame->GTimeS, oldLength, r->newDuration);

  if(r->shortLength != shortLength) {
    CfgMsgAddInfo("Frame resize: use %gs. intermediate frames to go from "
		  "%g to %gs.", shortLength, oldLength, r->newDuration);
    r->shortLength = shortLength;}

  /*-------------output frame parts, building first short frames if needed---*/
  for(i=0; i<nSplit; i++) {
    if(nSplit == 1) { 
      shortFrame = frame;}   
    else {
      shortFrame = FrameCopyPart(frame, i*r->shortLength, r->shortLength);
      if(shortFrame == NULL) 
	CfgMsgAddFatal("Cannot build shortFrame at %d i=%d",frame->GTimeS,i);}

    if(r->newDuration == r->shortLength) {
      outFrame = shortFrame;}
    else {
      outFrame = FrFrameEnlarge(shortFrame, &r->oFrame, r->newDuration);}
    if(outFrame == NULL) continue;

    /*---------------if we are not at the first requested frame, remove it---*/
    nowNano = 1.e9*outFrame->GTimeS + outFrame->GTimeN;
    if(nowNano < r->firstGPS) {
      FrameFree(outFrame);
      continue;}

    /*-----------------------------call next action for this resized frame---*/
    if(nextAction != NULL) nextAction->action(nextAction->data, outFrame);}

  /*------delete input frame if it was splited in part using FrameCopyPart---*/
  if(nSplit > 1) FrameFree(frame);

  return;
}
/*----------------------------------------------------------------------------*/
void FdIOSetInputFrameDuration(FdIO* fdIO, int length) 
/*----------------------------------------------------------------------------*/
/* this function adds the input frame resize action as the next to last FdIO 
   input action. This is a bit complex since the last input action is of a 
   special type. This function must be call after FdIOParseAndIni, not before. 
------------------------------------------------------------------------------*/
{
  FdAction **action, *last;

  if(fdIO == NULL) return; 

  if(fdIO->actionsIn == NULL) 
    CfgMsgAddFatal("FdIOSetInputFrameDuration: no input frame action defined");

  action = &(fdIO->actionsIn);
  while((*action)->next != NULL) {action = &((*action)->next);}

  last = *action;
  *action = NULL; 

  FdResizeAdd(&(fdIO->actionsIn), length, 0); 

  (*action)->next = last; 

  return; 
}
/*--------------------------------------------------------------------------*/
int FdFrSelectNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{ 
  FdFrSelect *select;

  select = (FdFrSelect*) calloc(1,sizeof(FdFrSelect));
  if(select == NULL) CfgMsgAddFatal("malloc FdFrSelect failed");
 
  select->action =  FdActionNew((FdAction**) arg, (void*) select, 
				FdFrSelectFrame, "Select frames");

  select->period = CfgParseGetNextDec(data);
  select->start  = CfgParseGetNextDec(data);
  select->end    = CfgParseGetNextDec(data);

  CfgMsgAddInfo("Add frame selection: period=%d keep %d to %d", 
	select->period, select->start, select->end);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdFrSelectFrame(FdFrSelect *s, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *nextAction;

  nextAction = s->action->next;
  if(nextAction == NULL) return;

  /*--------- if no frame is passed, this is to flush the frame at the end---*/
  if(frame == NULL) {
    nextAction->action(nextAction->data, NULL);
    return;}
 
  /*--------if the time is in the right part of the period, keep processing--*/
  int index = frame->GTimeS % s->period;
  if(index >= s->start && index < s->end) {
    nextAction->action(nextAction->data, frame);}

  else { /*-----------------------if not, delete frame and stop processing---*/
    FrameFree(frame);}

  return;
}
