/*-----------------------------------------------------------------------------*/
/* FdSoftInjection.c Program to generate surrogate data + software injections  */
/*-----------------------------------------------------------------------------*/

#define _GNU_SOURCE

#include "FdIO.h"

/*---------------------------------------------------------private structure---*/
typedef struct FdSiChannel   FdSiChannel;
typedef struct FdSiDqChannel FdSiDqChannel;
typedef struct FdSiEvent     FdSiEvent;
typedef struct FdSiEventCh   FdSiEventCh;
typedef struct FdSiTChannel  FdSiTChannel;


struct FdSiChannel{         /* to create a surrogate channel           */
  char*        name;        /* channel name                            */
  char*        unit;        /* unit of the channel                     */
  double       sampleRate;  /* frequency of the produced channel       */
  long         nBits;       /* number of bits used to store the channel*/
  char*        fileName;    /* name of the file with the h spectrum    */
  double       scale;       /* rescaling factor for the vector         */
  /*-------------------------- end of input parameters ----------------*/
  FrRandomG*   seed;        /* structure to host the seed information  */
  FrVect*      noise;       /* input white noise                       */
  FrvFDFilter* filter;      /* frequency domain filter                 */
  FdAction*    action;      /* FdAction object                         */
};  

struct FdSiTChannel{        /* to create a test channel                */
  char*        name;        /* channel name                            */
  char*        unit;        /* unit of the channel                     */
  double       sampleRate;  /* frequency of the produced channel       */
  long         nBits;       /* number of bits used to store the channel*/
  double       mean;        /* mean signal value                       */
  double       rms;         /* RMS of the white noise                  */
  double       amplitude;   /* amplitude of the sine wave              */
  double       frequency;   /* frequency of the sine wave              */
  double       scale;       /* rescaling factor for the vector         */
  /*-------------------------- end of input parameters ----------------*/
  double       phase;       /* phase for the sine wave                 */
  FdAction*    action;      /* FdAction object                         */
};  

struct FdSiDqChannel{        /* to create a surrogate DQ channel       */
  char*        name;         /* channel name                           */
  double       sampleRate;   /* frequency of the produced channel      */
  int          unlockedValue;/* value for the bad data quality         */
  int          lockedValue;  /* value for the good data quality value  */
  double       minSegLen;    /* minimal segment duration               */
  double       maxSegLen;    /* maximum segment duration               */
  double       dutyCycle;    /* duty cycle for the good value          */
  char*        hoftName;     /* name of the h(t) channel to be reset   */
  /*-------------------------- end of input parameters ----------------*/
  double       segStop;      /* end of the next segment                */
  double       seed1;        /* seed value for the random generator    */
  double       seed2;        /* seed value for the random generator    */
  double       totTime;      /* total processed time                   */
  double       downTime;     /* total down time                        */
  FdAction*    action;       /* FdAction object                        */
};  

struct FdSiEvent {
  char*        eventName;   /* name of the sime event to be injected           */
  char*        eventNewName;/* new event name if requested                     */
  char*        fileName;    /* name of the file with software injection        */
  int          tStart;      /* start to look in the file at this time          */
  int          duration;    /* length of the file used                         */
  int          dtScan;      /* dtto scan the file to get the waveform duration */
  double       period;      /* time between two injections                     */
  double       jitter;      /* jitter (RMS) to be apply on the injection time  */
  double       offset;      /* events will be genereated at t=N*period+offset  */
  double       scaleMin;    /* lower bound for the amplitude rescaleing        */
  double       scaleMax;    /* upper bound for the amplitude rescaleing        */
  char*        injFlagName; /* injection flag name                             */
  int          injFlagRate; /* sampling rate of the injection channel          */
  double       injFlagTBefore;/* time to veto before the injection             */
  double       injFlagTAfter; /* time to veto after the injection              */
  double       injFlagTStart; /* next start time to be vetoed for an injection */
  double       injFlagTEnd;   /* next   end time to be vetoed for an injection */
  /*---------------------------end of input parameters-------------------------*/
  FdSiEventCh* channels;    /* linked list of channels                         */
  double       tNext;       /* time of the next FrSimEvent                     */
  double       tStartNext;  /* start time of the next injection                */
  double       tEndNext;    /* end time for the next injection                 */
  FrFile*      file;        /* pointer to the file for the software injection  */
  double       dt;          /* (FrSimEvent time)-(new scheduled injection time)*/
  int          iSim;        /* current index of the event to be injected       */
  int          nEvents;     /* number of FrSimEvents                           */
  FrSimEvent*  newSimEvent; /* current simEvent to be injected                 */
  FrSimEvent** simEvents;   /* list of simEvent to be injected                 */
  double       scale;       /* current rescaling factor                        */
  long         seed;        /* seed for the jitter & rescaling random numbers  */
  FRBOOL       hoftSimMissing;/* flag use control the error messages           */
  FdAction*    action;      /* FdAction object                                 */
};

struct FdSiEventCh {
  char*        waveformName; /* name of the waveform channel           */
  char*        channelName;  /* channel to which the waveform is added */
  char*        simEventName; /* name of the simeEvent to be copied     */
  char*        hoftSavedName;/* name of the recorded waveform channel  */
  FRBOOL       errorFlag;    /* is the error message already printed?  */
  FdSiEventCh* next;         /* next object in the linked list         */
};

void FdSiChannelProcess(FdSiChannel *ch, FrameH* frame);
void FdSiDqChannelProcess(FdSiDqChannel *ch, FrameH* frame);

void FdSiChannelInit(FdSiChannel *ch, FrameH* frame);

int    FdSiEventAddChannel(void *arg, CfgDataType_t *dataType);
void   FdSiEventGetTBeforeAfter(FrFile* iFile, FrSimEvent *evt, char* waveformName, double dt);
void   FdSiEventInit(FdSiEvent *si);
void   FdSiEventProcess(FdSiEvent *si, FrameH* frame);
int    FdSiEventScheduleNext(FdSiEvent *si, FrameH *frame);

int  FdSiTChannelNew(void *arg, CfgDataType_t *dataType);
void FdSiTChannelProcess(FdSiTChannel *ch, FrameH* frame);
void FdSiEventInjFlagProcess(FdSiEvent *si, FrameH* frame);

/*-----------------------------------------------------------------------------*/
int FdSiChannelNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdSiChannel *ch;
  int seed;

  ch = malloc(sizeof(FdSiChannel));
  if(ch == NULL) CfgMsgAddFatal("malloc FdSiChannel failed");

  ch->action = FdActionNew((FdAction**) arg,   (void*) ch, 
			   FdSiChannelProcess, "SiChannel");

  FrStrCpy(&(ch->name),     CfgParseGetNextString(dataType));
  FrStrCpy(&(ch->unit),     CfgParseGetNextString(dataType));
  ch->sampleRate =          CfgParseGetNextReal  (dataType);
  ch->nBits      =          CfgParseGetNextDec   (dataType);
  FrStrCpy(&(ch->fileName), CfgParseGetNextString(dataType));
  ch->scale      =          CfgParseGetNextReal  (dataType);
  seed           =          CfgParseGetNextDec   (dataType);

  if(ch->scale == 0) ch->scale = 1.;
  CfgMsgAddInfo("Add channel %s unit:%s (%gHz) nBits:%3ld scale:%g "
                "from spectrum:%s",
		ch->name,  ch->unit,  ch->sampleRate, 
		ch->nBits, ch->scale, ch->fileName);

  if(ch->nBits != -64 && 
     ch->nBits != -32 &&
     ch->nBits !=  16 &&
     ch->nBits !=  32) CfgMsgAddFatal("Invalid nBits: %ld",ch->nBits);

  if(seed != 0) CfgMsgAddInfo("  Seed=%d", seed);
  ch->seed = FrRandGNew(seed);
  if(ch->seed == NULL) CfgMsgAddFatal("FrRandomGNew failed");
  if(ch->nBits > 0) CfgMsgAddInfo("  Numerical noise introduced by integer"
				  " storage is= %g%s/sqrt(Hz)",1./(ch->scale*sqrt(6*ch->sampleRate)),
				  ch->unit);

  ch->filter = NULL;

  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
void  FdSiChannelInit(FdSiChannel *ch, FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  FrProcData *proc;
  FrVect *hoff;
  int   nData, i, init, nLines;
  double scale, dnu, freq, h, hMin, hMax, fMin, fMax, lastFreq;
  FILE *fp;
  size_t len;
  char *line;

  CfgMsgAddInfo(" Init. new channel %s with FFT length=%gs",
                ch->name, 2.*frame->dt);

  /*----------------------------------------------------- create the filter----*/
  ch->filter = FrvFDFilterNew(1.e10, 1., 0., ch->sampleRate);
  if(ch->filter == NULL)
    CfgMsgAddFatal("could not create filter for %s", ch->name);

  /*------------------------- first init: read the spectrum for the spectrum---*/
  len = strlen(ch->fileName);
  if(strcmp(ch->fileName,"NORMAL") == 0){
    CfgMsgAddInfo("  %s will just be a normal distribution of sigma=%g",
		  ch->name, ch->scale);}

  /*--------------------first case: the sensitivity is stored as a text file---*/
  else if(strcmp(ch->fileName+len-4,".txt") == 0) {
    fp = fopen(ch->fileName,"r");
    if(fp == NULL) CfgMsgAddFatal("could not open file %s",ch->fileName);

    len = 0;
    line = NULL;
    nLines = 0;
    lastFreq = 0;
    hMin = 1.e40;
    hMax = 0;
    fMin = -1.;
    fMax = -1.;
    while (getline(&line, &len, fp) > 0) {
      if(sscanf(line,"%lg %lg",&freq, &h) != 2) break;
      nLines++;
      if(nLines == 1)  CfgMsgAddInfo("  First values: h[%g]=%g",freq, h);
      if(freq <= lastFreq) CfgMsgAddFatal("Decreasing frequency %g at line %d",
					  freq, nLines);
      lastFreq = freq;
      if(h < hMin) {
	hMin = h;
	fMin = freq;}
      if(h > hMax) {
	hMax = h;
	fMax = freq;}
      FrvFDFilterAddTFBin(ch->filter, freq ,h ,0);}
    CfgMsgAddInfo("  %d lines read from file %s",nLines, ch->fileName);
    CfgMsgAddInfo("  last value: h[%g]=%g min: h[%g]=%g max: h[%g]=%g", 
                  freq, h, fMin, hMin, fMax, hMax);}

  /*----------------- second option: the sensitivity is stored in an FrVect ---*/
  else {
    hoff = FrVectLoad(ch->fileName);
    if(hoff == NULL) 
      CfgMsgAddFatal("Could not read vector from %s", ch->fileName);

    proc = FrProcDataNewV(frame, hoff); /*-----------store the spectrum used---*/
    if(proc == NULL) 
      CfgMsgAddFatal("ProcDataNew failed at %d, %s",frame->GTimeS, ch->name);
    proc->type = 2;

    FrvFDFilterFreeTFBin(ch->filter);
    for(i=1; i<hoff->nData; i++) {
      dnu = hoff->dx[0];
      FrvFDFilterAddTFBin(ch->filter,i*dnu,FrVectGetValueI(hoff,i),0);}}

  /*------------------- if this is an update of the spectrum, rebuild the TF---*/
  if(ch->filter->tf != NULL) FrvFDFilterBuildTFBin(ch->filter);

  /*------------------------------------------ create the white noise vector---*/
  nData = frame->dt*ch->sampleRate + 0.1;
  ch->noise = FrVectNewTS(ch->name, ch->sampleRate, nData, -32);
  if(ch->noise == NULL) CfgMsgAddFatal("Malloc noise failed for %s",ch->name);

  /*------- fill and filter the white noise vector two times
    to avoid the FFT filtering transiant at start time -----------------------*/ 
  scale = sqrt(ch->sampleRate/2.);

  for(init=0; init<2; init++) {
    for(i=0; i<ch->noise->nData; i++) {
      ch->noise->dataF[i] = scale*FrRandG(ch->seed);}
    FrvFDFilterProc(ch->filter, ch->noise);}
 
  return;
}
/*-----------------------------------------------------------------------------*/
void  FdSiChannelProcessOne(FdSiChannel *ch, FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  int   i;
  double scale;
  char comment[128];

  /*------------------------- first init: read the spectrum for the spectrum---*/
  if(ch->filter == NULL) FdSiChannelInit(ch, frame);

  /*----------------------------------------- fill a vector with white noise---*/
  scale = sqrt(ch->sampleRate/2.);
  for(i=0; i<ch->noise->nData; i++) {
    ch->noise->dataF[i] = scale*FrRandG(ch->seed);}

  /*----------------------------- apply the TF to provide the rigth spectrum---*/
  FrvFDFilterProc(ch->filter, ch->noise);

  /*----------------------create the FrAdcData to store the filtered channel---*/
  if(FrameFindVect(frame, ch->name) != NULL) {
    CfgMsgAddFatal("%d: The channel %s already exist: we could not add it",
		   frame->GTimeS, ch->name);}

  sprintf(comment,"Produced by %.90s",CfgGetCmName());
  adc = FrAdcDataNewF(frame, ch->name, comment, 0, 0, ch->nBits, 0,1/ch->scale, 
                      ch->unit, ch->sampleRate, frame->dt*ch->sampleRate);
  if(adc == NULL) CfgMsgAddFatal("%d: could not create FrAdcData for %s", 
				 frame->GTimeS,ch->name);

  FrVectCopyTo(ch->filter->fOut->output, ch->scale, adc->data);

  adc->data->GTime = frame->GTimeS + 1.e-9*frame->GTimeN;

  return;
}
/*-----------------------------------------------------------------------------*/
void FdSiChannelProcess(FdSiChannel *ch,
			FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdSiChannelProcessOne(ch, frame);

  next = ch->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*-----------------------------------------------------------------------------*/
int FdSiDqChannelNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdSiDqChannel *ch;

  ch = calloc(1,sizeof(FdSiDqChannel));
  if(ch == NULL) CfgMsgAddFatal("malloc FdSiChannel failed");

  ch->action = FdActionNew((FdAction**) arg,   (void*) ch, 
			   FdSiDqChannelProcess, "SiChannel");

  FrStrCpy(&(ch->name), CfgParseGetNextString(dataType));
  ch->sampleRate    =   CfgParseGetNextReal  (dataType);
  ch->unlockedValue =   CfgParseGetNextDec   (dataType);
  ch->lockedValue   =   CfgParseGetNextDec   (dataType);
  ch->minSegLen     =   CfgParseGetNextReal  (dataType);
  ch->maxSegLen     =   CfgParseGetNextReal  (dataType);
  ch->dutyCycle     =   CfgParseGetNextReal  (dataType);
  ch->seed1         =   CfgParseGetNextDec   (dataType);
  ch->seed2         = 0;
  FrStrCpy(&(ch->hoftName), CfgParseGetNextString(dataType));

  ch->segStop  = 0;
  ch->totTime  = 0;
  ch->downTime = 0.;

  CfgMsgAddInfo("Add DQ channel %s (%gHz) unlocked=%d locked=%d "
                " min segment: %gs max=%gs duty cycle=%.3f h(t):%s",
		ch->name,  ch->sampleRate, ch->unlockedValue, ch->lockedValue, 
		ch->minSegLen, ch->maxSegLen, ch->dutyCycle, ch->hoftName);
  if(ch->dutyCycle < 0 ||  
     ch->dutyCycle > 1) 
    CfgMsgAddFatal("Duty cycle should be in the [0;1] range");
  if(ch->seed1 != 0) CfgMsgAddInfo("  Seed=%lg", ch->seed1);

  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
void  FdSiDqChannelProcessOne(FdSiDqChannel *ch, FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  FrVect *hoft;
  double gps, meanSegmentLocked, meanSegmentUnlocked, unlockRate, dt, segLen;
  double random;
  int   nData, i, ratio, j;

  nData = frame->dt*ch->sampleRate;
  adc = FrAdcDataNew(frame, ch->name, ch->sampleRate, nData, 32);
  if(adc == NULL) CfgMsgAddFatal("FdSiDqChannelProc malloc failed at %d",
				 frame->GTimeS);
  adc->data->GTime = frame->GTimeS + 1.e-9*frame->GTimeN;
  adc->data->type = FR_VECT_4U;

  /*-------------------------------------------------------- fill the vector---*/
  meanSegmentUnlocked = (ch->minSegLen+ch->maxSegLen)/2.;
  meanSegmentLocked   = meanSegmentUnlocked*ch->dutyCycle/(1.-ch->dutyCycle);
  unlockRate = 1./(ch->sampleRate*meanSegmentLocked);
  dt = 1./ch->sampleRate;

  ch->totTime += frame->dt;

  for(i=0; i<nData; i++) {
    gps = adc->data->GTime + i*dt;
    if(gps < ch->segStop) {
      adc->data->dataI[i] = ch->unlockedValue;}
    else {
      adc->data->dataI[i] = ch->lockedValue;
      if(FrUMrand2(&(ch->seed1),&(ch->seed2)) < unlockRate) {
	random = FrUMrand2(&(ch->seed1),&(ch->seed2));
	segLen = ch->minSegLen + random*(ch->maxSegLen-ch->minSegLen);
	ch->segStop = gps + segLen;
	ch->downTime += segLen;
	CfgMsgAddInfo("At %.0f start unlock segment of %11.1fs. "
		      "duty cycle=%5.3f - %5.3f", gps, segLen, 
		      1.-(ch->downTime-segLen)/ch->totTime,
		      1. -ch->downTime        /ch->totTime);}}}

  /*------------------------------------------------ reset h(t) if requested---*/
  hoft = FrameFindVect(frame, ch->hoftName);
  if(hoft != NULL) {
    ratio = hoft->nData/nData;
    for(i=0; i<nData; i++) {
      if(adc->data->dataI[i] == ch->lockedValue) continue;
      
      for(j=i*ratio; j<(i+1)*ratio; j++) {
        if     (hoft->type == FR_VECT_2S) hoft->dataS[j] = 0;
        else if(hoft->type == FR_VECT_4S) hoft->dataI[j] = 0;
        else if(hoft->type == FR_VECT_4R) hoft->dataF[j] = 0;
        else if(hoft->type == FR_VECT_8R) hoft->dataD[j] = 0;}}}
 
  return;
}
/*-----------------------------------------------------------------------------*/
void FdSiDqChannelProcess(FdSiDqChannel *ch,
			  FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdSiDqChannelProcessOne(ch, frame);

  next = ch->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*-----------------------------------------------------------------------------*/
int FdSiEventChannelNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdAction *lastAction;
  FrSimEvent *evt;
  FdSiEventCh *ch;
  FdSiEvent *si;
  int i;

  lastAction = FdActionGetLast(*((FdAction**) arg));
  if((lastAction == NULL) || (strcmp(lastAction->type,"SiEvent") != 0))
    CfgMsgAddFatal("The key FDIN_S_INJECTION_CHANNEL must follow a "
		   "FDIN_S_INJECTION_FILE key");

  ch = calloc(1,sizeof(FdSiEventCh));
  if(ch == NULL) CfgMsgAddFatal("malloc FdSiEventChannel failed");
  si = (FdSiEvent*) lastAction->data;
  ch->next = si->channels;
  si->channels = ch;

  FrStrCpy(&(ch->waveformName), CfgParseGetNextString (dataType));
  FrStrCpy(&(ch->channelName),  CfgParseGetNextString (dataType));
  FrStrCpy(&(ch->hoftSavedName),CfgParseGetNextString (dataType));
  ch->errorFlag = FR_NO;

  CfgMsgAddInfo(" Channel %s will be added to channel %s",
		ch->waveformName, ch->channelName);
  if(ch->hoftSavedName != NULL) CfgMsgAddInfo(
     "  injection channel will be recorded as %s",ch->hoftSavedName);

  if(si->eventName == NULL)
    FrStrCpy(&(ch->simEventName), CfgParseGetNextString (dataType));
  if(ch->simEventName != NULL) 
    CfgMsgAddInfo(" FrSimEvent %s structure will be copied",ch->simEventName);

   
  if(si->nEvents == 0) return(CFG_OK);
  if(si->period   < 0) return(CFG_OK); 

  /*---------scan the file to extract the event boundaries (if not yet set)---*/
  for(i=0; i<si->nEvents; i++) {
    evt = si->simEvents[i];
    if(evt->timeBefore != 0) continue;

    FdSiEventGetTBeforeAfter(si->file, evt, ch->waveformName, si->dtScan);

    /*-----------------check that injections are not overlaping; 
                             otherwise they would be injected twice or more ---*/
    if(i == 0) continue;
    double dt = evt->GTimeS - si->simEvents[i-1]->GTimeS 
        +1.e-9*(evt->GTimeN - si->simEvents[i-1]->GTimeN);
    if(evt->timeBefore < dt && si->simEvents[i-1]->timeAfter < dt) continue;
    CfgMsgAddFatal("The last two injections overlap: dt=%g s",dt);}
  
  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
int FdSiEventEvtNameNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdAction *lastAction;
  FdSiEvent *si;

  lastAction = FdActionGetLast(*((FdAction**) arg));
  if((lastAction == NULL) || (strcmp(lastAction->type,"SiEvent") != 0))
    CfgMsgAddFatal("The key FDIN_S_INJECTION_EVT_NAME must follow a "
		   "FDIN_S_INJECTION_FILE key");

  si = (FdSiEvent*) lastAction->data;

  if(si->channels != NULL) 
    CfgMsgAddFatal("The key FDIN_S_INJECTION_EVT_NAME must be before the "
                   "FDIN_S_INJECTION_CHANNEL keys");

  FrStrCpy(&si->eventName,    CfgParseGetNextString(dataType));
  FrStrCpy(&si->eventNewName, CfgParseGetNextString(dataType));
  si->dtScan = CfgParseGetNextDec(dataType);
  if(si->dtScan <= 0) si->dtScan = 40;

  si->tStartNext = 0;
  si->tEndNext   = 0; 

  if(si->eventNewName != NULL && strcmp(si->eventNewName,".") == 0) {
    free(si->eventNewName);
    si->eventNewName = NULL;}

  if(si->eventNewName == NULL)
    CfgMsgAddInfo(" Will read the \"%s\" FrSimEvents, keeping their name, dtScan=%d", 
               si->eventName, si->dtScan);
  else
    CfgMsgAddInfo(" Will read the \"%s\" FrSimEvents, renaming %s, dtScan=%d", 
               si->eventName, si->eventNewName, si->dtScan);

  FdSiEventInit(si);
 
  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
int FdSiEventFileNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdSiEvent *si;

  si= calloc(1,sizeof(FdSiEvent));
  if(si == NULL) CfgMsgAddFatal("malloc FdSiEvent failed");

  si->action = FdActionNew((FdAction**) arg,   (void*) si, 
			   FdSiEventProcess, "SiEvent");

  FrStrCpy(&si->fileName, CfgParseGetNextString(dataType));
  si->tStart   = CfgParseGetNextDec (dataType);
  si->duration = CfgParseGetNextDec (dataType);

  CfgMsgAddInfo("Software injection file: %s", si->fileName);

  si->period  = 0;
  si->nEvents = -1;
  si->iSim    = -1;
  si->tStartNext = 0;
  si->tEndNext   = 0;
  si->hoftSimMissing = FR_NO;
  si->scale = 1;

  /*---------------------------------------------------------------------------*/
  
  si->file = FrFileINew(si->fileName);
  if(si->file == NULL) CfgMsgAddFatal("Cannot open software injection file");

  /*--------------------------- if no sim event given, get the file limits-----*/
  si->tStartNext = FrFileITStart(si->file);
  si->tEndNext   = FrFileITEnd  (si->file);
  si->nEvents = 0;

  if(si->tStart   <= 0)  si->tStart = si->tStartNext;
  if(si->duration <= 0)  si->duration = si->tEndNext - si->tStart;
  if(si->tStart < si->tStartNext)  {
     si->duration -= si->tStartNext - si->tStart;
     si->tStart    = si->tStartNext;}
  if(si->duration + si->tStart > si->tEndNext)
    si->duration =  si->tEndNext - si->tStart;

  CfgMsgAddInfo(" File start:%.1f end:%.1f; use timeStart:%d duration:%d",
		si->tStartNext, si->tEndNext, si->tStart, si->duration);

  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
void FdSiEventGetTBeforeAfter(FrFile* iFile, FrSimEvent *evt, char* waveformName, double dt)
/*-----------------------------------------------------------------------------*/
{
  double tBefore, tAfter, tStart, tEvent, max, threshold, value;
  FrVect *hoftSim;
  int i, iMax, nData;

  /*---------------------------search for the maximum h(t) within +- 100 ms---*/
  tEvent  = evt->GTimeS+1.e-9*evt->GTimeN; 
  hoftSim = FrFileIGetVect(iFile, waveformName, tEvent-0.1, 0.2);
  if(hoftSim == NULL) return;

  max = -1;
  iMax = 0;
  for(i=0; i<hoftSim->nData; i++) {
    if(hoftSim->type == FR_VECT_4R) value = fabsf(hoftSim->dataF[i]);
    else                            value = fabs(hoftSim->dataD[i]);
    if(value < max) continue;
    iMax = i;
    max  = value;}

  tEvent = hoftSim->GTime + iMax*hoftSim->dx[0];
  threshold = max/10000;

  /*----get the injection start by asking for two consecutive small samples---*/
  tBefore = 0;
  tStart = tEvent - dt;

  while(1) {
    hoftSim = FrFileIGetVect(iFile, waveformName, tStart, dt);
    if(hoftSim == NULL) break;

    if(hoftSim->type == FR_VECT_4R) {
      for(i = hoftSim->nData - 1; i >= 1; i--) {
        if(fabsf(hoftSim->dataF[i])   > threshold) continue;
        if(fabsf(hoftSim->dataF[i-1]) < threshold) break;}}
    else {
      for(i = hoftSim->nData - 1; i >= 1; i--) {
        if(fabs(hoftSim->dataD[i])   > threshold) continue;
        if(fabs(hoftSim->dataD[i-1]) < threshold) break;}}

   tBefore = tEvent - (hoftSim->GTime + i * hoftSim->dx[0]);
   FrVectFree(hoftSim); 

   if(i > 0) break;

   tStart -= dt;}

  /*----get the injection tAfter by asking for two consecutive small samples---*/
  tAfter = 0;
  tStart = tEvent;

  while(1) {
    hoftSim = FrFileIGetVect(iFile, waveformName, tStart, dt);
    if(hoftSim == NULL) break;

    nData = hoftSim->nData;
    if(hoftSim->type == FR_VECT_4R) {
      for(i = 0; i<nData - 1; i++) {
        if(fabsf(hoftSim->dataF[i])   > threshold) continue;
        if(fabsf(hoftSim->dataF[i+1]) < threshold) break;}}
    else {
      for(i = 0; i<nData - 1; i++) {
        if(fabs(hoftSim->dataD[i])   > threshold) continue;
        if(fabs(hoftSim->dataD[i+1]) < threshold) break;}}

    tAfter = hoftSim->GTime + i * hoftSim->dx[0] - tEvent;
    FrVectFree(hoftSim);

    if(i < nData-1) break;

    tStart += dt;}

  /*-----------------------------------------------------------------------*/
  if(tBefore > evt->timeBefore) evt->timeBefore = tBefore;
  if(tAfter  > evt->timeAfter)  evt->timeAfter  = tAfter;

  CfgMsgAddInfo("  FrSimEvent %s at %.3f max=%.3e tBefore=%8.3f tAfter=%8.3f",
                evt->name, evt->GTimeS+1.e-9*evt->GTimeN, max, 
		evt->timeBefore, evt->timeAfter);
  CfgMsgSendWithTimeout(0.);

  return;
}
/*-----------------------------------------------------------------------------*/
void FdSiEventInit(FdSiEvent *si)
/*-----------------------------------------------------------------------------*/
{
  int i;
  FrSimEvent *simEvents, *evt;
  double tEvent;

  /*---------------------------- first extract the FrSimEvent from the file---*/ 
  simEvents = FrSimEventReadT(si->file, si->eventName, 
			      si->tStart, si->duration, 0., 1.e10);

  si->nEvents = 0;
  for(evt = simEvents; evt != NULL; evt = evt->next) {si->nEvents++;}

  CfgMsgAddInfo(" %d FrSimEvent(s) found:", si->nEvents);
  if(si->nEvents == 0) return;

  si->simEvents = (FrSimEvent**) calloc(si->nEvents, sizeof(FrSimEvent*));
  i = 0;
  for(evt = simEvents; evt != NULL; evt = evt->next) {
    si->simEvents[i] = evt;
    i++;}

  /*-------------------------- scan the file to extract the event boundaries---*/
  for(i=0; i<si->nEvents; i++) {
    evt = si->simEvents[i];
    tEvent = evt->GTimeS+1.e-9*evt->GTimeN;

    evt->timeBefore = 0;
    evt->timeAfter = 0;

    if(si->eventNewName == NULL) 
      CfgMsgAddInfo("  FrSimEvent %s at %.3f", evt->name, tEvent);
    else {
      CfgMsgAddInfo("  FrSimEvent %s renamed %s at %.3f", 
	evt->name, si->eventNewName, tEvent);
      free(evt->name);
      FrStrCpy(&evt->name, si->eventNewName);}}

  return;
}
/*-----------------------------------------------------------------------------*/
void FdSiEventProcessOne(FdSiEvent *si, FrameH *frame)
/*-----------------------------------------------------------------------------*/
{
  int i, decimate;
  FrVect *hoft, *hoftSim, *hoftSaved;
  FdSiEventCh *ch;
  double frameStart, frameEnd, w;
  FrSimEvent *simEvent;

  frameStart = frame->GTimeS + 1.e-9*frame->GTimeN;
  frameEnd   = frameStart + frame->dt;

  /*--------------------------------------------create the no injection flag---*/
  if(si->injFlagName != NULL) FdSiEventInjFlagProcess(si, frame);

  /*-------------------- do nothing if not yet at the time of the next event---*/
  if(si->tStartNext > frameEnd) return;

  /*------------------- if the injection is completed; schedule the next one---*/
  if(si->tEndNext < frameStart && si->nEvents > 0) {
    if(FdSiEventScheduleNext(si, frame) != 0) return;
    if(si->tStartNext > frameEnd) return;}

  /*-------------------------------------- save event structure if requested---*/
  if(si->tNext < frameEnd && si->newSimEvent != NULL) {
    si->newSimEvent->next = frame->simEvent;
    frame->simEvent = si->newSimEvent;
    si->newSimEvent = NULL;}

  /*---------------------- get waveform(s) and add it to the h(t) channel(s)---*/
  for(ch = si->channels; ch != NULL; ch = ch->next) {
    hoftSim = FrFileIGetVect(si->file, ch->waveformName, 
			     frame->GTimeS-si->dt, frame->dt);
    if(hoftSim == NULL) {
      if(si->hoftSimMissing == FR_NO) {
        CfgMsgAddError("At %d injection waveform %s start to be missing (%.2f)",
                       frame->GTimeS, ch->waveformName, frame->GTimeS-si->dt);
        si->hoftSimMissing = FR_YES;}
      continue;}
    else if(si->hoftSimMissing == FR_YES) {
      CfgMsgAddInfo("At %d injection waveform %s is back (%.2f)",
                     frame->GTimeS, ch->waveformName, frame->GTimeS-si->dt);
      si->hoftSimMissing = FR_NO;}

    if((hoftSim->type != FR_VECT_4R) &&
       (hoftSim->type != FR_VECT_8R))
      CfgMsgAddFatal("FdSiEventAdd: %d wrong type %d for %s",
		       frame->GTimeS, hoftSim->type, hoftSim->name);
    FrVectFixNAN(hoftSim);
    FrVectFixName(hoftSim);
    FrvScale(si->scale, hoftSim, hoftSim, NULL);
    FrProcDataNewV(frame, hoftSim);

    hoft = FrameFindVect(frame, ch->channelName);
    if(hoft == NULL) {
      if(ch->errorFlag == FR_NO) {
        CfgMsgAddError("At %d; FdSiEventAdd: could not find %s",
		     frame->GTimeS, ch->channelName);
        ch->errorFlag = FR_YES;}
      continue;}
    else if(ch->errorFlag == FR_YES) {
      CfgMsgAddInfo("At %d; FdSiEventAdd: channel %s is back",
                      frame->GTimeS, ch->channelName);
      ch->errorFlag = FR_NO;}

    if((hoft->type != FR_VECT_4R) &&
       (hoft->type != FR_VECT_8R)) {
      CfgMsgAddFatal("FdSiEventAdd: %d wrong hoft type %d for %s",
		     frame->GTimeS, hoft->type, hoft->name);
      continue;}

    decimate = hoftSim->nData/hoft->nData;
    if(decimate == 0) 
      CfgMsgAddFatal("FdSiEventAdd: decimate=0 (%"FRLLD"/%"FRLLD") at %d",
		     hoftSim->nData, hoft->nData, frame->GTimeS);
    if(decimate*hoft->nData != hoftSim->nData) 
      CfgMsgAddFatal("FdSiEventAdd: bad nData values (%"FRLLD"/%"FRLLD") at %d",
		     hoftSim->nData, hoft->nData, frame->GTimeS);

    if(ch->hoftSavedName != NULL) {
      hoftSaved = FrVectCopyToF(hoftSim, 1., ch->hoftSavedName);
      if(decimate != 1) FrVectDecimate(hoftSaved, -decimate, NULL);
      FrProcDataNewVT(frame, hoftSaved, 1);}

    for(i=0; i<hoft->nData; i++) {
      if(hoftSim->type == FR_VECT_4R) w = hoftSim->dataF[i*decimate];
      else                            w = hoftSim->dataD[i*decimate];
      if(hoft->type == FR_VECT_4R) hoft->dataF[i] += w;
      else                         hoft->dataD[i] += w;}

     if(ch->simEventName != NULL) {
       simEvent = FrSimEventReadT(si->file, ch->simEventName,
                                hoft->GTime, frame->dt, 0., 1.e10);
       if(simEvent != NULL) {
         CfgMsgAddInfo("FrSimEvent structure %s for %.4f copied",
                       simEvent->name, simEvent->GTimeS + 1.e-9*simEvent->GTimeN);
         simEvent->next = frame->simEvent;
         frame->simEvent = simEvent;}}
     }

  return;}

/*-----------------------------------------------------------------------------*/
void FdSiEventProcess(FdSiEvent *si,
		      FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdSiEventProcessOne(si, frame);

  next = si->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*-----------------------------------------------------------------------------*/
int FdSiEventRescaleNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdAction *lastAction;
  FdSiEvent *si;

  lastAction = FdActionGetLast(*((FdAction**) arg));
  if((lastAction == NULL) || (strcmp(lastAction->type,"SiEvent") != 0))
    CfgMsgAddFatal("The key FDIN_S_INJECTION_RESCALE must follow a "
		   "FDIN_S_INJECTION_FILE key");

  si = (FdSiEvent*) lastAction->data;

  si->scaleMin = CfgParseGetNextReal(dataType);
  si->scaleMax = CfgParseGetNextReal(dataType);
  si->scale = si->scaleMin;

  if(si->scaleMax < si->scaleMin) si->scaleMax = si->scaleMin;

  CfgMsgAddInfo(" Injection(s) will be rescaled between %g and %g; default:%g",
		si->scaleMin, si->scaleMax, si->scale);
 
  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
int FdSiEventRescheduleNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdAction *lastAction;
  FdSiEvent *si;

  lastAction = FdActionGetLast(*((FdAction**) arg));
  if((lastAction == NULL) || (strcmp(lastAction->type,"SiEvent") != 0))
    CfgMsgAddFatal("The key FDIN_S_INJECTION_RESCHEDULE must follow a "
		   "FDIN_S_INJECTION_FILE key");

  si = (FdSiEvent*) lastAction->data;

  if(si->eventName == NULL) 
    CfgMsgAddFatal("Could not reschedule injections if the key"
                   " FDIN_S_INJECTION_EVT_NAME is missing");

  si->period   = CfgParseGetNextReal(dataType);
  si->jitter   = CfgParseGetNextReal(dataType);
  si->offset   = CfgParseGetNextReal(dataType);

  CfgMsgAddInfo(" Injection(s) will be rescheduled: period=%g jitter=%g "
                "offset=%g", si->period, si->jitter, si->offset);
 
  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
int FdSiEventInjFlagNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdAction *lastAction;
  FdSiEvent *si;

  lastAction = FdActionGetLast(*((FdAction**) arg));
  if((lastAction == NULL) || (strcmp(lastAction->type,"SiEvent") != 0))
    CfgMsgAddFatal("The key FDIN_S_INJECTION_FLAG must follow a "
		   "FDIN_S_INJECTION_FILE key");

  si = (FdSiEvent*) lastAction->data;

  FrStrCpy(&si->injFlagName, CfgParseGetNextString(dataType));
  si->injFlagRate    = CfgParseGetNextDec(dataType);
  si->injFlagTBefore = CfgParseGetNextReal(dataType);
  si->injFlagTAfter  = CfgParseGetNextReal(dataType);

  CfgMsgAddInfo(" Injection flag %s at %d Hz will be created with tBefore=%g "
	"tAfter=%g ", si->injFlagName, si->injFlagRate, 
	si->injFlagTBefore, si->injFlagTAfter);
 
  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdSiEventInjFlagProcess(FdSiEvent *si,
                         FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *flag;
  FrVect *vect;
  long nData, i;
  double t;
 
  vect = FrameFindVect(frame, si->injFlagName);

  nData = frame->dt*si->injFlagRate;

  if(vect != NULL) {
    if(vect->nData != nData)
      CfgMsgAddFatal("FdSiEventInjectionFlag: nData missmatch %ld %ld",
	vect->nData, nData); 
    if(vect->type != FR_VECT_4S) 
      CfgMsgAddFatal("FdSiEventInjectionFlag: wrong vector type");}

  else {
    flag = FrAdcDataNew(frame, si->injFlagName, si->injFlagRate, nData, 32);
    if(flag == NULL) CfgMsgAddFatal("FdSiEventInjectionFlag: malloc failed");
    vect = flag->data;
    for(i=0; i<nData; i++) {vect->dataI[i] = 1;}}


  /*---------------------update tStart/tEnd once the injection is fully over---*/
  if(si->injFlagTEnd < frame->GTimeS) {
     si->injFlagTEnd   = si->tNext + si->injFlagTAfter;
     si->injFlagTStart = si->tNext - si->injFlagTBefore;}

  for(i=0; i<nData; i++) {
    t = frame->GTimeS +i*vect->dx[0];
    if(t < si->injFlagTStart || t > si->injFlagTEnd) continue;
    vect->dataI[i] = 0;}

  return;
}
/*-----------------------------------------------------------------------------*/
int FdSiEventScheduleNext(FdSiEvent *si, FrameH *frame)
/*-----------------------------------------------------------------------------*/
/* This function get the next simulated event to be injected                   */
/*-----------------------------------------------------------------------------*/
{
  int i;
  FrSimEvent *evt;
  double frameStart, random, tMod, tModMin, evtTime;

  frameStart = frame->GTimeS + 1.e-9*frame->GTimeN;

  /*------------------------------ find the next event if we reschedule them---*/
  if(si->period > 0) {
    si->iSim = 0;
    tModMin = si->period;
    for(i = 0; i < si->nEvents; i++) {
      evtTime = si->simEvents[i]->GTimeS + 1.e-9*si->simEvents[i]->GTimeN;
      tMod = fmod(evtTime - frameStart - si->offset, si->period);
      if(tMod < 0) tMod += si->period;
      if(tMod > tModMin) continue;
      tModMin = tMod;
      si->iSim = i;}

    /*----------------------compute new time: add time jitter if requested ---*/
    si->tNext = frameStart + tModMin;
    si->tNext += si->jitter * FrRand1(&si->seed);
    evt = si->simEvents[si->iSim];
    si->dt = si->tNext - (evt->GTimeS+1.e-9*evt->GTimeN);
    si->tStartNext = si->tNext - evt->timeBefore;
    si->tEndNext   = si->tNext + evt->timeAfter;}

  /*-----------no rescheduling: get next event which could still be injected---*/
  else {
    while(si->tEndNext < frameStart) {
      si->iSim++;
      if(si->iSim >= si->nEvents) return(1);  /*---- no more event to inject---*/
      si->dt = si->offset;
      evt = si->simEvents[si->iSim];
      si->tNext = evt->GTimeS+1.e-9*evt->GTimeN + si->dt;
      si->tStartNext = si->tNext - evt->timeBefore;
      si->tEndNext   = si->tNext + evt->timeAfter;}}

  /*------------------------------------------------create a new FrSimEvent----*/
  evt = FrSimEventCopy(si->simEvents[si->iSim]);
  if(evt == NULL) CfgMsgAddFatal("%d could not copy event %d",
				 frame->GTimeS, si->iSim);
  si->newSimEvent = evt;

  /*------------------------------------------- update FrSimEvent parameters---*/
  FrSimEventSetParam(evt, "d_time", si->dt);

  FrSimEventUpdateTime(evt, si->dt);

  /*-------------------------------------------- rescale the event amplitude---*/
  if(si->scaleMax > 0) {
    random = FrRand1(&si->seed);
    si->scale = random * (si->scaleMax - si->scaleMin) + si->scaleMin;}
  else {
    si->scale = 1.;}
  evt->amplitude *= si->scale;
  FrSimEventSetParam(evt, "rescale", si->scale);  
  FrSimEventSetParam(evt, "eff_dist_h", 
		     FrSimEventGetParam(evt,"eff_dist_h")/si->scale);
  FrSimEventSetParam(evt, "eff_dist_l", 
		     FrSimEventGetParam(evt,"eff_dist_l")/si->scale);
  FrSimEventSetParam(evt, "eff_dist_v", 
		     FrSimEventGetParam(evt,"eff_dist_v")/si->scale);
  FrSimEventSetParam(evt, "eff_dist_t", 
		     FrSimEventGetParam(evt,"eff_dist_t")/si->scale);

  CfgMsgAddInfo("At %d schedule next injection; tStart=%.3f max at %.3f "
                "scale=%g index=%d",  
                frame->GTimeS, si->tStartNext, si->tNext, si->scale, si->iSim);

  return(0);
}
/*-----------------------------------------------------------------------------*/
void FdSiParserAdd(FdIO *fdIO)
/*-----------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_ADD_SURROGATE_CHANNEL", 
			 FdSiChannelNew,  (void *) &(fdIO->actionsIn), 7,
			 CfgString, CfgString, CfgReal, CfgDec,
			 CfgString, CfgReal,   CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_ADD_SURROGATE_DQ_CHANNEL", 
			 FdSiDqChannelNew, (void *) &(fdIO->actionsIn), 9,
			 CfgString, CfgReal, CfgDec,  CfgDec,
			 CfgReal,   CfgReal, CfgReal, CfgDec, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_ADD_TEST_CHANNEL", 
			 FdSiTChannelNew,  (void *) &(fdIO->actionsIn), 9,
			 CfgString, CfgString, CfgReal, CfgDec, CfgReal,
			 CfgReal,   CfgReal,   CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_S_INJECTION_FILE", 
			 FdSiEventFileNew,  (void *) &(fdIO->actionsIn), 3,
			 CfgString, CfgDec, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_S_INJECTION_FLAG", 
			 FdSiEventInjFlagNew,  (void *) &(fdIO->actionsIn), 4,
			 CfgString, CfgDec, CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_S_INJECTION_CHANNEL",
			 FdSiEventChannelNew, (void *) &(fdIO->actionsIn), 4,
			 CfgString, CfgString, CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_S_INJECTION_EVT_NAME",
			 FdSiEventEvtNameNew, (void *) &(fdIO->actionsIn), 3,
			 CfgString, CfgString, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_S_INJECTION_RESCALE",
			 FdSiEventRescaleNew, (void *) &(fdIO->actionsIn), 2,
			 CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_S_INJECTION_RESCHEDULE",
			 FdSiEventRescheduleNew, (void *) &(fdIO->actionsIn), 3,
			 CfgReal, CfgReal, CfgReal);

  return;
}
/*-----------------------------------------------------------------------------*/
int FdSiTChannelNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdSiTChannel *ch;

  ch = malloc(sizeof(FdSiTChannel));
  if(ch == NULL) CfgMsgAddFatal("malloc FdSiTChannel failed");

  ch->action = FdActionNew((FdAction**) arg,   (void*) ch, 
			   FdSiTChannelProcess, "SiTChannel");

  FrStrCpy(&(ch->name),     CfgParseGetNextString(dataType));
  FrStrCpy(&(ch->unit),     CfgParseGetNextString(dataType));
  ch->sampleRate =          CfgParseGetNextReal  (dataType);
  ch->nBits      =          CfgParseGetNextDec   (dataType);
  ch->mean       =          CfgParseGetNextReal  (dataType);
  ch->rms        =          CfgParseGetNextReal  (dataType);
  ch->amplitude  =          CfgParseGetNextReal  (dataType);
  ch->frequency  =          CfgParseGetNextReal  (dataType);
  ch->scale      =          CfgParseGetNextReal  (dataType);
  ch->phase      = 0.;

  if(ch->scale == 0) ch->scale = 1.;
  CfgMsgAddInfo("Add channel %s unit:%s (%gHz) nBits:%3ld "
                "mean:%g rms:%g amplitude:%g freq:%g scale:%g",
		ch->name,  ch->unit,  ch->sampleRate, ch->nBits, 
                ch->mean,  ch->rms, ch->amplitude, ch->frequency, ch->scale);

  if(ch->nBits != -64 && 
     ch->nBits != -32 &&
     ch->nBits !=  16 &&
     ch->nBits !=  32) CfgMsgAddFatal("Invalid nBits: %ld",ch->nBits);

  if(ch->nBits > 0) CfgMsgAddInfo("  Numerical noise introduced by integer"
				  " storage is= %g%s/sqrt(Hz)",1./(ch->scale*sqrt(6*ch->sampleRate)),
				  ch->unit);

  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
void  FdSiTChannelProcessOne(FdSiTChannel *ch, FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  FrVect *vect;
  int   i;
  double phase, dPhi, result, amplitude, rms, mean;
  char comment[128];

  /*----------------------create the FrAdcData to store the filtered channel---*/
  if(FrameFindVect(frame, ch->name) != NULL) {
    CfgMsgAddFatal("%d: The channel %s already exist: we could not add it",
		   frame->GTimeS, ch->name);}
 
  sprintf(comment,"Produced by %.90s",CfgGetCmName());
  adc = FrAdcDataNewF(frame, ch->name, comment, 0, 0, ch->nBits, 0, ch->scale, 
                      ch->unit, ch->sampleRate, frame->dt*ch->sampleRate);
  if(adc == NULL) CfgMsgAddFatal("%d: could not create FrAdcData for %s", 
				 frame->GTimeS,ch->name);
  adc->data->GTime = frame->GTimeS + 1.e-9*frame->GTimeN;
  vect = adc->data;

  /*-----------------------------------------------------------fill sine wave---*/
  dPhi = FRTWOPI * ch->frequency * vect->dx[0];
  phase = ch->phase;
  amplitude = ch->amplitude/ch->scale;
  mean      = ch->mean     /ch->scale;
  rms       = ch->rms      /ch->scale;

  for(i=0; i<vect->nData; i++)  {
    phase += dPhi;
    result = amplitude * sin(phase) + mean + rms * FrvGaus(0);
    if     (vect->type == FR_VECT_2S) vect->dataS[i] = result;
    else if(vect->type == FR_VECT_8R) vect->dataD[i] = result;
    else if(vect->type == FR_VECT_4R) vect->dataF[i] = result;
    else if(vect->type == FR_VECT_4S) vect->dataI[i] = result;
    else if(vect->type == FR_VECT_8S) vect->dataL[i] = result;
    else if(vect->type == FR_VECT_C8) vect->dataF[i] = result;}

  ch->phase = fmod(phase,FRTWOPI);

  return;
}
/*-----------------------------------------------------------------------------*/
void FdSiTChannelProcess(FdSiTChannel *ch, FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdSiTChannelProcessOne(ch, frame);

  next = ch->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
