/*-----------------------------------------------------------------------------*/
/* MbtaThread.c */
/*-----------------------------------------------------------------------------*/

#define _GNU_SOURCE
#include <pthread.h>
#include "FdIO.h"
#include <sys/types.h>
#include <sys/syscall.h> 

typedef struct FdThread  FdThread;

struct FdThread {       
  int          nFrames;     /* number of frames in the output buffer   */
  char        *name;        /* name of the thread                      */
  /*-------------------------- end of input parameters ----------------*/
  FrameH     **frames;      /* arrays of frames to be processed        */
  int          indexIn;     /* index of the next frame to write        */
  int          indexOut;    /* index for the next frame to read        */
  double       maxWaitTime; /* maximum waiting time to feed the thread */
  pthread_t    thrdId;      /* thread info                             */
  sem_t        semNewData;  /* tell is new data should be processed    */
  pthread_mutex_t lock;     /* mutex to control the frame passing      */
  FdAction    *action;      /* FdAction object                         */
};

void FdThreadProcess(FdThread *thrd, FrameH* frame);
void FdThreadMain(FdThread *thrd);

/*-----------------------------------------------------------------------------*/
int FdThreadNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdThread *thrd;
  pthread_t myId;
  cpu_set_t cpuset;
  int irc;

  /*------------------------------------------------create thread structure---*/
  thrd = calloc(1, sizeof(FdThread));
  if(thrd == NULL) CfgMsgAddFatal("malloc FdThread failed");

  thrd->action = FdActionNew((FdAction**) arg,   (void*) thrd,
                           FdThreadProcess, "Thread");

  thrd->nFrames =         CfgParseGetNextDec   (dataType);
  FrStrCpy(&(thrd->name), CfgParseGetNextString(dataType));
  if(thrd->name == NULL) FrStrCpy(&(thrd->name), "no_name");

  CfgMsgAddInfo("Add a thread %s with up to %d frames of buffering",
                thrd->name, thrd->nFrames);
  if(thrd->nFrames <= 0) 
    CfgMsgAddFatal("The buffer must be at least one frame long");

  thrd->frames   = calloc(thrd->nFrames, sizeof(FrameH*));
  thrd->indexIn  = 0;
  thrd->indexOut = 0;

  /*---------------------------------------------------initialize semaphore---*/
  sem_init(&thrd->semNewData,  0, 0);

 /*-----------------------------------------------------------create thread---*/
  irc = pthread_create(&(thrd->thrdId), NULL, (void *(*)(void *))FdThreadMain, 
                       (void *)thrd);
  if(irc == -1) CfgMsgAddFatal("FdThreadMain creation failed");

  /*------------put the thread on the same CPU to avoid cache inefficiency---*/
  myId = pthread_self();   /*----------get thread id for the main---*/
  pthread_getaffinity_np(myId,         sizeof(cpu_set_t), &cpuset);
  pthread_setaffinity_np(thrd->thrdId, sizeof(cpu_set_t), &cpuset);

  return(CFG_OK);
}

/*-----------------------------------------------------------------------------*/
void FdThreadProcess(FdThread *thrd,
                      FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  struct timeval tS, tE;
  double waitTime;
  int gps;

  /*---------------------------------------if needed, wait for an empty slot---*/
  if(thrd->frames[thrd->indexIn] != NULL) {
    gettimeofday(&tS, NULL);
    while(thrd->frames[thrd->indexIn] != NULL) {usleep(1000);}
    gettimeofday(&tE, NULL);
    waitTime = tE.tv_sec-tS.tv_sec + 1.e-6 * (tE.tv_usec-tS.tv_usec);
    if(waitTime > thrd->maxWaitTime) {
      thrd->maxWaitTime = waitTime;
      gps = 0;
      if(frame != NULL) gps = frame->GTimeS;
      CfgMsgAddInfo("FdThread %s: at gps=%d new max waiting time:%g s",
	thrd->name, gps, thrd->maxWaitTime);}}

  /*-------------------------------------------put frame in the thread list---*/
  thrd->frames[thrd->indexIn] = frame;
  thrd->indexIn = (thrd->indexIn+1)%thrd->nFrames;

  sem_post(&thrd->semNewData);  /*------------------wake up the next thread----*/

  /*------------------------------------if frame=NULL, this is the last call---*/
  if(frame == NULL) {
    CfgMsgAddInfo("Wait for the thread %s to finish", thrd->name);
    pthread_join(thrd->thrdId, NULL);}
  return;
}
/*-----------------------------------------------------------------------------*/
void FdThreadMain(FdThread *thrd)
/*-----------------------------------------------------------------------------*/
{
  FdAction *next;
  int tmo;
  FrameH *frame;
  struct timespec currTime;
  clockid_t threadClockId;

  FdIO *fdIO = FdIOGetRoot(); 
  int tid = syscall(__NR_gettid);
                                   
  CfgMsgAddInfo("Start Fd thread %s; tid=%d pid=%d", thrd->name, tid, getpid());

  /*---------------------------------- Set the thread as Asynchronous thread---*/
  pthread_setcancelstate( PTHREAD_CANCEL_ENABLE, &tmo );
  pthread_setcanceltype ( PTHREAD_CANCEL_ASYNCHRONOUS, &tmo );

  /*----------------------- process all frame that are passed to this thread---*/
  while (1) {

    /*----------------------------------------------wait for the next frame---*/
    sem_wait(&thrd->semNewData);

    /*-----------------------extract the frame from the list and process it---*/
    pthread_mutex_lock (&thrd->lock);
    frame = thrd->frames[thrd->indexOut];
    thrd->frames[thrd->indexOut] = NULL;
    thrd->indexOut = (thrd->indexOut+1)%thrd->nFrames;
    pthread_mutex_unlock (&thrd->lock);
    next = thrd->action->next;
    if(next == NULL) CfgMsgAddFatal("FdThreadMain: no next action"); 
    next->action(next->data, frame);

    if(frame == NULL) {   /*----flush the frame output, close files if neded---*/
      while(fdIO->lastOut == FR_NO) {
        next->action(next->data, NULL);}
      break;}
  }

  /*-------------------------------- Get thread clock Id and then clock time---*/
  pthread_getcpuclockid(thrd->thrdId, &threadClockId);
  clock_gettime(threadClockId, &currTime);

  CfgMsgAddInfo("Fd thread %s ended; CPU: %g s",thrd->name,
        currTime.tv_sec + 1.e-9*currTime.tv_nsec);

  return;
}
/*-----------------------------------------------------------------------------*/
void FdThreadParserAdd(FdIO *fdIO)
/*-----------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_THREAD",
                         FdThreadNew,  (void *) &(fdIO->actionsOut), 2,
                         CfgDec, CfgString);
  return;
}
